/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2008 Adobe Systems Incorporated
 * All Rights Reserved
 *
 * Notice: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */

package com.adobe.xfa.wsdl;

import java.util.ArrayList;
import java.util.List;

import com.adobe.xfa.Element;
import com.adobe.xfa.ut.StringUtils;

/**
 * The <code>WSDLOperation</code> interface is the primary datatype for the entire
 * Web Services Description Language Model.
 * @exclude from published api.
 */
public class WSDLOperation extends WSDLNode {
	public static final int OP_TYPE_UNKNOWN = 1;
	public static final int OP_TYPE_ONE_WAY = 2;
	public static final int OP_TYPE_REQUEST_RESPONSE = 3;
	public static final int OP_TYPE_SOLICIT_RESPONSE = 4;
	public static final int OP_TYPE_NOTIFICATION = 5;

	private int meOperationType = OP_TYPE_UNKNOWN;
	private List<WSDLNode> mBindingOperationNodes;

	public WSDLOperation (WSDLDocument poDocument, Element oSrc) {
		super (poDocument, oSrc, WSDLNode.WSDL_OPERATION);
	}

/*
 * Clone a WSDLOperation.
 * @param bDeep - if true then clone recursively.
 */
	// Note: cloneNode() commented out.  In C++, it is not used by XFA,
	// Designer or Acrobat.  There is currently no requirement for it in
	// Java.  The ported code below is untested.  This code could be restored
	// if a requirement arose.
//	public WSDLOperation cloneNode (boolean bDeep) {
//		WSDLOperation poNewNode = new WSDLOperation (getWSDLOwnerDocument(), getDomNode());
//		cloneNodeHelper (poNewNode, bDeep);
//		poNewNode.meOperationType = meOperationType;
//// don't do anything fancy, just get a copy of these nodes.
//		if (poNewNode.mBindingOperationNodes == null) {
//			mBindingOperationNodes = null;
//		} else {
//			int count = poNewNode.mBindingOperationNodes.size();
//			for (int i = 0; i < count; i++) {
//				addBindingOperation (poNewNode.mBindingOperationNodes.get(i));
//			}
//		}
//		poNewNode.mBindingOperationNodes = mBindingOperationNodes;
//		return poNewNode;
//	}

/**
 * Get the message type for the message
 * @return the message type
 */
	public int getOperationType () {
		return meOperationType;
	}

	void setOperationType (int operationType) {
		meOperationType = operationType;
	}
/**
 * The name of Input (if any) for this operation, defaulting the
 * name according to the WSDL spec if necessary
 * @return the input name
 */
	public String getInputName () {
// default the name according the WSDL spec rules
// one-way op: inputName = opName
// req-resp op: inputName = opNameRequest, outputName = opNameResponse
// sol-resp op: outputName = opNameSolicit, inputName = opNameResponse
// notification op: outputName = opName
		return getInOutName (WSDL_INPUT, "Request", "Response");
	}

/**
 * The input message (if any) for this operation
 * @return the input message (may be null)
 */
	public WSDLMessage getInputMessage () {
		WSDLNode oNode = getWSDLChildNode (WSDL_INPUT, "");
		if (oNode == null) {
			return null;
		}

		String sMessage = oNode.getWSDLAttribute (WSDLA_MESSAGE);
		WSDLNode oDefinitions = getWSDLOwnerDocument().getDefinitionsNode();
		WSDLNode oMessage = oDefinitions.getWSDLChildNode (WSDL_MESSAGE, sMessage);

		return (oMessage instanceof WSDLMessage) ? (WSDLMessage) oMessage : null;
	}

/**
 * The name of Output (if any) for this operation, defaulting the
 * name according to the WSDL spec if necessary
 * @return the output name
 */
	public String getOutputName () {
// default the name according the WSDL spec rules
// one-way op: inputName = opName
// req-resp op: inputName = opNameRequest, outputName = opNameResponse
// sol-resp op: outputName = opNameSolicit, inputName = opNameResponse
// notification op: outputName = opName
		return getInOutName (WSDL_OUTPUT, "Response", "Solicit");
	}

/**
 * The output message (if any) for this operation
 * @return the output message (may be null)
 */
	public WSDLMessage getOutputMessage () {
		WSDLNode oNode = getWSDLChildNode (WSDLNode.WSDL_OUTPUT, "");
		if (oNode == null) {
			return null;
		}

		String sMessage = oNode.getWSDLAttribute (WSDLNode.WSDLA_MESSAGE);

		WSDLNode oDefinitions = getWSDLOwnerDocument().getDefinitionsNode();
		WSDLNode oMessage = oDefinitions.getWSDLChildNode (WSDLNode.WSDL_MESSAGE, sMessage);

		return (oMessage instanceof WSDLMessage) ? (WSDLMessage) oMessage : null;
	}

/**
 * The corresponding binding operation nodes for this operation.
 * all nodes returned will be of type <code>WSDLNode::WSDL_BINDING_OPERATION
 * @return a <code>WSDLNodeArray</code> of binding operation nodes
 */
	public List<WSDLNode> getBindingOperationNodes () {
		return mBindingOperationNodes;
	}

	void addBindingOperation (WSDLNode operation) {
		if (mBindingOperationNodes == null) {
			mBindingOperationNodes = new ArrayList<WSDLNode>();
		}
		mBindingOperationNodes.add (operation);
	}

	private String getInOutName (int childType, String requestAppend, String SolicitAppend) {
		WSDLNode child = getWSDLChildNode (childType, null);
		if (child == null) {
			return "";
		}
		String sName = child.getWSDLName(); // operation name
		if (StringUtils.isEmpty (sName)) {
			sName = getWSDLName();
			String appendText = null;
			switch (meOperationType) {
				case WSDLOperation.OP_TYPE_REQUEST_RESPONSE:
					appendText = requestAppend;
					break;
				case WSDLOperation.OP_TYPE_SOLICIT_RESPONSE:
					appendText = SolicitAppend;
					break;
			}
			if (appendText != null) {
				StringBuilder builder = new StringBuilder (sName);
				builder.append (appendText);
				sName = builder.toString();
			}
		}

		return sName;
	}
}
