/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.dam.scene7.api;

import java.util.List;

import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;

import org.osgi.annotation.versioning.ProviderType;

import com.day.cq.dam.api.Asset;
import com.day.cq.dam.scene7.api.constants.Scene7AssetType;

/**
 * The {@code Scene7AssetMimeTypeService} is responsible for identifying the Scene7 "fake" mime type for the various S7 imported assets.
 */
@ProviderType
public interface Scene7AssetMimetypeService {

    static final String SCR_PROP_NAME_ASSET_MIMETYPES = "cq.dam.scene7.assetmimetypeservice.mapping";
    static final String[] SCR_PROP_DEFAULT_ASSET_MIMETYPES = new String[] {
            "Generic=image/s7asset",
            "Image=image/*",
            "Image=image/jpeg",
            "Video=video/*",
            "Video.mp4=video/mp4",
            "Video.f4v=video/mp4",
            "Video.flv=video/x-flv",
            "PDF=application/pdf",
            "VTT=text/vtt",
            "CSS=text/css",
            "DOC=application/msword",
            "AI=application/postscript",
            "PS=application/postscript",
            "RTF=application/rtf",
            "ICC=application/vnd.iccprofile",
            "ICM=application/vnd.iccprofile",
            "XLS=application/vnd.ms-excel",
            "PPT=application/vnd.ms-powerpoint",
            "OTF=application/x-font-otf",
            "TTC=application/x-font-ttf",
            "TTF=application/x-font-ttf",
            "AFM=application/x-font-type1",
            "PFB=application/x-font-type1",
            "PFM=application/x-font-type1",
            "INDD=application/x-indesign",
            "XML=application/xml",
            "MP3=audio/mpeg",
            "AAC=audio/x-aac",
            "AIFF=audio/x-aiff",
            "WMA=audio/x-ms-wma",
            "WAV=audio/x-wav",
            "MTS=model/vnd.mts",
            "EPS=application/postscript",
            "PSD=image/vnd.adobe.photoshop",
            "FLA=application/x-shockwave-flash",
            "SWF=application/x-shockwave-flash"
    };

	static final String[] BLACK_LISTED_MIME_TYPES = {"image/x-raw-*", "image/vnd.dxf", "image/vnd.dwg"};
	static final String IGNORED_MIMETYPES_FOR_PROFILE = "cq.dam.scene7.assetmimetypeservice.ignoredforprofile";
	static final String[] DEFAULT_IGNORED_MIMETYPES_FOR_PROFILE = new String[] {"application/pdf"};

	static final String REL_PATH_MIME_TYPES = "mimeTypes";

    /**
     * Returns a configured MIME type for a CQ DAM asset imported from Scene7.
     * 
     * @param asset
     *            the CQ DAM asset
     * @return a String containing the MIME type or <strong>null</strong> if no MIME type has been identified
     */
    public String getMimeTypeForAsset(Asset asset);

    /**
     * Returns a configured mime type for a Scene7 asset type.
     * 
     * @param assetType
     *            the asset type
     * @param extension
     *            the extension for the asset
     * @return a String containing the MIME type or <strong>null</strong> if no MIME type has been identified
     */
    public String getMimeTypeForAssetType(Scene7AssetType assetType, String extension);

	/**
	 * Returns a list of supported MIME types
	 * 
	 * @return a list of String of MIME types
	 */
	public List<String> getSupportedMimeTypes();

	/**
	 * Updates the list of supported MIME types from OSGi configuration to
	 * 
	 * @param s7ConfigResource
	 *            the targeted Scene7 cloud configuration
	 * @param rr
	 *            resource resolver to update Scene7 configuration resource
	 */
	public void updateSupportedMimeTypes(Resource s7ConfigResource,
			ResourceResolver rr);

	/**
	 * Updates the list of supported MIME types from OSGi configuration to
	 */
	public void updateSupportedMimeTypes();

    /**
     * Check if  it's previewable by Scene7 IS server.
     * @param resolver
     * @param resourcePath
     */
    public Boolean isMimetypePreviewable(ResourceResolver resolver, String resourcePath);
	
    /**
    * Check if processing profile use is allowed for mimeType
    * @param mimeType
    */
    public Boolean isMimeTypeAllowedProfileUse(String mimeType);

	/**
	 * Check if mimeType is supported or not
	 * @param mimeType
	 */
	public Boolean isMimeTypeSupported(String mimeType);
}
