/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.tagging;

import java.util.Arrays;
import java.util.Comparator;
import java.util.Locale;
import java.util.Set;
import java.util.TreeSet;

import org.apache.sling.api.resource.Resource;

/**
 * <code>TagCloud</code> helps displaying a tag cloud with n-tile font sizes.
 */
public class TagCloud {
    
    private final Tag[] tags;
    private final long[] frequencies;

    public TagCloud(Tag[] tags) {
        this(tags, null);
    }

    public TagCloud(Tag[] tags, Locale locale) {
        this.tags = tags;
        this.frequencies = calculateFrequencies();
        sortTags(locale);
    }

    @Deprecated
    public TagCloud(TagManager tagManager) {
        this(tagManager, null);
    }
    
    @Deprecated
    public TagCloud(TagManager tagManager, Resource resource) {
        if (resource != null) {
            this.tags = tagManager.getTags(resource);
        } else {
            this.tags = tagManager.findByTitle("*").tags;
        }
        this.frequencies = calculateFrequencies();
        sortTags(null);
    }
    
    private void sortTags(final Locale locale) {
        Arrays.sort(tags, new Comparator<Tag>() {
            public int compare(Tag tag1, Tag tag2) {
                return tag1.getTitle(locale).compareTo(tag2.getTitle(locale));
            }
        });
    }
    
    public Tag[] getTags() {
        return this.tags;
    }
    
    public boolean isEmpty() {
        return this.tags.length == 0;
    }
    
    /**
     * Returns a list of compressed frequencies of the tag counts, ie. each
     * frequency is listed only once in the result.
     */
    private long[] calculateFrequencies() {
        // we need an ordered set here, hence TreeSet
        Set<Long> frequencySet = new TreeSet<Long>();
        for (Tag tag : tags) {
            long count = tag.getCount();
            if (count > 0) {
                frequencySet.add(tag.getCount());
            }
        }
        
        // a bit verbose way to convert the set back to a simple, ordered array of longs
        long[] frequencies = new long[frequencySet.size()];
        int i = 0;
        for (long freq : frequencySet) {
            frequencies[i] = freq;
            i++;
        }
        
        return frequencies;
    }

    /**
     * Calculates the n-tiles (eg. dectiles for n=10) for a specific
     * frequency / tag count, and using the frequency set of the tags
     * in this cloud.
     * 
     * @param tagCount
     *            the tagCount for which to calculate the result
     * @param n
     *            how many tiles the resulting value should have
     * @return a value between 1 and n, the n-tile
     */
    public int calculateNtiles(long tagCount, int n) {
        return calculateNtiles(tagCount, this.frequencies, n);
    }
    
    /**
     * Calculates the n-tiles (eg. dectiles for n=10) for a specific
     * frequency.
     * 
     * @param frequency
     *            the frequency for which to calculate the result
     * @param frequencies
     *            a list of all frequencies
     * @param n
     *            how many tiles the resulting value should have
     * @return a value between 1 and n, the n-tile
     */
    public static int calculateNtiles(long frequency, long[] frequencies, int n) {
        int i = 0;
        while (true) {
            // if we reach the end of the array, return the maximum
            // otherwise if we found the frequency or a higher value in the array
            if ((i >= (frequencies.length - 1)) || (frequencies[i] >= frequency)) {
                return (int) Math.ceil( (double) (i+1) / frequencies.length * n );
            }
            i++;
        }
    }
}
