/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */

package com.day.cq.wcm.api;

import java.util.Calendar;
import java.util.Iterator;
import java.util.Locale;

import org.apache.sling.api.adapter.Adaptable;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ValueMap;
import org.osgi.annotation.versioning.ProviderType;

import com.day.cq.commons.Filter;
import com.day.cq.commons.LabeledResource;
import com.day.cq.tagging.Tag;

/**
 * Defines the interface of a CQ WCM Page.
 */
@ProviderType
public interface Page extends Adaptable, LabeledResource {

    /**
     * Returns the path of the page.
     * @return path of the page.
     */
    String getPath();

    /**
     * Convenience method that returns the manager of this page.
     * @return the page manager.
     */
    PageManager getPageManager();

    /**
     * Returns the resource of the page's content or <code>null</code> if
     * the page has no content.
     * @return resource or <code>null</code>
     *
     * @see #hasContent()
     */
    Resource getContentResource();

    /**
     * Returns the addressed resource of the page's content or <code>null</code>
     * if the respective resource does not exist. If <code>relPath</code> is
     * <code>null</code> or an empty string the toplevel content resource is
     * returned.
     *
     * @param relPath relative path into content
     * @return resource or <code>null</code>
     *
     * @throws IllegalArgumentException if path is not relative.
     */
    Resource getContentResource(String relPath);

    /**
     * Returns an iterator over all child resources that adapt to a page.
     * other child resources are skipped.
     * @return iterator of child pages
     */
    Iterator<Page> listChildren();

    /**
     * Returns an iterator over all child resources that adapt to a page and
     * that are included in the given filter. other child resources are skipped.
     * @param filter for iteration. may be <code>null</code>
     * @return iterator of child pages
     */
    Iterator<Page> listChildren(Filter<Page> filter);

    /**
     * Returns an iterator over descendant resources that adapt to a page and
     * that are included in the given filter. other child resources are skipped.
     * @param filter for iteration. may be <code>null</code>
     * @param deep false traverses only children; true traverses all descendants
     * @return iterator of child pages
     */
    Iterator<Page> listChildren(final Filter<Page> filter, final boolean deep);

    /**
    * Checks if the page has a child page named <code>name</code>
    * @param name Name of the child
    * @return <code>true</code> if the page has this child;
    *         <code>false</code> otherwise.
    */
    boolean hasChild(String name);

    /**
     * Returns the hierarchical depth of the page. The depth is the number of
     * {@link #getParent()} calls would be needed to reach the root node.
     * @return the depth of the page.
     */
    int getDepth();

    /**
     * Returns the parent page if it's resource adapts to page.
     * @return the parent page or <code>null</code>
     */
    Page getParent();

    /**
     * Returns the relative parent page. If no page exists at that level,
     * <code>null</code> is returned.
     *
     * Example (this path == /content/geometrixx/en/products)
     * <pre>
     * | level | returned                        |
     * |     0 | /content/geometrixx/en/products |
     * |     1 | /content/geometrixx/en          |
     * |     2 | /content/geometrixx             |
     * |     3 | /content                        |
     * |     4 | null                            |
     * </pre>
     *
     * @param level hierarchy level of the parent page to retrieve
     * @return the respective parent page or <code>null</code>
     */
    Page getParent(int level);

    /**
     * Returns the absolute parent page. If no page exists at that level,
     * <code>null</code> is returned.
     *
     * Example (this path == /content/geometrixx/en/products)
     * <pre>
     * | level | returned                        |
     * |     0 | /content                        |
     * |     1 | /content/geometrixx             |
     * |     2 | /content/geometrixx/en          |
     * |     3 | /content/geometrixx/en/products |
     * |     4 | null                            |
     * </pre>
     *
     * @param level hierarchy level of the parent page to retrieve
     * @return the respective parent page or <code>null</code>
     */
    Page getAbsoluteParent(int level);

    /**
     * Returns a value map of the page content. If the page has no content
     * an empty map is returned.
     *
     * @return a value map
     * @see #hasContent()
     */
    ValueMap getProperties();

    /**
     * Returns a value map of the address content of the page. If the addressed
     * content does not exit, <code>null</code> is returned. If <code>relPath</code> is
     * <code>null</code> or an empty string the toplevel content properties are
     * returned.
     *
     * @param relPath relative path to the content
     * @return a property map or <code>null</code>
     *
     * @throws IllegalArgumentException of the given path is not relative.
     */
    ValueMap getProperties(String relPath);

    /**
     * Returns the name of the page. this is the last path segment of the
     * page path.
     * @return page name
     */
    String getName();

    /**
     * Returns the title of the page or <code>null</code> if none defined.
     * The title is a short form of the page title, usually displayed in the
     * site admin.
     * @return title of the page
     */
    String getTitle();

    /**
     * Returns the page title of the page or <code>null</code> if none defined.
     * @return page title
     */
    String getPageTitle();

    /**
     * Returns the navigation title of the page or <code>null</code> if none
     * defined. The navigation title is usually used when drawing navigation
     * links.
     * @return navigation title
     */
    String getNavigationTitle();

    /**
     * Returns the "hideInNav" flag or <code>false</code> if non defined.
     * The hideInNav flag is usually used to control if a page should be
     * displayed in a navigation.
     * @return hide in navigation flag
     */
    boolean isHideInNav();

    /**
     * Checks if the page has content attached.
     * @return <code>true</code> if the page has content;
     *         <code>false</code> otherwise.
     */
    boolean hasContent();

    /**
     * Checks if the page is valid. It is valid if it has content and if
     * the on-/off time range spans the current time.
     * @return <code>true</code> if the page is valid;
     *         <code>false</code> otherwise.
     */
    boolean isValid();

    /**
     * Returns the number of milliseconds when this page gets valid. If the page
     * is already valid, <code>0</code> is returned. If the page is out dated,
     * i.e. the offTime is in the past, a negative number is returned. If this
     * page has no content {@link Long#MIN_VALUE} is returned.
     * @return milliseconds until page gets valid.
     */
    long timeUntilValid();

    /**
     * Returns the <code>onTime</code> of the page. The onTime defines after
     * which time it is valid. If no onTime is specified <code>null</code> is
     * returned and the onTime is not respected in the {@link #isValid()}
     * calculation.
     * @return onTime or <code>null</code>
     */
    Calendar getOnTime();

    /**
     * Returns the <code>offTime</code> of the page. The offTime defines before
     * which time it is valid. If no offTime is specified <code>null</code> is
     * returned and the offTime is not respected in the {@link #isValid()}
     * calculation.
     * @return offTime or <code>null</code>
     */
    Calendar getOffTime();

    /**
     * Checks if the page is in a deletion process
     *
     * @return <code>modification date</code> if the page is in process to be deleted
     *         <code>null</code> otherwise.
     */
    Calendar getDeleted();

    /**
     * Return the user id that triggered the deletion of page
     * or <code>null</code> if the page was not deleted.
     *
     * @return user id or <code>null</code>
     */
    String getDeletedBy();

    /**
     * Return the user id that last modified the page or <code>null</code> if this
     * information is not available or if the page has no content.
     * @return user id or <code>null</code>
     */
    String getLastModifiedBy();

    /**
     * Returns the time when the page was last modified or <code>null</code> if
     * this information is not available or if the page has no content.
     * @return last modification date or <code>null</code>
     */
    Calendar getLastModified();

    /**
     * Returns the vanity url.
     * @return The vanity url for this page or <code>null</code>
     */
    String getVanityUrl();
    
    /**
     * Returns the tags set on this page.
     * @return The tags set on this page.
     */
    Tag[] getTags();

    /**
     * Lock a page.
     *
     * @throws WCMException if an error during this operation occurs.
     */
    void lock() throws WCMException;

    /**
     * Checks if this page is locked.
     * @return <code>true</code> if this page is locked.
     */
    boolean isLocked();

    /**
     * Return the lock owner of a page.
     *
     * @return lock owner; <code>null</code> if the page is not locked
     */
    String getLockOwner();

    /**
     * Return a flag indicating whether the current user can unlock a page.
     *
     * @return <code>true</code> if the page can be unlocked;
     *         <code>false</code> otherwise
     */
    boolean canUnlock();

    /**
     * Unlock a page.
     *
     * @throws WCMException if an error during this operation occurs.
     */
    void unlock() throws WCMException;

    /**
     * Returns the template that was used to create this page or <code>null</code>
     * if none defined or the current user does not have read access to the 
     * template resource specified by the path in the page's template property. 
     * This is the case on publish instances using default ACL configuration 
     * (anonymous cannot read templates).
     *
     * @return template or <code>null</code>
     */
    Template getTemplate();

    /**
     * Returns the content language of the page. The language is usually defined
     * on the page content via a <code>jcr:language</code> property containing
     * the iso codes for language and country. if the property is not defined
     * on this page all ancestors are search for such an property. If no language
     * is defined at all, the path is examined if it contains a iso label. If no
     * language can be found at all, the systems default locale is returned.
     *
     * If <code>ignoreConent</code> is <code>true</code>, only the names of the
     * path is used to determine the language.
     * 
     * Note that this has nothing to do with i18n of the cq5 itself.
     *
     * @param ignoreContent if <code>true</code> only the path is used to
     *        determine the language.
     * @return the language of the content
     */
    Locale getLanguage(boolean ignoreContent);

    /**
     * Returns the content language of the page. The language is usually defined
     * on the page content via a <code>jcr:language</code> property containing
     * the iso codes for language and country. if the property is not defined
     * on this page all ancestors are search for such an property. If no language
     * is defined at all, the path is examined if it contains a iso label. If no
     * language can be found at all, the systems default locale is returned.
     *
     * Note that this has nothing to do with i18n of the cq5 itself.
     *
     * @return the language of the content
     */
    Locale getLanguage();

}
