package org.json;

import java.util.Iterator;

/**
 * This provides static methods to convert an XML text into a JSONObject, and to
 * covert a JSONObject into an XML text.
 *
 * @author JSON.org
 * @version 2016-08-10
 * @deprecated Use javax.json or other other json libraries instead
 */
@Deprecated
public class XML {

    /** The Character '&amp;'. */
    public static final Character AMP = new Character('&');
    /** The Character '''. */
    public static final Character APOS = new Character('\'');
    /** The Character '!'. */
    public static final Character BANG = new Character('!');
    /** The Character '='. */
    public static final Character EQ = new Character('=');
    /** The Character '>'. */
    public static final Character GT = new Character('>');
    /** The Character '&lt;'. */
    public static final Character LT = new Character('<');
    /** The Character '?'. */
    public static final Character QUEST = new Character('?');
    /** The Character '"'. */
    public static final Character QUOT = new Character('"');
    /** The Character '/'. */
    public static final Character SLASH = new Character('/');

    /**
     * Replace special characters with XML escapes:
     *
     * <pre>
     * &amp; <small>(ampersand)</small> is replaced by &amp;amp;
     * &lt; <small>(less than)</small> is replaced by &amp;lt;
     * &gt; <small>(greater than)</small> is replaced by &amp;gt;
     * &quot; <small>(double quote)</small> is replaced by &amp;quot;
     * &apos; <small>(single quote / apostrophe)</small> is replaced by &amp;apos;
     * </pre>
     *
     * @param string
     *            The string to be escaped.
     * @return The escaped string.
     */
    public static String escape(String string) {
        StringBuffer sb = new StringBuffer();
        int i = 0;

        for (int length = string.length(); i < length; ++i) {
            char c = string.charAt(i);
            switch (c) {
                case '"' :
                    sb.append("&quot;");
                    break;
                case '&' :
                    sb.append("&amp;");
                    break;
                case '\'' :
                    sb.append("&apos;");
                    break;
                case '<' :
                    sb.append("&lt;");
                    break;
                case '>' :
                    sb.append("&gt;");
                    break;
                default :
                    sb.append(c);
            }
        }

        return sb.toString();
    }

    /**
     * Throw an exception if the string contains whitespace. Whitespace is not
     * allowed in tagNames and attributes.
     *
     * @param string
     *            A string.
     * @throws JSONException Thrown if the string contains whitespace or is empty.
     */
    public static void noSpace(String string) throws JSONException {
        int length = string.length();
        if (length == 0) {
            throw new JSONException("Empty string.");
        } else {
            for (int i = 0; i < length; ++i) {
                if (Character.isWhitespace(string.charAt(i))) {
                    throw new JSONException("'" + string + "' contains a space character.");
                }
            }

        }
    }

    private static boolean parse(XMLTokener x, JSONObject context, String name) throws JSONException {
        JSONObject jsonobject = null;
        Object token = x.nextToken();
        String string;
        if (token == BANG) {
            char c = x.next();
            if (c == '-') {
                if (x.next() == '-') {
                    x.skipPast("-->");
                    return false;
                }

                x.back();
            } else if (c == '[') {
                token = x.nextToken();
                if (token.equals("CDATA") && x.next() == '[') {
                    string = x.nextCDATA();
                    if (string.length() > 0) {
                        context.accumulate("content", string);
                    }

                    return false;
                }

                throw x.syntaxError("Expected 'CDATA['");
            }

            int i = 1;

            do {
                token = x.nextMeta();
                if (token == null) {
                    throw x.syntaxError("Missing '>' after '<!'.");
                }

                if (token == LT) {
                    ++i;
                } else if (token == GT) {
                    --i;
                }
            } while (i > 0);

            return false;
        } else if (token == QUEST) {
            x.skipPast("?>");
            return false;
        } else if (token == SLASH) {
            token = x.nextToken();
            if (name == null) {
                throw x.syntaxError("Mismatched close tag " + token);
            } else if (!token.equals(name)) {
                throw x.syntaxError("Mismatched " + name + " and " + token);
            } else if (x.nextToken() != GT) {
                throw x.syntaxError("Misshaped close tag");
            } else {
                return true;
            }
        } else if (token instanceof Character) {
            throw x.syntaxError("Misshaped tag");
        } else {
            String tagName = (String) token;
            token = null;
            jsonobject = new JSONObject();

            while (true) {
                if (token == null) {
                    token = x.nextToken();
                }

                if (!(token instanceof String)) {
                    if (token == SLASH) {
                        if (x.nextToken() != GT) {
                            throw x.syntaxError("Misshaped tag");
                        }

                        if (jsonobject.length() > 0) {
                            context.accumulate(tagName, jsonobject);
                        } else {
                            context.accumulate(tagName, "");
                        }

                        return false;
                    }

                    if (token != GT) {
                        throw x.syntaxError("Misshaped tag");
                    }

                    while (true) {
                        token = x.nextContent();
                        if (token == null) {
                            if (tagName != null) {
                                throw x.syntaxError("Unclosed tag " + tagName);
                            }

                            return false;
                        }

                        if (token instanceof String) {
                            string = (String) token;
                            if (string.length() > 0) {
                                jsonobject.accumulate("content", stringToValue(string));
                            }
                        } else if (token == LT && parse(x, jsonobject, tagName)) {
                            if (jsonobject.length() == 0) {
                                context.accumulate(tagName, "");
                            } else if (jsonobject.length() == 1 && jsonobject.opt("content") != null) {
                                context.accumulate(tagName, jsonobject.opt("content"));
                            } else {
                                context.accumulate(tagName, jsonobject);
                            }

                            return false;
                        }
                    }
                }

                string = (String) token;
                token = x.nextToken();
                if (token == EQ) {
                    token = x.nextToken();
                    if (!(token instanceof String)) {
                        throw x.syntaxError("Missing value");
                    }

                    jsonobject.accumulate(string, stringToValue((String) token));
                    token = null;
                } else {
                    jsonobject.accumulate(string, "");
                }
            }
        }
    }

    /**
     * This method is the same as {@link JSONObject#stringToValue(String)}.
     *
     * @param string String to convert
     * @return JSON value of this string or the string
     */
    public static Object stringToValue(String string) {
        if (string.equals("")) {
            return string;
        } else if (string.equalsIgnoreCase("true")) {
            return Boolean.TRUE;
        } else if (string.equalsIgnoreCase("false")) {
            return Boolean.FALSE;
        } else if (string.equalsIgnoreCase("null")) {
            return JSONObject.NULL;
        } else if (string.equals("0")) {
            return new Integer(0);
        } else {
            try {
                char initial = string.charAt(0);
                boolean negative = false;
                if (initial == '-') {
                    initial = string.charAt(1);
                    negative = true;
                }

                if (initial == '0' && string.charAt(negative ? 2 : 1) == '0') {
                    return string;
                }

                if (initial >= '0' && initial <= '9') {
                    if (string.indexOf(46) >= 0) {
                        return Double.valueOf(string);
                    }

                    if (string.indexOf(101) < 0 && string.indexOf(69) < 0) {
                        Long myLong = new Long(string);
                        if (myLong == myLong.intValue()) {
                            return new Integer(myLong.intValue());
                        }

                        return myLong;
                    }
                }
            } catch (Exception var4) {

            }

            return string;
        }
    }

    /**
     * Convert a well-formed (but not necessarily valid) XML string into a
     * JSONObject. Some information may be lost in this transformation because
     * JSON is a data format and XML is a document format. XML uses elements,
     * attributes, and content text, while JSON uses unordered collections of
     * name/value pairs and arrays of values. JSON does not does not like to
     * distinguish between elements and attributes. Sequences of similar
     * elements are represented as JSONArrays. Content text may be placed in a
     * "content" member. Comments, prologs, DTDs, and <code>&lt;[ [ ]]></code>
     * are ignored.
     *
     * @param string
     *            The source string.
     * @return A JSONObject containing the structured data from the XML string.
     * @throws JSONException Thrown if there is an errors while parsing the string
     */
    public static JSONObject toJSONObject(String string) throws JSONException {
        JSONObject jo = new JSONObject();
        XMLTokener x = new XMLTokener(string);

        while (x.more() && x.skipPast("<")) {
            parse(x, jo, (String) null);
        }

        return jo;
    }

    /**
     * Convert a JSONObject into a well-formed, element-normal XML string.
     *
     * @param object
     *            A JSONObject.
     * @return A string.
     * @throws JSONException Thrown if there is an error parsing the string
     */
    public static String toString(Object object) throws JSONException {
        return toString(object, (String) null);
    }

    /**
     * Convert a JSONObject into a well-formed, element-normal XML string.
     *
     * @param object
     *            A JSONObject.
     * @param tagName
     *            The optional name of the enclosing tag.
     * @return A string.
     * @throws JSONException Thrown if there is an error parsing the string
     */
    public static String toString(Object object, String tagName) throws JSONException {
        StringBuffer sb = new StringBuffer();
        int i;
        JSONArray ja;
        int length;
        String string;
        if (!(object instanceof JSONObject)) {
            if (object.getClass().isArray()) {
                object = new JSONArray(object);
            }

            if (object instanceof JSONArray) {
                ja = (JSONArray) object;
                length = ja.length();

                for (i = 0; i < length; ++i) {
                    sb.append(toString(ja.opt(i), tagName == null ? "array" : tagName));
                }

                return sb.toString();
            } else {
                string = object == null ? "null" : escape(object.toString());
                return tagName == null
                        ? "\"" + string + "\""
                        : (string.length() == 0
                                ? "<" + tagName + "/>"
                                : "<" + tagName + ">" + string + "</" + tagName + ">");
            }
        } else {
            if (tagName != null) {
                sb.append('<');
                sb.append(tagName);
                sb.append('>');
            }

            JSONObject jo = (JSONObject) object;
            Iterator keys = jo.keys();

            while (true) {
                while (true) {
                    while (keys.hasNext()) {
                        String key = keys.next().toString();
                        Object value = jo.opt(key);
                        if (value == null) {
                            value = "";
                        }

                        if (value instanceof String) {
                            string = (String) value;
                        } else {
                            string = null;
                        }

                        if (key.equals("content")) {
                            if (value instanceof JSONArray) {
                                ja = (JSONArray) value;
                                length = ja.length();

                                for (i = 0; i < length; ++i) {
                                    if (i > 0) {
                                        sb.append('\n');
                                    }

                                    sb.append(escape(ja.get(i).toString()));
                                }
                            } else {
                                sb.append(escape(value.toString()));
                            }
                        } else if (value instanceof JSONArray) {
                            ja = (JSONArray) value;
                            length = ja.length();

                            for (i = 0; i < length; ++i) {
                                value = ja.get(i);
                                if (value instanceof JSONArray) {
                                    sb.append('<');
                                    sb.append(key);
                                    sb.append('>');
                                    sb.append(toString(value));
                                    sb.append("</");
                                    sb.append(key);
                                    sb.append('>');
                                } else {
                                    sb.append(toString(value, key));
                                }
                            }
                        } else if (value.equals("")) {
                            sb.append('<');
                            sb.append(key);
                            sb.append("/>");
                        } else {
                            sb.append(toString(value, key));
                        }
                    }

                    if (tagName != null) {
                        sb.append("</");
                        sb.append(tagName);
                        sb.append('>');
                    }

                    return sb.toString();
                }
            }
        }
    }
}
