/*
 * File: F26Dot6.java
 * 
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is
 *      obtained from Adobe Systems Incorporated.
 *
 */

package com.adobe.fontengine.math;

import com.adobe.agl.text.DecimalFormat;

/** Constants and operations on fixed 26.6 numbers.
 * 
 * The value <it>v</it> is represented by the <code>int</code> value
 * 2^6 * <it>v</it>.
 */

public final class F26Dot6 {
  public static final int /*26.6*/ ZERO = 0x00;
  public static final int /*26.6*/ ONE = 1 << 6;
  public static final int /*26.6*/ ONE_HALF = ONE / 2;
  public static final int /*26.6*/ MAX_VALUE = Integer.MAX_VALUE;
  public static final int /*26.6*/ MIN_VALUE = Integer.MIN_VALUE;
  
  public static int /*26.6*/ truncate (int /*26.6*/ v) {
    return v & 0xffffffc0;
  }
  
  public static int /*26.6*/ ceiling (int /*26.6*/ v) {
    return truncate (v + ONE - 1);
  }
  
  public static int /*26.6*/ floor (int /*26.6*/ v) {
    return truncate (v);
  }
  
  public static int /*26.6*/ round (int /*26.6*/ v) {
    return truncate (v + ONE_HALF);
  }
  
  public static int /*26.6*/ roundHalfUp (int /*26.6*/ v) {
    return truncate (v + ONE_HALF); 
  }
  
  public static int /*26.6*/ roundHalfDown (int /*26.6*/ v) {
    return truncate (v + ONE_HALF - 1);
  }
  
  public static int /*26.6*/ negate (int /*26.6*/ v) {
    return -v; 
  }
  
  public static int /*26.6*/ abs (int /*26.6*/ v) {
    return (v < 0) ? -v : v;
  }
  
  public static int /*26.6*/ add (int /*26.6*/ v1, int /*26.6*/ v2) {
    return v1 + v2; 
  }
  
  public static int /*26.6*/ subtract (int /*26.6*/ v1, int /*26.6*/ v2) {
    return v1 - v2; 
  }
  
  private static int /*26.6*/ clamp (long x) {
    if (x > Integer.MAX_VALUE) {
      return Integer.MAX_VALUE; }
    else if (x < Integer.MIN_VALUE) {
      return Integer.MIN_VALUE; }
    else {
      return (int) x; }
  }
  
  public static int /*26.6*/ multiply (int /*26.6*/ v1, int /*26.6*/ v2) {
    return clamp (((long) v1 * (long) v2) >> 6);
  }
  
  public static int /*26.6*/ divide (int /*26.6*/ v1, int /*26.6*/ v2) {
    return (int) (((long) v1 ) << 6 ) / v2;
  }
  
  public static int /*26.6*/ multiplyByF2Dot14 (int /*26.6*/ v1, int /*2.14*/ v2) {
    return clamp (((((long) v1 * (long) (short) v2) >> 13) + 1) >> 1);
  }
  
  public static int /*26.6*/ divideByF2Dot14 (int /*26.6*/ v1, int /*2.14*/ v2) {
    return (int) (((long) v1) << 14) / v2;
  }
  
  public static int /*26.6*/ multiplyByF16Dot16 (int /*26.6*/ v1, int /*16.16*/ v2) {
    return clamp (((((long) v1 * (long) v2) >> 15) + 1) >> 1);
  }
  
  public static int /*26.6*/ divideByF16Dot16 (int /*26.6*/ v1, int /*16.16*/ v2) {
	return (int) ((((long) v1) << 16) / v2);
  }
  
  public static int /*26.6*/ multiplyDivide (int /*26.6*/ v1, int /*26.6*/ v2, int /*26.6*/ v3) {
    long numerator = ((long) v1) * ((long) v2);
    long denominator = v3;
    boolean negate = false;
    if (numerator < 0) {
      numerator = - numerator;
      negate = ! negate; }
    if (denominator < 0) {
      denominator = - denominator;
      negate = ! negate; }
    long v = (numerator + denominator / 2) / denominator;
    if (negate) {
      return clamp (-v); }
    else {
      return clamp (v); }
  }
  
  public static int /*26.6*/ multiplyByF2Dot14DivideByF2Dot14 (int /*26.6*/ v1, int /*2.14*/ v2, int /*2.14*/ v3) {
    return multiplyDivide (v1, v2, v3);
  }
  
  public static boolean isEven (int /*26.6*/ v1) {
    return (v1 & 0x40) == 0;
  }
  
  public static boolean sameSign (int /*26.6*/ v1, int /*26.6*/ v2) {
    return (v1 & 0x80000000) == (v2 & 0x80000000);
  }
  
  public static int min /*26.6*/ (int /*26.6*/ v1, int /*2.14*/ v2) {
    return Math.min (v1, v2);
  }
  
  public static int max /*26.6*/ (int /*26.6*/ v1, int /*2.14*/ v2) {
    return Math.max (v1, v2);
  }
  
  
  public static double toDouble (int /*26.6*/ v) {
    return ((double) v) / ONE;
  }
  
  public static int /*26.6*/ fromDouble (double v) {
    return (int) Math.round (v * ONE);
  }
  
  public static int toInt (int /*26.6*/ v) {
    return v >> 6;
  }
  
  public static int /*26.6*/ fromInt (int v) {
    return v << 6;
  }
  
 
  private final static DecimalFormat df = new DecimalFormat ("0.##");
  
  public static String toString (int /*26.6*/ v) {
    return df.format (toDouble (v));
  }
}