/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.workflow.exec;

import com.adobe.granite.workflow.WorkflowException;
import com.adobe.granite.workflow.WorkflowSession;
import com.adobe.granite.workflow.metadata.MetaDataMap;

import java.io.Serializable;

/**
 * This is the interface to use to proxy to <code>{@link WorkflowExternalProcess}</code>s) services.
 */
public interface WorkflowExternalProcessProxy {
    /**
     * Executes the external process step. In order to keep track of this asynchronous process, an identifier must be returned. One can return <code>null</code>
     * to indicate that nothing was executed. In this case, the step directly advances to the next step.
     * @see WorkflowExternalProcess#execute(WorkItem, com.adobe.granite.workflow.WorkflowSession, com.adobe.granite.workflow.metadata.MetaDataMap)
     *
     * @param processName The name of the WorkflowExternalProcess service to proxy
     * @param item      The current work item
     * @param session   The current workflow session
     * @param arguments The current steps node arguments (aka as node meta data)
     * @return Any serializable identifier for the external process executed, or <code>null</code>
     * @throws com.adobe.granite.workflow.WorkflowException thrown in case
     *     something goes wrong during execution
     */
    public Serializable execute(String processName, WorkItem item, WorkflowSession session, MetaDataMap arguments) throws WorkflowException;

    /**
     * Tests the external process, given by the external process id, for completeness.
     * @see WorkflowExternalProcess#hasFinished(java.io.Serializable, WorkItem, com.adobe.granite.workflow.WorkflowSession, com.adobe.granite.workflow.metadata.MetaDataMap)
     *
     * @param processName The name of the WorkflowExternalProcess service to proxy
     * @param externalProcessId The external process to test for completeness
     * @param item              The current work item
     * @param session           The current workflow session
     * @param arguments         The current steps node arguments (aka as node meta data)
     * @return {@code true} if the specified external process has finished 
     */
    public boolean hasFinished(String processName, Serializable externalProcessId, WorkItem item, WorkflowSession session, MetaDataMap arguments);

    /**
     * Get and handle the result of the external process, given by the external process id. This would include storing any results where appropriate for
     * further use.
     * @see WorkflowExternalProcess#handleResult(java.io.Serializable, WorkItem, com.adobe.granite.workflow.WorkflowSession, com.adobe.granite.workflow.metadata.MetaDataMap)
     *
     * @param processName The name of the WorkflowExternalProcess service to proxy
     * @param externalProcessId The external process to handle the results for
     * @param workItem          The current work item
     * @param session           The current workflow session
     * @param arguments         The current steps node arguments (aka as node meta data)
     * @throws com.adobe.granite.workflow.WorkflowException thrown in case
     *     something goes wrong during execution
     */
    public void handleResult(String processName, Serializable externalProcessId, WorkItem workItem, WorkflowSession session, MetaDataMap arguments) throws WorkflowException;

    /**
     * returns true if this proxy can execute requests on behalf of the specified processName
     * @param processName The name of the WorkflowExternalProcess service to proxy
     * @return true if this proxy can handle the request, false otherwise.
     */
    boolean canHandle(String processName);
}
