package com.adobe.xfa.text;

import com.adobe.xfa.ut.CoordPair;
import com.adobe.xfa.ut.Rect;
import com.adobe.xfa.ut.UnitSpan;

/**
 * @exclude from published api.
 */

abstract class FrameCaret extends FrameLineHelper {
	private int meCaret;
	private int mnValidFrameIndex;
	private int mnValidLineIndex;
	private UnitSpan moValidOffset = UnitSpan.ZERO;

	FrameCaret (TextSparseStream poStream, int eProcess) {
		super (poStream, eProcess);
		meCaret = DispLine.CARET_INVALID;
		mnValidFrameIndex = 0;
		mnValidLineIndex = 0;
	}

	boolean success () {
		return meCaret != DispLine.CARET_INVALID;
	}

	protected boolean finish () {
// Note: Normally caret processing will dynamically load frames as
// required.  However, that load will be suppressed if there is a caret
// call during a dynamic load operation.  This could cause the base class
// to run to the end of frames and end up with an invalid frame index.
		if (meCaret != DispLine.CARET_INVALID) {
			setFrameIndex (mnValidFrameIndex);
			setLineIndex (mnValidLineIndex);
			setOffset (moValidOffset);
		}
		return true;
	}

	protected boolean processOneLine (DispLineWrapped poLine) {
		int eCaret = processCaretLine (poLine);

		if (eCaret != DispLine.CARET_INVALID) {
			meCaret = eCaret;
			mnValidFrameIndex = getFrameIndex();
			mnValidLineIndex = getLineIndex();
			moValidOffset = getOffset();
			if (eCaret == DispLine.CARET_PRESENT) {
				return false; // stop
			}
		}

		return true; // keep looking
	}

	abstract int processCaretLine (DispLineWrapped poLine);
}

class FrameCaretRect extends FrameCaret {
	private final TextPosnBase moPosition;
	private final boolean mbAllowDangling;
	private final boolean mbAB;
	private final DispLineWrapped.CaretInfo mCaretInfo = new DispLineWrapped.CaretInfo();
	private Rect mCaret;

	FrameCaretRect (TextSparseStream poStream, TextPosnBase oPosition, boolean bAllowDangling, boolean bAB) {
		super (poStream, PROCESS_NULL_FRAMES);
		moPosition = new TextPosnBase (oPosition);
		mbAllowDangling = bAllowDangling;
		mbAB = bAB;
		moPosition.position (TextPosn.POSN_BEFORE);
	}

	Rect getCaret () {
		return mCaret;
	}

	TextPosnBase getPosition () {
		return moPosition;
	}

	protected boolean processOneFrame (TextFrame poFrame) {
		if (poFrame == null) {
			if (getStream().isAutoLoadSuppressed()) {
				return true;
			}

			int nFrameIndex = getFrameIndex();
			poFrame = getStream().getFrame (nFrameIndex, true);
			assert (poFrame != null);
			assert (poFrame.isNullFrame() != null);

			if (! moPosition.isSamePosition (poFrame.getStart())) {
				if ((nFrameIndex + 1) != getStream().getFrameCount()) {
					return true;
				}
				moPosition.position (TextPosn.POSN_AFTER);
			}

			poFrame = getStream().forceFrame (nFrameIndex);
			setFrame (poFrame);
		}

		return super.processOneFrame (poFrame);
	}

	protected int processCaretLine (DispLineWrapped poLine) {
		int eCaret = poLine.getCaretRect (moPosition, mbAB, mCaretInfo);
		if (eCaret != DispLine.CARET_INVALID) {
			mCaret = mCaretInfo.mCaret;
			if (mbAllowDangling && (eCaret == DispLine.CARET_CONDITIONAL)) {
				eCaret = DispLine.CARET_PRESENT;
			}
		}

		return eCaret;
	}
}

class FrameCaretPosn extends FrameCaret {
	private TextStream mpoSearchStream;
	private CoordPair moSearchPoint;
	private boolean mbAllowDescendents;
	private TextPosnBase mpoResult;

	FrameCaretPosn (TextSparseStream poStream, TextStream poSearchStream, CoordPair oSearchPoint, boolean bAllowDescendents, TextPosnBase poResult) {
		super (poStream, PROCESS_LOADED_FRAMES);
		mpoSearchStream = poSearchStream;
		moSearchPoint = oSearchPoint;
		mbAllowDescendents = bAllowDescendents;
		mpoResult = poResult;
	}

	protected int processCaretLine (DispLineWrapped poLine) {
		TextFrame poFrame = getFrame();
		int nCurrentFrame = getFrameIndex();
		int nCurrentLine = getLineIndex();

		CoordPair oSearch = new CoordPair (moSearchPoint.x(), moSearchPoint.y().subtract (getOffset()));
		oSearch = ABXY.toAB (poLine.getXYOrigin(), oSearch, poFrame.getLayoutOrientation());

		if ((nCurrentFrame > 0) || (nCurrentLine > 0)) {
			if (oSearch.y().value() < 0) {
				return DispLine.CARET_INVALID; // above top of this line
			}
		}

		if (((nCurrentFrame + 1) < poFrame.getStream().getFrameCount()) || ((nCurrentLine + 1) < poFrame.getLineCount())) {
			if (oSearch.y().gt (poLine.getBExtent())) {
				return DispLine.CARET_INVALID; // below bottom of this line
			}
		}
		return poLine.getCaretPosn (mpoSearchStream, oSearch.x(), mpoResult, mbAllowDescendents);
	}
}

class FrameCaretStartEnd extends FrameCaret {
	private TextPosnBase moPosition;
	private boolean mbEnd;
	private boolean mbVisual;
	private TextPosnBase mpoResult;

	FrameCaretStartEnd (TextSparseStream poStream, TextPosnBase oPosition, boolean bEnd, boolean bVisual, TextPosnBase poResult) {
		super (poStream, PROCESS_LOADED_FRAMES);
		moPosition = oPosition;
		mbEnd = bEnd;
		mbVisual = bVisual;
		mpoResult = poResult;
	}

	protected int processCaretLine (DispLineWrapped poLine) {
		int eCaret = poLine.validateCaretPosn (moPosition, true);
		if (eCaret != DispLine.CARET_INVALID) {
			poLine.getCaretStartEnd (moPosition.stream(), mbEnd, mbVisual, mpoResult);
		}

		return eCaret; // keep looking if not CARET_PRESENT
	}
}
