/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */

package com.adobe.xfa.ut;

/**
 * This class provides an implementation of the Peer interface.
 * Classes that implement Peer will typically hold an instance of this interface
 * and delegate calls on the Peer interface to the instance of PeerImpl..
 * 
 * <p>
 * A <code>Peer</code> can add another <code>Peer</code> to its peer list
 * via <code>addPeer()</code>. All <code>Peers</code> in the peer list will
 * be notified of peer changes when <code>updateFromPeer()</code> is called.
 *
 * @exclude from published api -- Mike Tardif, May 2006.
 */
public final class PeerImpl {
	
	static private final int REALLOC_SIZE = 4;

	// if muted and deaf flags
	private boolean mbDeaf;
	private boolean mbMute;

	private Peer[] mPeeredNodeList;
	private Peer[] mPeerList;

	private final Peer mSrc;

	public PeerImpl(Peer src) {
		mSrc = src;
	}

	/**
	 * Add a peer node to be notified of state changes.
	 * 
	 * @param peerNode -
	 *            The reference to the peer object to be added.
	 */
	public void addPeer(Peer peerNode) {
		if (peerNode != null) {
			peerNode.addPeeredNode(mSrc);
			mPeerList = addToList(peerNode, mPeerList);
		}
	}

	/**
	 * Called when a peer adds this as a peer.
	 * 
	 * @param peer the peer adding this as a peer.
	 */
	public void addPeeredNode(Peer peer) {
		mPeeredNodeList = addToList(peer, mPeeredNodeList);
	}

	private Peer[] addToList(Peer peer, Peer[] peerList) {
		if (peerList == null) {
			peerList = new Peer[REALLOC_SIZE];
			peerList[0] = peer;
			return peerList;
		}
		
		final int nCnt = peerList.length;

		// Check if we have enough space allocated to add one more
		for (int i = 0; i < nCnt; i++) {
			if (peerList[i] == null) {
				peerList[i] = peer;
				return peerList;
			}			
		}
		
		// Reallocate to add more space
		Peer newList[] = new Peer[nCnt + REALLOC_SIZE];
		System.arraycopy(peerList, 0, newList, 0, nCnt);
		newList[nCnt] = peer;
		return newList;
	}

	/**
	 * clear all peer relationships
	 * @exclude
	 */
	public void clearPeers() {

		// removePeer() will shuffle all the entries in the
		// array down one notch if a previous entry is removed.
		// Delete from the end to avoid missing any entries.
		
		if (mPeeredNodeList != null) {
			for (int i = mPeeredNodeList.length - 1; i >= 0; i--) {
				Peer peer = mPeeredNodeList[i];
				if (peer != null) {					
					peer.removePeer(mSrc);
				}
			}
		}

		// Let all peers know that we're going away by removing this Peer from
		// their mPeeredNodeList list.

		if (mPeerList != null) {
			for (int i = 0; i < mPeerList.length; i++) {
				Peer peer = mPeerList[i];
				if (peer != null) {
					peer.removePeeredNode(mSrc);
					mPeerList[i] = null;
				}
			}
		}
	}

	public void deafen() {
		mbDeaf = true;
	}

	/**
	 * return the requested peer
	 * 
	 * @param nPeer -
	 *            the 0-based position of the peer to retrieve.
	 * @return the peer at the requested position. When there are not more peers
	 *         to return, this will return a null object.
	 */
	public Peer getPeer(int nPeer /* =0 */) {
		if (mPeerList != null && nPeer < mPeerList.length) {
			return mPeerList[nPeer];
		}

		return null;
	}

	public boolean isDeaf() {
		return mbDeaf;
	}

	public boolean isMute() {
		return mbMute;
	}

	public void mute() {
		mbMute = true;
	}

	public void notifyPeers(int eventType, String arg1, Object arg2) {
		if (mbMute)
			return;

		if (mPeerList == null)
			return;

		for (int i = 0; i < mPeerList.length; i++) {
			Peer peer = mPeerList[i];
			
			if (peer == null)
				break;
			
			if (!peer.isDeaf())
				peer.updateFromPeer(mSrc, eventType, arg1, arg2);
		}
	}

	public void removeFromList(Peer peer, Peer[] peerList) {
		if (peerList != null) {
			
			final int nCnt = peerList.length;

			for (int i = 0; i < nCnt; i++) {
				if (peerList[i] == null)
					break;

				if (peerList[i] == peer) {
					System.arraycopy(peerList, i + 1, peerList, i, nCnt - i - 1);
					peerList[nCnt - 1] = null;
					return;
				}
			}
		}
	}

	/**
	 * Remove a peer node from the notification list.
	 * 
	 * @param peerNode -
	 *            The reference to the peer object to be removed.
	 */
	public void removePeer(Peer peerNode) {
		if (peerNode != null) {
			removeFromList(peerNode, mPeerList);
			peerNode.removePeeredNode(mSrc);
		}
	}

	public void removePeeredNode(Peer peer) {
		removeFromList(peer, mPeeredNodeList);
		mSrc.peerRemoved(peer);
	}

	public void unDeafen() {
		mbDeaf = false;
	}

	public void unMute() {
		mbMute = false;
	}
}