/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.cq.gfx;

import java.io.InputStream;

import org.apache.sling.api.resource.ResourceResolver;

import aQute.bnd.annotation.ProviderType;

/**
 * Graphics rendering service.
 *
 * <p>
 * A {@link Plan} describes the rendering operations.
 * These follow the <a href="http://microsite.omniture.com/t2/help/en_US/s7/is_ir_api/index.html#is_api-http_ref-Image_Serving_HTTP_Protocol_Command___Overview">
 * Scene7 Image Serving Protocol</a>. Different {@link Renderer} implementations
 * can be active and differ in the supported set of operations or input file formats. The best available
 * renderer will be picked depending on the plan.
 *
 * <h2>General principles</h2>
 * <ul>
 *     <li>
 *         completely descriptive
 *         <ul>
 *             <li>
 *                 makes no assumption about the actual implementation
 *             </li>
 *             <li>
 *                 allows for optimizations (examples: tile-based processing, use of a pre-created file
 *                 with already scaled layers (pyramid tiff), reorganization of operations)
 *             </li>
 *         </ul>
 *     </li>
 *     <li>
 *         extensible
 *         <ul>
 *             <li>
 *                 map-like interface, operation names are strings
 *             </li>
 *             <li>
 *                 conceptually not restricted to 2D image rendering, could apply to 3D, audio, video
 *                 or any asset processing as well (special commands would indicate that, but "Plan"
 *                 data structure identical)
 *             </li>
 *         </ul>
 *     </li>
 *     <li>
 *         input
 *         <ul>
 *             <li>
 *                 referenced files in the JCR repository (usually using "src" attributes)
 *             </li>
 *             <li>
 *                 or plain drawing operations from scratch
 *             </li>
 *             <li>
 *                 specific renderer define what source formats are supported and what mechanism might
 *                 be used to get an image rendition
 *             </li>
 *             <li>
 *                 referenced files in the resource tree (usually using "src" attributes, usually JCR)
 *                 can have rendering operations attached ("modifiers"), these will be automatically applied
 *             </li>
 *         </ul>
 *     </li>
 *     <li>
 *         output
 *         <ul>
 *             <li>
 *                 a binary stream of a serialized image file
 *             </li>
 *             <li>
 *                 allows to store it back in the repository or stream it out over HTTP
 *             </li>
 *         </ul>
 *     </li>
 * </ul>
 *
 * <h2>Example usage</h2>
 * <p>
 * Resize an image from the repository and output it as png file:
 * <pre>
     // build a plan from scratch
     Plan plan = gfx.createPlan();

     // single asset as source (single layer only)
     plan.layer(0).set("src", "/content/dam/asset.jpg");

     Plan.Map view = plan.view();
     // define target size
     view.set("wid", 300);
     view.set("hei", 200);
     // png format
     view.set("fmt", "png");

     InputStream in;
     try {
         // start rendering
         in = gfx.render(plan, resourceResolver);
         // use stream to store in repository or stream as servlet response
         // ...
     } finally {
         // ensure the stream is closed (to clean up temporary files etc.)
         IOUtils.closeQuietly(in);
     }
 * </pre>
 */
@ProviderType
public interface Gfx {

    /**
     * Create an empty plan that can be set up from scratch.
     * @return an empty rendering plan
     */
    Plan createPlan();

    // - create plan from JCR
    // Plan createPlan(Resource resource);

    // - create plan from URL query string (special case for /is/image, should not be exposed)
    // Plan createPlan(String queryString);

    /**
     * Renders the given plan and returns a binary file stream, typically an
     * image file format, depending on the plan.
     *
     * @param plan describes the rendering operations
     * @param resolver a resource resolver to access files referenced in the plan
     * @return A binary file stream, typically an image file format.
     *         Callers must ensure that the InputStream is closed after usage.
     *         If the plan cannot be rendered, <code>null</code> will be returned.
     */
    InputStream render(Plan plan, ResourceResolver resolver);
}
