/*
 * Copyright 1997-2009 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.dam.api;

import java.io.InputStream;
import java.util.Calendar;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.jcr.Binary;
import javax.jcr.RepositoryException;

import org.apache.jackrabbit.api.security.user.User;
import org.apache.sling.api.adapter.Adaptable;
import org.apache.sling.api.resource.Resource;

import com.adobe.granite.asset.api.RenditionHandler;

import aQute.bnd.annotation.ProviderType;


/**
 * The <code>Asset</code> interface describes the <code>digital asset</code> with its methods.
 */

@ProviderType
public interface Asset extends Adaptable {
    /**
     * Returns the repository path of the asset.
     *
     * @return path of the asset.
     */
    String getPath();

    /**
     * Returns the name of the asset.
     *
     * @return page name
     */
    String getName();

    /**
     * Returns the metadata <i>String</i> value of an asset by the given property name.
     * Note: Falls back to parse xmp in asset if property not found in JCR.
     * Use of {@linkplain #getMetadataValueFromJcr} is recommended over this method
     * <p>
     * Retrieving a metadata value works as follows:
     * <blockquote><pre>
     *      ...
     *      Asset asset = resource.adaptTo(Asset.class);
     *      String assetTitle = asset.getMetadataValue("dc:title");
     *      ...
     * </pre></blockquote>
     *
     * @param name metadata property name
     * @return the value or an empty string if the metadata property does not exist.
     */
    String getMetadataValue(String name);

    /**
     * Returns the generic metadata value of an asset by the given property name. Falls back to parse xmp in asset if property not found in JCR.
     * <p>
     * As asset metadata values are JCR repository based, a value may be of any type supported by the repository. As
     * such for example a <i>Calendar</i> type property may be retrieved:
     * <blockquote><pre>
     *      ...
     *      Asset asset = resource.adaptTo(Asset.class);
     *      Calendar modified = (Calendar) asset.getMetadata(DamConstants.PN_MODIFIED);
     *      ...
     * </pre></blockquote>
     *
     * @param name metadata property name
     * @return the <code>Object</code> or <code>null</code> if the property does not exist.
     */
    Object getMetadata(String name);

    /**
     * Returns the date of last modification of the asset's main content node (<i>jcr:lastModified</i> property).
     *
     * @return lastmodified date as long
     */
    long getLastModified();

    /**
     * This method returns the {@link Rendition} specified by its name. Available renditions are configured as part of
     * the DAM Asset Update Workflow (see: http://dev.day.com/docs/en/cq/current/dam/customizing_and_extendingcq5dam.html#Customizing%20Renditions).
     * <p>
     * Renditions are stored under the asset's main content node in the "renditions" folder, e.g.
     * <i>/content/dam/geometrixx/banners/banner-mono.png/jcr:content/renditions</i>. The node name of a rendition in
     * this folder is the name to be used as the method argument. For example the rendition path
     * <i>/content/dam/geometrixx/banners/banner-mono.png/jcr:content/renditions/cq5dam.thumbnail.140.100.png</i>
     * results in the rendition name "cq5dam.thumbnail.140.100.png". Sample:
     * <blockquote><pre>
     *      ...
     *      Asset asset = resource.adaptTo(Asset.class);
     *      Resource rendition = asset.getRendition("cq5dam.thumbnail.140.100.png");
     *      ...
     * </pre></blockquote>
     *
     * @param name rendition name
     * @return the rendition or <code>null</code>
     * @see #getRenditions()
     * @see #getRendition(RenditionPicker)
     */
    Rendition getRendition(String name);

    /**
     * This method returns a suitable {@link Rendition} for rendering image rendition derivatives (thumbnails, previews, ptiff).
     * If a suitable rendition can't be found, the method will return the original rendition as a fallback.
     *
     * @return Preview rendition, original rendition, or <code>null</code> if the original rendition doesn't exist.
     */
    Rendition getImagePreviewRendition();

    /**
     * Returns the asset's original as a {@link Rendition}. The original represents the binary file that was initially
     * uploaded as the asset. It is the unmodified version of the asset. The original also represents a rendition of the
     * asset, as such its storage path is in the asset's rendition folder, e.g. <i>/content/dam/geometrixx/banners/banner-mono.png/jcr:content/renditions/original</i>.
     * The original, as any rendition, is persisted as a node of type <i>nt:file</i> in the repository. Subsequently
     * access to the original's file data works through accessing its <i>jcr:content</i> subnode via resource API, or
     * using the {@link com.day.cq.wcm.foundation.Download} or {@link com.day.cq.wcm.foundation.Image} class. The
     * Download and Image convenience classes automatically retrieve the rendition with the name <i>original</i> from
     * the given asset resource.
     * <p>
     * Using the resource API:
     * <blockquote><pre>
     *      ...
     *      Asset asset = resource.adaptTo(Asset.class);
     *      Resource original = asset.getOriginal();
     *      InputStream stream = original.adaptTo(InputStream.class);
     *      ...
     * </pre></blockquote>
     * <p>
     * Using the asset as a download via the <code>Download</code> class:
     * <blockquote><pre>
     *      ...
     *      Asset asset = resource.adaptTo(Asset.class);
     *      Download download = new Download(resource);
     *      String title = download.getTitle();
     *      String mimeType = download.getMimeType();
     *      javax.jcr.Property p = download.getData();
     *      javax.jcr.Binary b = p.getBinary();
     *      long size = b.getSize();
     *      InputStream stream = b.getStream();
     *      ...
     * </pre></blockquote>
     * <p>
     * Using the asset's original as an image via the <code>Image</code> class:
     * <blockquote><pre>
     *      ...
     *      Asset asset = resource.adaptTo(Asset.class);
     *      Image image = new Image(resource);
     *      Layer layer = image.getLayer(true, true, true);
     *      ...
     * </pre></blockquote>
     *
     * @return file resource of the original file, or <code>null</code> if the original resource doesn't exist.
     */
    Rendition getOriginal();

    /**
     * Returns the {@link Rendition} representing the asset's rendition currently marked as the asset's unmodified,
     * original version. A rendition is marked as the original of an asset via the existence of the property
     * <i>currentOriginal</i> on the asset's content node. The property contains the path of the rendition to be
     * considered the original. E.g. <i>/content/dam/geometrixx/banners/banner-mono.png/jcr:content/currentOriginal</i>
     * with value <i>/content/dam/geometrixx/banners/banner-mono.png/jcr:content/renditions/myOriginal</i>.
     *
     * @return current original rendition
     * @see #getOriginal()
     * @deprecated Will be removed without replacement. The concept of current originals will henceforth be managed
     *             through versioning the asset upon modifying the original. Use {@link #getOriginal()} instead.
     */
    @Deprecated
    //todo: remove for CQ 5.5
    Rendition getCurrentOriginal();

    /**
     * Indicates if this asset is a <code>Sub Asset</code>. Sub assets are assets stored under another asset, as a
     * result of splitting up the asset in fragments during its import, e.g. the single pages of a multi-page PDF file.
     *
     * @return <code>true</code> if it is a <code>Sub Asset</code>
     */
    boolean isSubAsset();

    /**
     * Returns all available metadata as map.
     *
     * @return metadata
     * @see #getMetadataValue(String)
     */
    Map<String, Object> getMetadata();

    /**
     * Saves a new rendition and returns the newly saved file {@link org.apache.sling.api.resource.Resource}. The
     * modification date of the asset will be set automatically.
     *
     * @param name     rendition name
     * @param is       <i>InputStream</i> of rendition
     * @param mimeType The mime type of the rendition
     * @return rendition as resource
     * @see #getRendition(String)
     * @see #getRenditions()
     * @since 5.3
     * @deprecated Use {@link #addRendition(String, java.io.InputStream, String)} instead.
     */
    @Deprecated
    //todo: remove for CQ 5.5
    Resource setRendition(String name, InputStream is, String mimeType);

    /**
     * Defines which rendition is marked as the asset's <code>original</code> version.
     *
     * @param name name of rendition
     * @see #getCurrentOriginal()
     * @deprecated Will be removed without replacement.
     */
    @Deprecated
    //todo: remove for CQ 5.5
    void setCurrentOriginal(String name);

    /**
     * Create a new {@link Revision} of the asset. The revision will be created as a standard JCR version of the
     * underlying asset node. Owner of the session through which {@link Asset} was created is added as
     * {@link DamConstants.PN_VERSION_CREATOR} of the asset.
     *
     * @param label   version label. Note that the version label must be unique across all versions.
     * @param comment version comment
     * @return The created revision
     * @throws Exception Thrown when an error during version creation occurred.
     */
    Revision createRevision(String label, String comment) throws Exception;

    /**
     * Create a new {@link Revision} of the asset. The revision will be created as a standard JCR version of the
     * underlying asset node. This API allows specifying principal of an AEM User to be added a
     * {@link DamConstants.PN_VERSION_CREATOR} of the asset being versioned.
     *
     * @param label   version label. Note that the version label must be unique across all versions.
     * @param comment version comment
     * @param versionCreator version creator. If null, this API is equivalent to
     *                  {@link createRevision(String label, String comment)}
     * @return The created revision
     * @throws Exception Thrown when an error during version creation occurred.
     */
    Revision createRevision(String label, String comment, User versionCreator) throws Exception;

    /**
     * Returns a list of all {@link Rendition}s of this asset.
     *
     * @return The renditions.
     */
    List<Rendition> getRenditions();

    /**
     * Returns an iterator for all {@link Rendition}s of this asset.
     *
     * @return The renditions.
     * @since 5.4
     */
    Iterator<Rendition> listRenditions();

    /**
     * Returns the {@link Rendition}, based on a custom {@link RenditionPicker} implementation.
     * <code>RenditionPicker</code> implementations are free to choose which rendition is to be returned.
     *
     * @param picker The <code>RenditionPicker</code> implementation to use for selecting an asset's rendition.
     * @return the desired rendition or <code>null</code> if not existing.
     */
    Rendition getRendition(RenditionPicker picker);

    /**
     * Returns the name of the modifier who modified the asset last
     *
     * @return modifier userId
     */
    String getModifier();

    /**
     * Restores a revision of this asset as identified by the given <code>revisionId</code>. If the revision ID does not
     * pertain to the current asset or does not exist, <code>null</code> is returned, otherwise the restored asset.
     *
     * @param revisionId The id of the revision to restore.
     * @return The restored asset or <code>null</code> if the revision does not exist or doesn't pertain to the asset.
     * @throws Exception If an error is encountered during restoring the asset.
     * @since CQ 5.4.0
     */
    Asset restore(String revisionId) throws Exception;

    /**
     * Returns the revisions available for the current asset. The revisions can be filtered by date. If <code>cal</code>
     * is given, only revisions older or equal to the specified date are returned.
     *
     * @param cal The calendar object to filter the revisions by.
     * @return The collection of revisions for this asset, or an empty collection if none were found.
     * @throws Exception If an error occurred upon retrieving the revisions.
     * @since CQ 5.4.0
     */
    Collection<Revision> getRevisions(Calendar cal) throws Exception;

    /**
     * Returns the mime type of the asset's original binary. If a <i>current original</i> is marked, its mime type will
     * be returned, otherwise the mime type of the asset's original rendition (<i>./jcr:content/renditions/original</i>).
     * If no original could be found, or no mime type is set, <code>null</code> is returned.
     *
     * @return The mime type of the asset's original binary.
     * @since CQ 5.4.0
     */
    String getMimeType();

    /**
     * Adds/replaces a {@link Rendition}. The modification date of the asset will be set automatically. If a rendition
     * with the given <code>name</code> already exists, it will be replaced.
     *
     * @param name     rendition name
     * @param is       <i>InputStream</i> of rendition
     * @param mimeType The mime type of the rendition
     * @return rendition as resource
     * @see #getRendition(String)
     * @see #getRenditions()
     * @since CQ 5.4.0
     */
    Rendition addRendition(String name, InputStream is, String mimeType);

    /**
     * Adds/replaces a {@link Rendition}.
     * The modification date of the asset will be set automatically. If a rendition
     * with the given <code>name</code> already exists, it will be replaced.
     * Actual creation of rendition is delegated to the {@link RenditionHandler}.
     *
     * @param name     rendition name
     * @param is InputStream to create this rendition or null if stream is not available
     * @param map to be used by the rendition handler, it must contain {@link RenditionHandler#PROPERTY_ID}
     * and optionally more properties as defined/required by the respective {@link RenditionHandler}
     * @return rendition as resource
     * @see #getRendition(String)
     * @see #getRenditions()
     * @see RenditionHandler
     * @see RenditionHandler#PROPERTY_ID
     * @see com.adobe.granite.asset.api.Asset#setRendition(String, InputStream, Map)
     * @since CQ 5.6.0
     */
    Rendition addRendition(String name, InputStream is, Map<String, Object> map);

    /**
     * Adds/replaces a {@link Rendition} with given binary for supporting oak direct binary. The modification date of
     * the asset will be set automatically. If a rendition with the given <code>name</code> already exists, it will be
     * replaced.
     *
     * @param name     rendition name
     * @param binary   binary of rendition
     * @param mimeType The mime type of the rendition
     * @return rendition as resource
     * @since CQ 6.5.0
     */
    Rendition addRendition(String name, Binary binary, String mimeType);

    /**
     * Adds/replaces a {@link Rendition} with given binary for supporting oak direct binary. The modification date of
     * the asset will be set automatically. If a rendition with the given <code>name</code> already exists, it will be
     * replaced. It will not delegate the actual creation of rendition to the {@link RenditionHandler} like {@link
     * Asset#addRendition(String, InputStream, Map)}
     *
     * @param name   rendition name
     * @param binary binary to create this rendition or null if binary is not available
     * @param map    it contain properties for the renditions
     * @return rendition as resource
     * @since CQ 6.5.0
     */
    Rendition addRendition(String name, Binary binary, Map<String, Object> map);

    /**
     * Adds/replaces a {@link Rendition} with given binary for supporting oak direct binary. The modification date of the asset will be set
     * automatically. If a rendition with the given <code>name</code> already exists, it will be replaced. The difference with addRendition
     * is this method will not touch the asset state for original rendition
     *
     * @param name     rendition name
     * @param binary   binary of rendition
     * @param mimeType The mime type of the rendition
     *
     * @return rendition as resource, return null if for any reason failed to create replace rendition
     *
     * @since CQ 6.6.0
     */
    Rendition setRendition(String name, Binary binary, String mimeType);

    /**
     * Adds/replaces a sub-asset to this asset. If a sub-asset already exists, it will be replaced.
     *
     * @param name     The name of the sub-asset.
     * @param mimeType The mime type of the sub-asset's original binary.
     * @param stream   The input stream of the sub-asset's original binary.
     * @return The newly added sub-asset.
     * @since CQ 5.4.0
     */
    Asset addSubAsset(String name, String mimeType, InputStream stream);

    /**
     * Returns all sub-assets that exist for this asset.
     *
     * @return The asset's collection of sub-assets.
     * @since CQ 5.4.0
     */
    Collection<Asset> getSubAssets();

    /**
     * Removes the rendition identified by <code>name</code>
     *
     * @param name The name of the rendition.
     * @since CQ 5.4.0
     */
    void removeRendition(String name);

    /**
     * Mark this asset as being part of a batch-mode process. Setting <code>mode</code> to <code>true</code> causes all
     * changes made to the asset (e.g. via {@link #addRendition(String, java.io.InputStream, String)} not to be saved.
     * It is then the responsibility of the caller to call a session save. This mode is particularly useful for batch
     * changes to many assets at once with only one save call at the end so as to speed up operations. The only method
     * exempt from batch mode is {@link #createRevision(String, String)} (will always cause save).
     *
     * @param mode Set to <code>true</code> to enable batch mode for this asset.
     */
    void setBatchMode(boolean mode);

    /**
     * Indicates whether batch mode is enabled or not. If batch mode is enabled, any changes to the asset must be saved
     * externally through a session save.
     *
     * @return Whether batch mode is enabled or not. Default is <code>false</code>.
     */
    boolean isBatchMode();

    /**
     * This method is different from {@linkplain #getMetadataValue} as it doesn't fallback to parse xmp in asset if property not found in JCR.
     * Returns the metadata <i>String</i> value of an asset by the given property name. Available property names are defined by the asset
     * editor configuration used for a particular asset (see
     * http://dev.day.com/docs/en/cq/current/dam/customizing_and_extendingcq5dam.html).
     * <p>
     * Retrieving a metadata value works as follows: <blockquote>
     *
     * <pre>
     *      ...
     *      Asset asset = resource.adaptTo(Asset.class);
     *      String assetTitle = asset.getMetadataValue("dc:title");
     *      ...
     * </pre>
     *
     * </blockquote>
     * @param name metadata property name
     * @return the value or an empty string if the metadata property does not exist.
     */
    String getMetadataValueFromJcr(String name);

    /**
     * This method returns the ID associated with an Asset
     * Returns the non-null <i>String</i> value of Asset's ID
     * <i>May return an empty (non-null) string</i>
     *
     * @return the ID of an asset.
     */

    String getID();

    /**
     * This method initializes <code>DamConstants.DAM_ASSET_STATE</code> property to
     * <code>DamConstants.DAM_ASSET_STATE_UNPROCESSED</code> on the current asset.
     *
     * @throws RepositoryException Thrown in case of error in initialization.
     */

    public void initAssetState() throws RepositoryException;

}
