/*
 * File: Type2HintedRedirector.java
 * 
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is
 *      obtained from Adobe Systems Incorporated.
 *
 */

package com.adobe.fontengine.font.cff;

import java.util.ArrayList;
import java.util.ListIterator;

import com.adobe.fontengine.font.HintedOutlineConsumer;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.InvalidGlyphException;
import com.adobe.fontengine.font.OutlineConsumer;

/**
 * This class receives hinted type2 charstring commands and forwards them
 * to a HintedOutlineConsumer.
 */
final class Type2HintedRedirector extends Type2UnhintedRedirector{
  private ArrayList a = new ArrayList();
  private int lastHintRedirected = -1;
  private int seacHintBase = 0;

  private boolean hintmaskSeen = false;
  private boolean newHints = false;

  private static class Stem {
    final double e1;
    final double e2;
    final boolean isVertical;

    Stem(double e1, double e2, boolean isVertical) {
      this.e1 = e1;
      this.e2 = e2;
      this.isVertical = isVertical;
    }
  }

  /**
   * Prepare 'this' to receive a new charstring, which will be directed to
   * 'outlineConsumer'. 'outlineConsumer' must be an instance of a 
   * {@link com.adobe.fontengine.font.HintedOutlineConsumer}.
   */
  void reset(OutlineConsumer outlineConsumer) {
    super.reset(outlineConsumer);
    a.clear();
    hintmaskSeen = false;
    newHints = false;
    lastHintRedirected = -1;
  }

  public void cntrmask (double[] stack, int stackDepth, CFFByteArray data,
      int offset, int length) throws InvalidFontException {
    processMask(data, offset, length, true);
  }

  public void endchar (double stack[], int stackDepth) {
    outlineConsumer.endchar();
  }

  private void addMask ()  {
    if (!hintmaskSeen || newHints) {
      // if we have seen hints before and now we are seeing them again, 
      // we must need hintsubstitution because we must be seeing a seac
      // accent (in all other type2 hintsubstitution cases, we will see a mask)
      boolean startNewGroup = hintmaskSeen;
      int i = lastHintRedirected+1;
      ListIterator iter = a.listIterator(i);
      while (iter.hasNext()) {
        Stem s = (Stem)iter.next();
        ((HintedOutlineConsumer)outlineConsumer).stem(s.e1, s.e2, startNewGroup, s.isVertical, false);
        lastHintRedirected++;
        startNewGroup = false; }
      hintmaskSeen = true;
      newHints = false; }
  }

  public void rmoveto (double stack[], int stackDepth) {
    addMask();
    super.rmoveto(stack, stackDepth);
  }

  public void hmoveto (double stack[], int stackDepth) {
    addMask();
    super.hmoveto(stack, stackDepth);
  }

  public void vmoveto (double stack[], int stackDepth) {
    addMask();
    super.vmoveto(stack, stackDepth);
  }

  public void moveto(double stack[], int stackDepth) {
    addMask();
    super.moveto(stack, stackDepth);
  }

  public void flex (double[] stack, int stackDepth) {
    double x1, y1, x2, y2, x3, y3, x4, y4, x5, y5;    
    int i = 0;

    x += stack [i++];
    y += stack [i++];
    x1 = x;
    y1 = y;
    x += stack [i++];
    y += stack [i++];
    x2 = x;
    y2 = y;
    x += stack [i++];
    y += stack [i++];
    x3 = x;
    y3 = y;
    x += stack [i++];
    y += stack [i++];
    x4 = x;
    y4 = y;
    x += stack [i++];
    y += stack [i++];
    x5 = x;
    y5 = y;
    x += stack [i++];
    y += stack [i++]; 

    ((HintedOutlineConsumer)outlineConsumer).flex(stack[i], 
        x1, y1, x2, y2, x3, y3, 
        x4, y4, x5, y5, x, y);
  }

  public void flex1 (double[] stack, int stackDepth) {
    double x1, y1, x2, y2, x3, y3, x4, y4, x5, y5;    
    int i = 0;

    double startx = x;
    double starty = y;
    x += stack [i++];
    y += stack [i++];
    x1 = x;
    y1 = y;
    x += stack [i++];
    y += stack [i++];
    x2 = x;
    y2 = y;
    x += stack [i++];
    y += stack [i++];  
    x3 = x;
    y3 = y;
    x += stack [i++];
    y += stack [i++];
    x4 = x;
    y4 = y;
    x += stack [i++];
    y += stack [i++];
    x5 = x;
    y5 = y;

    if (Math.abs (x - startx) > Math.abs (y - starty))   {
      x += stack [i++]; 
      y = starty; }
    else {
      y += stack [i++]; 
      x = startx; }     

    ((HintedOutlineConsumer)outlineConsumer).flex(50, 
        x1, y1, x2, y2, x3, y3, 
        x4, y4, x5, y5, x, y);
  }


  public void hflex (double[] stack, int stackDepth) {
    double x1, x2, x3, x4, x5, starty;    
    int i = 0;

    starty = y;
    x += stack [i++];
    x1 = x;
    x += stack [i++];
    y += stack [i++];
    x2 = x;
    x += stack [i++];
    x3 = x;

    x += stack [i++];
    x4 = x;
    x += stack [i++];
    x5 = x;
    x += stack [i++];     
    ((HintedOutlineConsumer)outlineConsumer).flex(50, 
        x1, starty, x2, y, x3, y, 
        x4, y, x5, starty, x, starty);

    y = starty;
  }

  public void hflex1 (double[] stack, int stackDepth) {
    double x1, y1, x2, y2, x3, x4, x5, y5, startY;    
    int i = 0;

    startY = y;
    x += stack [i++];
    y += stack [i++];
    x1 = x;
    y1 = y;
    x += stack [i++];
    y += stack [i++];
    x2 = x;
    y2 = y;
    x += stack [i++];  
    x3 = x;

    x += stack [i++];
    x4 = x;
    x += stack [i++];
    y += stack [i++];
    x5 = x;
    y5 = y;
    x += stack [i++];
    y = startY;    
    ((HintedOutlineConsumer)outlineConsumer).flex(50, 
        x1, y1, x2, y2, x3, y2, 
        x4, y2, x5, y5, x, y);
  }

  private void processMask(CFFByteArray data, int offset, int length, boolean isCounter)
  throws InvalidFontException {
    int i;
    boolean byteOn = false;
    boolean startNewGroup = isCounter || hintmaskSeen;

    for (i = 0; i < length; i++) {
      if (data.getcard8(offset+i) != 0) {
        byteOn = true; }}

    if (byteOn)  {
      for (i = 0; i < length; i++) {
        int value = data.getcard8(offset+i);
        if (value != 0) {
          for (int j = 7; j >= 0; j--) {
            if ((value & (1<<j)) != 0) {
              int stemIndex = (i * 8) + (7 - j) + seacHintBase;
              if (stemIndex >= a.size())
                throw new InvalidGlyphException("Invalid hint mask");
              Stem s = (Stem)a.get(stemIndex);
              ((HintedOutlineConsumer)outlineConsumer).stem(s.e1, s.e2,startNewGroup, s.isVertical, isCounter );
              startNewGroup = false; }}}}} 
    else if (isCounter) {
      ((HintedOutlineConsumer)outlineConsumer).noCounters(); }
    else {
      ((HintedOutlineConsumer)outlineConsumer).noHints(); }
  }

  public void hintmask (double stack[], int stackDepth, 
      CFFByteArray data, int offset, int length) 
  throws InvalidFontException {

    processMask(data, offset, length, false);
    hintmaskSeen = true;
    newHints = false;
  }

  private void addStem (double[] stack, int stackDepth, boolean isVertical) {
    double e1 = 0, e2;

    if (!newHints) {
      seacHintBase = a.size();
      lastHintRedirected = seacHintBase - 1;
      newHints = true;
    }
    int i = (stackDepth % 2 != 0) ? 1: 0;
    while ( i < stackDepth ) {
      e1 += stack[i++];
      e2 = e1 + stack[i++];
      a.add(new Stem(e1, e2, isVertical));
      e1 = e2; }     
  }

  public void hstem (double[] stack, int stackDepth) {
    addStem(stack, stackDepth, false);
  }

  public void hstemhm (double[] stack, int stackDepth) {
    hstem(stack, stackDepth);
  }

  public void implicit_vstemhm (double[] stack, int stackDepth) {
    vstem(stack, stackDepth);
  }

  public void vstem (double[] stack, int stackDepth) {
    addStem(stack, stackDepth, true);
  }

  public void vstemhm (double[] stack, int stackDepth) {
    vstem(stack, stackDepth);
  }

  public boolean width (double w) {
    return ((HintedOutlineConsumer)outlineConsumer).width(w);
  }

  public void globalColorMe(double[] stack, int stackDepth)  {
    ((HintedOutlineConsumer)outlineConsumer).globalColorOn();
  }
}
