/*************************************************************************
* ADOBE CONFIDENTIAL
* ___________________
*
* Copyright 2013 Adobe
* All Rights Reserved.
*
* NOTICE: All information contained herein is, and remains
* the property of Adobe and its suppliers, if any. The intellectual
* and technical concepts contained herein are proprietary to Adobe
* and its suppliers and are protected by all applicable intellectual
* property laws, including trade secret and copyright laws.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe.
**************************************************************************/
package com.adobe.granite.ui.components.ds;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;

import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;

import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceMetadata;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.SyntheticResource;
import org.apache.sling.api.resource.ValueMap;
import org.apache.sling.api.wrappers.ValueMapDecorator;

/**
 * A {@link SyntheticResource} that will return the passed ValueMap when this
 * resource is adapted to ValueMap.
 */
public class ValueMapResource extends SyntheticResource {
    @Nonnull
    private ValueMap vm;

    private Collection<Resource> children;

    /**
     * Instantiates with an empty ValueMap initially.
     *
     * @param resourceResolver
     *            The resource resolver
     * @param path
     *            The resource path
     * @param resourceType
     *            The resource type
     */
    public ValueMapResource(ResourceResolver resourceResolver, String path, String resourceType) {
        this(resourceResolver, path, resourceType, new ValueMapDecorator(new HashMap<>()));
    }

    /**
     * Instantiates with the given ValueMap.
     *
     * @param resourceResolver
     *            The resource resolver
     * @param path
     *            The resource path
     * @param resourceType
     *            The resource type
     * @param vm
     *            The ValueMap of the resource
     */
    public ValueMapResource(ResourceResolver resourceResolver, String path, String resourceType, @Nonnull ValueMap vm) {
        this(resourceResolver, path, resourceType, vm, null);
    }

    /**
     * Instantiates with the given child resources.
     *
     * @param resourceResolver
     *            The resource resolver
     * @param path
     *            The resource path
     * @param resourceType
     *            The resource type
     * @param vm
     *            The ValueMap of the resource
     * @param children
     *            The children of the resource
     */
    public ValueMapResource(ResourceResolver resourceResolver, String path, String resourceType, @Nonnull ValueMap vm,
            @CheckForNull Collection<Resource> children) {
        super(resourceResolver, path, resourceType);
        this.vm = vm;
        this.children = children;
    }

    /**
     * Instantiates with an empty ValueMap initially.
     *
     * @param resourceResolver
     *            The resource resolver
     * @param rm
     *            The resource meta data
     * @param resourceType
     *            The resource type
     */
    public ValueMapResource(ResourceResolver resourceResolver, ResourceMetadata rm, String resourceType) {
        this(resourceResolver, rm, resourceType, new ValueMapDecorator(new HashMap<>()));
    }

    /**
     * Instantiates with the given ValueMap.
     *
     * @param resourceResolver
     *            The resource resolver
     * @param rm
     *            The resource meta data
     * @param resourceType
     *            The resource type
     * @param vm
     *            The ValueMap of the resource
     */
    public ValueMapResource(ResourceResolver resourceResolver, ResourceMetadata rm, String resourceType,
            @Nonnull ValueMap vm) {
        this(resourceResolver, rm, resourceType, vm, null);
    }

    /**
     * Instantiates with the given child resources.
     *
     * @param resourceResolver
     *            The resource resolver
     * @param rm
     *            The resource meta data
     * @param resourceType
     *            The resource type
     * @param vm
     *            The ValueMap of the resource
     * @param children
     *            The children of the resource
     */
    public ValueMapResource(ResourceResolver resourceResolver, ResourceMetadata rm, String resourceType,
            @Nonnull ValueMap vm, @CheckForNull Collection<Resource> children) {
        super(resourceResolver, rm, resourceType);
        this.vm = vm;
        this.children = children;
    }

    @SuppressWarnings("unchecked")
    @Override
    public <Type> Type adaptTo(@SuppressWarnings("null") Class<Type> type) {
        if (ValueMap.class.equals(type)) {
            return (Type) vm;
        } else {
            return super.adaptTo(type);
        }
    }

    @Override
    public Resource getChild(String relPath) {
        if (children == null) {
            return super.getChild(relPath);
        }
        for (Resource child : children) {
            if (child.getPath().equals(getPath() + "/" + relPath)) {
                return child;
            }
        }
        return null;
    }

    @Override
    public Iterator<Resource> listChildren() {
        if (children == null) {
            return super.listChildren();
        }
        return children.iterator();
    }

    @Override
    public Iterable<Resource> getChildren() {
        // This method cannot be removed due to baseline, even though it is identical to
        // the super class
        return super.getChildren();
    }

    @Override
    public boolean hasChildren() {
        if (children == null) {
            return super.hasChildren();
        }
        return !children.isEmpty();
    }
}
