/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */

package com.adobe.xfa;

import java.util.ArrayList;
import java.util.List;

/**
 * LogMessageHandler base class. All message handlers
 * will derive from this class. 
 * Class LogMessageHandler will eventually will support message filtering by Id.
 *
 * @exclude from published api -- Mike Tardif, May 2006.
 */

public abstract class LogMessageHandler {
	//int mnProcessId;

	private final long mlThreadId;

	private int mnThreshold;

	private final List<LogMessage> moMessages = new ArrayList<LogMessage>(); // cache all messages

	private String msAppContext;

	//private LogMessage msEmptyMessage;
	
	
	public LogMessageHandler() {
		mlThreadId = java.lang.Thread.currentThread().getId();
	}

	/**
	 * Retrieve a message based on its index.
	 * where 0 <= index < size()
	 * <br><br> - only valid for handlers that return true for the
	 * supportsRead method.
	 *
	 * @param nIndex - index into the collection
	 * @return a LogMessageobject
	 */
	public LogMessage get(int nIndex) {
		return moMessages.get(nIndex);
	}
	
	/**
	 * Clears out message buffer(s)
	 */
	public void clearMessages() {
		// do nothing
	}

	protected String getAppContext() {
		return msAppContext;
	}

	protected List<LogMessage> getMessages() {
		return moMessages;
	}

//	/** 
//	 * Retrieve the messagehandler's process id
//	 * @return a pid
//	 */
//	public int getProcessId() {
//		throw new ExFull(ResId.UNSUPPORTED_OPERATION, "LogMessageHandler#getProcessID");
//	}

	/** 
	 * Retrieve the messagehandler's thread id
	 * @return a threadId
	 */
	public long getThreadId() {
		return mlThreadId;
	}

	public int getThreshold() {
		return mnThreshold;
	}

	/**
	 * Pure virtual function. Implemented in derived classes to handle message
	 * output.
	 * <br><br> if you want to cache messages, then the derived implementation
	 * of this method should add the following lines:
	 * <code>
	 * LogMessage poMessage = new LogMessage(oMessage);
	 * getMessages().append(poMessage);
	 * </code>
	 *
	 * @param oMessage - the message to send
	 */
	abstract public void sendMessage(LogMessage oMessage);
	
	protected void setAppContext(String sAppContext) {
		msAppContext = sAppContext;
	}

	public void setThreshold(int nThreshold) {
		mnThreshold = nThreshold;
	}

	abstract public void flush();

	/**
	 * Retrieve the number of LogMessageobjects are in the collection.
	 * <br><br> only valid for handlers that return true for the
	 * supportsRead method.
	 * @return - the number of LogMessageobjects in the collection
	 */
	public int size() {
		return moMessages.size();
	}

	/**
	 * Check whether the handler supports read operations.
	 * If this class' <code>supportsRead()</code> method returns true, then
	 * the messageHandler is capable of caching all of it's messages.  These
	 * messages can be accessed via  <code>operator[]</code>.  The
	 * <code>size()</code> method can be used to find out the number of
	 * messages cached.
	 */
	public boolean supportsRead() {
		return false;
	}
}