/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.wcm.mobile.core;

import com.day.cq.wcm.api.Page;
import com.day.cq.wcm.api.PageManager;
import com.day.cq.wcm.mobile.api.MobileConstants;
import com.day.cq.wcm.mobile.api.device.DeviceGroup;
import com.day.cq.wcm.mobile.api.device.DeviceGroupList;
import com.day.cq.wcm.mobile.api.device.capability.DeviceCapability;
import org.apache.commons.lang.StringUtils;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceUtil;
import org.apache.sling.api.resource.ValueMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.day.cq.wcm.mobile.api.MobileConstants.MOBILE_BASE_RESOURCE_TYPE;

/**
 * The <code>MobileUtil</code> provides various utility methods revolving around WCM Mobile.
 */
public class MobileUtil {

    private static final Logger log = LoggerFactory.getLogger(MobileUtil.class);

    /**
     * Checks whether the given {@link Resource} represents a <i>WCM Mobile Device Group</i>. Resource that match the
     * resource type {@link com.day.cq.wcm.mobile.api.device.DeviceGroup#RESOURCE_TYPE} are considered device groups.
     *
     * @param resource The resource to check.
     *
     * @return <code>true</code> if the resource is in fact a device group.
     *
     * @see com.day.cq.wcm.mobile.api.device.DeviceGroup
     */
    public static boolean isDeviceGroup(final Resource resource) {
        return null != resource && ResourceUtil.isA(resource, DeviceGroup.RESOURCE_TYPE);
    }

    /**
     * Checks whether the given {@link Page} represents a <i>WCM Mobile Device Group</i>. The check is performed on the
     * page's content resource.
     *
     * @param page The page to check.
     *
     * @return <code>true</code> if the page is in fact a device group.
     *
     * @see #isDeviceGroup(org.apache.sling.api.resource.Resource)
     * @see com.day.cq.wcm.mobile.api.device.DeviceGroup
     */
    public static boolean isDeviceGroup(final Page page) {
        return null != page && isDeviceGroup(page.getContentResource());
    }

    /**
     * This method finds the device groups currently assigned to the given page or its closest ancestor. From the
     * assigned device groups, the first group is returned.
     *
     * @param page The page to check for device group mappings.
     *
     * @return The first {@link DeviceGroup}.
     */
    public static DeviceGroup getDefaultDeviceGroup(final Page page) {
        final DeviceGroupList mappedDeviceGroups = page.adaptTo(DeviceGroupList.class);
        if (null != mappedDeviceGroups && !mappedDeviceGroups.isEmpty()) {
            return mappedDeviceGroups.get(0);
        }
        return null;
    }

    /**
     * Returns the last selector found in the request URL. Within a mobile site context, this selector usually defines
     * the device group name for the active device group.
     *
     * @param request The {@link org.apache.sling.api.SlingHttpServletRequest} to retrieve selectors from.
     *
     * @return A <code>String</code> representing the selector or <code>null</code> if not found.
     */
    public static String getDeviceGroupSelector(final SlingHttpServletRequest request) {
        final String[] selectors = request.getRequestPathInfo().getSelectors();
        return (0 < selectors.length) ? selectors[selectors.length - 1] : null;
    }

    /**
     * This method retrieves the {@link DeviceGroup} from the current request and checks the group whether it offers the
     * given <code>capability</code>.
     *
     * @param request    The current request.
     * @param capability The {@link DeviceCapability} to check for.
     *
     * @return <code>true</code> if the current device group offers the capability.
     */
    public static boolean hasCapability(final SlingHttpServletRequest request, final DeviceCapability capability) {
        final DeviceGroup group = request.adaptTo(DeviceGroup.class);
        return (null != group && group.hasCapability(capability));
    }

    /**
     * True if the request's user agent is a mobile device. The user agent header is matched for "Mobile" and
     * "Blackberry".
     *
     * @param r The request to check.
     *
     * @return <code>true</code> if this is a mobile request.
     */
    public static boolean isMobileRequest(SlingHttpServletRequest r) {
        return StringUtils.contains(r.getHeader("User-Agent"), "Mobile")
               || StringUtils.contains(r.getHeader("User-Agent"), "Blackberry");
    }

    /**
     * True if given resource's parent page (or page itself) or any of its parent pages is based on the mobile resource
     * type {@link MobileConstants#MOBILE_BASE_RESOURCE_TYPE}.
     *
     * @param r The resource to check, may be null.
     *
     * @return <code>true</code> if this resource is a mobile resource.
     */
    public static boolean isMobileResource(Resource r) {
        return isMobileResource(r, null);
    }

    /**
     * True if given resource's parent page (or page itself) or any of its parent pages is based on the mobile resource
     * type {@link MobileConstants#MOBILE_BASE_RESOURCE_TYPE} or any other type provided in the types array.
     *
     * @param r The resource to check, may be null.
     * @param types The additional resource types to check.  May be null.
     *
     * @return <code>true</code> if this resource is a mobile resource.
     */
    public static boolean isMobileResource(Resource r, String[] types) {

        if (r == null) {
            log.debug("Resource is null, isMobileResource() returns false");
            return false;
        }

        boolean isMobileResource = false;

        final PageManager pm = r.getResourceResolver().adaptTo(PageManager.class);
        if (null != pm) {
            Page page = pm.getContainingPage(r);
            if (null != page) {

                while (!isMobileResource && null != page) {
                    final Resource content = page.getContentResource();
                    isMobileResource = null != content && ResourceUtil.isA(content, MOBILE_BASE_RESOURCE_TYPE);
                    if (!isMobileResource && null != content && null != types) {
                        for(String type : types) {
                            if(type != null && type.length() > 0) {
                                isMobileResource = ResourceUtil.isA(content, type);
                            }
                            if (isMobileResource) break;
                        }
                    }
                    if (isMobileResource) {
                        log.debug("isMobileResource: resource [{}] is a mobile resource by associated page [{}]",
                                  r.getPath(), page.getPath());
                    }
                    page = page.getParent();
                }
            } else {
                log.debug("isMobileResource: cannot check resource [{}], not part of a page.", r.getPath());
            }
        } else {
            log.error("isMobileResource: cannot check resource [{}], page manager unavailable.", r.getPath());
        }

        if (!isMobileResource) {
            log.debug("isMobileResource: resource [{}] is not a mobile resource.", r.getPath());
        }

        return isMobileResource;
    }

    /**
     * Checks if the given resource prevents redirection
     * @param r resource
     * @return <code>true</code> if redirection is prevented
     */
    public static boolean isPreventRedirect(Resource r) {
        // don't use page manager (bug #37475)
        
        String path = r.getPath();
        int idx = path.indexOf("/jcr:content");
        if (idx >= 0) {
            path = path.substring(0, idx);
        }
        Resource content = r.getResourceResolver().getResource(path + "/jcr:content");
        if (content != null) {
            ValueMap props = content.adaptTo(ValueMap.class);
            return props.get(MobileConstants.MOBILE_PREVENT_REDIRECT_PROPERTY, false);
        } else {
            return false;
        }
    }
    
    public static boolean isNoMatch(String[] selectors) {
        return selectors != null && selectors.length > 0 && selectors[selectors.length - 1]
                .equals(MobileConstants.NO_MATCH_SELECTOR);
    }
}