/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.cq.gfx;

import aQute.bnd.annotation.ProviderType;

/**
 * An extended map storing key-value instructions for {@link com.adobe.cq.gfx.Plan rendering plans}.
 *
 * <p>
 * Provides type conversion and default value handling through generic getters
 * and a convenience setter method allowing for chained operations.
 * </p>
 * <h2>Type Conversions</h2>
 * When values are added to the instructions, they will be stored as is. Any type conversion is
 * done on reading only, using {@link #get(String, Class)} or {@link #get(String, Object)},
 * depending on what format the consumer requires.
 *
 * <p>
 * When getting an object a consumer could ask for an object &lt;T&gt; or for a {@link String}.
 * The list below shows the availabe conversions between &lt;T&gt; and String, which will
 * work in both ways (parsing string to retrieve object or serializing the object as string).
 * </p>
 *
 * Available type conversions between String and &lt;T&gt;:
 * <ul>
 *     <li>{@link Boolean}: true if string is "1"</li>
 *     <li>{@link Integer}: normal decimal representation "123" using {@link Integer#toString(int)}</li>
 *     <li>{@link Float}: normal floating point representation "3.14" using {@link Float#toString(float)}</li>
 *     <li>{@link java.awt.Dimension Dimension}: represented as "width,height"</li>
 *     <li>{@link java.awt.Rectangle Rectangle}: represented as "x,y,width,height"</li>
 *     <li>other Objects are represented using {@link Object#toString()} when converted to string, but cannot be parsed from a string</li>
 * </ul>
 * 
 */
@ProviderType
public interface Instructions extends java.util.Map<String, Object> {

    /**
     * Get a named property and {@link Instructions convert it} into the given type if necessary.
     * If the property is not present or cannot be converted, return <code>null</code>.
     *
     * @see Instructions class javadoc for available type conversions
     *
     * @param name The name of the property
     * @param type The class of the type
     * @return Return named value converted to type T or <code>null</code> if
     *         non existing or cannot be converted.
     */
    <T> T get(String name, Class<T> type);

    /**
     * Get a named property and {@link Instructions convert it} into the type
     * given by the defaultValue argument, if necessary.
     * If the property is not present or cannot be converted, return the defaultValue.
     *
     * @see Instructions class javadoc for available type conversions
     *
     * @param name The name of the property
     * @param defaultValue The default value to use if the named property does
     *            not exist or cannot be converted to the requested type. The
     *            default value is also used to define the type to convert the
     *            value to. If this is <code>null</code> any existing property is
     *            not converted.
     * @return Return named value converted to type T or the default value if
     *         non existing or cannot be converted.
     */
    <T> T get(String name, T defaultValue);

    /**
     * Sets the value of a named property.
     *
     * <p>
     * This allows chaining by returning the Instructions itself.
     * Note that the object will be stored as is and no conversion will happen.
     *
     * @param name The name of the property
     * @param value The value to set.
     * @return Returns <code>this</code> to allow for method chaining.
     */
    <T> Instructions set(String name, T value);
}
