/*
 * File: OutlineConsumerAdapter.java
 * 
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is
 *      obtained from Adobe Systems Incorporated.
 *
 */

package com.adobe.fontengine.font;

/** Wraps an {@link OutlineConsumer2} as an {@link OutlineConsumer}.
 * 
 * The client of this class should take care to call {@link #startOutline}
 * and {@link #endOutline} appropriately, as those methods are not 
 * invoked through the {@link OutlineConsumer} interface.
 */

public class OutlineConsumerAdapter implements OutlineConsumer {
  final OutlineConsumer2 dest;
  
  protected boolean contourStarted;
  double startX, startY;
  double currentX, currentY;
  
  public OutlineConsumerAdapter (OutlineConsumer2 dest) {
    this.dest = dest;
  }

  public void startOutline () {
    dest.startOutline ();
    contourStarted = false;
  }
  
  protected Matrix toEmMatrix = Matrix.IDENTITY_MATRIX;
  protected Matrix emToDeviceMatrix = Matrix.IDENTITY_MATRIX;
  protected Matrix compositeMatrix = Matrix.IDENTITY_MATRIX;

  public void setMatrix (Matrix m) {
    setMatrices (m, emToDeviceMatrix);
  }

  public void setEmToPixelMatrix (Matrix m) {
    setMatrices (toEmMatrix, m);
  }

  protected void setMatrices (Matrix toEmMatrix, Matrix emToDeviceMatrix) {
    this.toEmMatrix = toEmMatrix;
    this.emToDeviceMatrix = emToDeviceMatrix;

    if (toEmMatrix.isIdentity ()) {
      compositeMatrix = emToDeviceMatrix; }
    else if (emToDeviceMatrix.isIdentity ()) {
      compositeMatrix = toEmMatrix; }
    else {
      compositeMatrix = toEmMatrix.multiply (emToDeviceMatrix); }
  }

  
  public void moveto (double x, double y) {
    if (contourStarted) {
      dest.line (compositeMatrix.applyToXYGetX (currentX, currentY),
                 compositeMatrix.applyToXYGetY (currentX, currentY),
                 compositeMatrix.applyToXYGetX (startX, startY),
                 compositeMatrix.applyToXYGetY (startX, startY));
      dest.endContour (); 
      contourStarted = false; }
    
    currentX = x;
    currentY = y;    
  }
  
  public void lineto (double x2, double y2) {
    if (! contourStarted) {
      dest.startContour ();
      contourStarted = true; 
      startX = currentX;
      startY = currentY; }
    dest.line (compositeMatrix.applyToXYGetX (currentX, currentY),
               compositeMatrix.applyToXYGetY (currentX, currentY),
               compositeMatrix.applyToXYGetX (x2, y2),
               compositeMatrix.applyToXYGetY (x2, y2));
    currentX = x2;
    currentY = y2;
  }

  public void curveto (double x2, double y2, double x3, double y3) {
    if (! contourStarted) {
      dest.startContour ();
      contourStarted = true; 
      startX = currentX;
      startY = currentY; }
    dest.quadraticCurve (compositeMatrix.applyToXYGetX (currentX, currentY),
                         compositeMatrix.applyToXYGetY (currentX, currentY),
                         compositeMatrix.applyToXYGetX (x2, y2),
                         compositeMatrix.applyToXYGetY (x2, y2),
                         compositeMatrix.applyToXYGetX (x3, y3),
                         compositeMatrix.applyToXYGetY (x3, y3));
    currentX = x3;
    currentY = y3;
  }

  public void curveto (double x2, double y2, double x3, double y3, double x4, double y4) {
    if (! contourStarted) {
      dest.startContour ();
      contourStarted = true; 
      startX = currentX;
      startY = currentY; }
    dest.cubicCurve (compositeMatrix.applyToXYGetX (currentX, currentY),
                     compositeMatrix.applyToXYGetY (currentX, currentY),
                     compositeMatrix.applyToXYGetX (x2, y2),
                     compositeMatrix.applyToXYGetY (x2, y2),
                     compositeMatrix.applyToXYGetX (x3, y3),
                     compositeMatrix.applyToXYGetY (x3, y3),
                     compositeMatrix.applyToXYGetX (x4, y4),
                     compositeMatrix.applyToXYGetY (x4, y4));
    currentX = x4;
    currentY = y4;
  }

  public void endOutline () {
    if (contourStarted) {
      if (currentX != startX || currentY != startY) {
        dest.line (compositeMatrix.applyToXYGetX (currentX, currentY),
                   compositeMatrix.applyToXYGetY (currentX, currentY),
                   compositeMatrix.applyToXYGetX (startX, startY),
                   compositeMatrix.applyToXYGetY (startX, startY)); }
      dest.endContour (); }
    dest.endOutline ();
  }

  public void endchar() {}
}
