/*
*
*	File: GsubSubsetter.java
*
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2008 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/
package com.adobe.fontengine.font.opentype;

import java.util.Arrays;
import java.util.Map;

import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.LayoutTable.CoverageConsumer;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;

class GsubSubsetter extends LookupTableSubsetter {
	
	public GsubSubsetter(LookupTable origGsub, int numGlyphs) {
		super(origGsub, OTByteArray.getOTByteArrayBuilderInstance(10), numGlyphs);
	}
	
	void patchSubtableCoverage(int origSTOffset, int newSTOffset, Map coverageInfo, int lookupType) throws InvalidFontException
	{
		switch (lookupType)
		{
		case 1:
		case 2: 
		case 3:
		case 4:
			patchCoverageAtOffset(origTable, builder, origSTOffset, newSTOffset, 2, coverageInfo);
			break;
		case 5:
			ContextualGenerator.patchSubtableCoverage(origTable, builder, origSTOffset, newSTOffset, coverageInfo, lookupType);
			break;

		case 6: 
			ChainingGenerator.patchSubtableCoverage(origTable, builder, origSTOffset, newSTOffset, coverageInfo, lookupType);
			break;

		case 7:
			// nothing to do.
			break;
			
		default:
			throw new InvalidFontException("Unrecognized lookup type: " + lookupType);	
		}
	}

	interface Type1TargetGIDFetcher
	{
		int getTarget(int originalGID, int coverageIndex) throws InvalidFontException;
	}
	
	class Type1Format1TargetGIDFetcher implements Type1TargetGIDFetcher
	{
		private int delta;
		void setDelta(int delta) {
			this.delta = delta;
		}
		
		public int getTarget(int originalGID, int coverageIndex) {
			return originalGID + delta;
		}
	}
	
	class Type1Format2TargetGIDFetcher implements Type1TargetGIDFetcher
	{
		private int origSTOffset;
		void setOrigSTOffset(int theOffset)
		{
			origSTOffset = theOffset;
		}
		
		public int getTarget(int originalGID, int coverageIndex) throws InvalidFontException {
			return origTable.data.getuint16(origSTOffset + 6 + 2*coverageIndex);
		}
	}
	
	/*
	 * In the new subset, can we continue to use Format1 for our Type1 lookup, or do holes in the subset
	 * gids prevent that?
	 */
	class DeltaDeterminer implements CoverageConsumer {
		private final Subset subset;
		private boolean format1Ok = true;
		private int delta;
		private boolean deltaInitialized = false;
		private Type1TargetGIDFetcher fetcher;
		
		public DeltaDeterminer(Subset subset, Type1TargetGIDFetcher fetcher) {
			this.subset = subset;
			this.fetcher = fetcher;
		}
		
		public boolean glyphInfo(int gid, int coverageIndex) throws InvalidFontException, UnsupportedFontException {
			int subsetGid = subset.getExistingSubsetGid(gid);

			int origTarget = fetcher.getTarget(gid, coverageIndex);
			int newTarget = subset.getExistingSubsetGid(origTarget);
			if (deltaInitialized)
			{
				if (newTarget - subsetGid == delta)
					return true;
				else
				{
					format1Ok = false;
					return false;
				}
			} else
			{
				delta = newTarget - subsetGid;
				deltaInitialized = true;
				// delta needs to fit in a signed int16
				if (delta > 32767 || delta < -32768)
				{
					format1Ok = false;
					return false;
				}
			}
			return true;
		}
		
		boolean canUseFormat1()
		{
			return format1Ok;
		}
		
		int delta()
		{
			return delta;
		}
	}
	
	private int writeType1Format1(int origSTOffset, int newSTOffset, int newDelta) throws InvalidFontException
	{
		builder.ensureCapacity(newSTOffset + 6);
		builder.setuint16(newSTOffset, 1);
		builder.setint16(newSTOffset + 4, newDelta);
		
		return 6;
	}
	
	private class Type1Format2Generator implements CoverageConsumer, LookupSubtableGenerator
	{
		private final Subset subset;
		private final OTByteArrayBuilder builder;
		private final int newSTOffset;
		int glyphCount = 0;
		private final Type1TargetGIDFetcher fetcher;
		private int[] subsetToTarget;
		private final int origSTOffset;
		private final Map newCoverages;
		
		Type1Format2Generator(int origSTOffset, Subset subset, OTByteArrayBuilder builder, int newSTOffset, 
				Type1TargetGIDFetcher fetcher, Map newCoverages)
		{
			this.builder = builder;
			this.subset = subset;
			this.newSTOffset = newSTOffset;
			this.origSTOffset = origSTOffset;
			this.fetcher = fetcher;
			this.newCoverages = newCoverages;
			this.subsetToTarget = new int[subset.getNumGlyphs()];
			Arrays.fill(subsetToTarget, -1);
		}
		

		void writeHeader(int glyphCount)
		{
			builder.ensureCapacity(newSTOffset + 6 + 2*glyphCount);
			builder.setuint16(newSTOffset, 2);
			builder.setuint16(newSTOffset + 4, glyphCount);
		}
		
		
		public boolean glyphInfo(int gid, int coverageIndex) throws InvalidFontException, UnsupportedFontException {
							
			int matchedGid = fetcher.getTarget(gid, coverageIndex);
			int subsetGid = subset.getExistingSubsetGid(matchedGid);
			subsetToTarget[subset.getExistingSubsetGid(gid)] = subsetGid;
			if (subsetGid == -1)
				throw new RuntimeException("Matched a gid not in our subset?? Can't happen..");
			
			glyphCount++;
			return true;
		}
		
		public int writeSubtable() throws InvalidFontException, UnsupportedFontException
		{
			int coverageOffset = origTable.data.getOffset(origSTOffset, 2);
			NewCoverage newCoverage = (NewCoverage)newCoverages.get(new Integer(coverageOffset));
			this.writeHeader(newCoverage.glyphCount);
			
			origTable.iterateCoverage(coverageOffset, subset, this);
			
			for (int i = 0, j = 0; i < subsetToTarget.length; i++)
			{
				if (subsetToTarget[i] != -1)
				{
					builder.setuint16(newSTOffset + 6 + 2*j, subsetToTarget[i]);
					j++;
				}
			}
			

			return 6 + 2*newCoverage.glyphCount;
		}
		
	}
	
	private class Type2Or3Generator implements CoverageConsumer, LookupSubtableGenerator
	{
		private final Subset subset;
		private final OTByteArrayBuilder builder;
		private final int newSTOffset;
		private final int origSTOffset;
		private int[] subsetToCoverageIndex;
		private final int glyphCount;
		
		Type2Or3Generator(Subset subset, OTByteArrayBuilder newData, int newSTOffset, int origSTOffset, Map newCoverages) throws InvalidFontException
		{
			int coverageOffset = origTable.data.getOffset(origSTOffset, 2);
			NewCoverage newCoverage = (NewCoverage)newCoverages.get(new Integer(coverageOffset));
			
			this.subset = subset;
			this.builder = newData;
			this.newSTOffset = newSTOffset;
			this.origSTOffset = origSTOffset;
			this.glyphCount = newCoverage.glyphCount;
			this.subsetToCoverageIndex = new int[subset.getNumGlyphs()];
			Arrays.fill(subsetToCoverageIndex, -1);
		}
		

		private void writeHeader(int glyphCount)
		{
			builder.ensureCapacity(newSTOffset + 6 + 2*glyphCount);
			builder.setuint16(newSTOffset, 1);
			builder.setuint16(newSTOffset + 4, glyphCount);
		}

		public boolean glyphInfo(int gid, int coverageIndex) throws InvalidFontException, UnsupportedFontException {
			subsetToCoverageIndex[subset.getExistingSubsetGid(gid)] = coverageIndex;
			return true;
		}
		
		public int writeSubtable() throws InvalidFontException, UnsupportedFontException {

			int subtableSize = 6 + 2*glyphCount;
			
			writeHeader(glyphCount);
			origTable.iterateCoverage(origTable.data.getOffset(origSTOffset, 2), subset, this);
			
			for (int i = 0, j=0; i < subsetToCoverageIndex.length; i++)
			{
				if (subsetToCoverageIndex[i] != -1)
				{
					builder.setuint16(newSTOffset + 6 + 2*j, subtableSize);
					int origSequenceOrSetOffset = origTable.data.getOffset(origSTOffset, 6 + 2*subsetToCoverageIndex[i]);
					int glyphCount = origTable.data.getuint16(origSequenceOrSetOffset);
					builder.ensureCapacity(newSTOffset + subtableSize + 2 + 2*glyphCount);
					builder.setuint16(newSTOffset + subtableSize, glyphCount);
					
					// write out the new glyphids.
					for (int k = 0; k < glyphCount; k++)
					{
						int subsetGid = subset.getExistingSubsetGid(origTable.data.getuint16(origSequenceOrSetOffset + 2 + 2*k));
						if (subsetGid == -1)
							throw new RuntimeException("Matched a gid not in our subset?? Can't happen..");
						builder.setuint16(newSTOffset + subtableSize + 2 + 2*k, subsetGid);
					}
					subtableSize += 2 + 2*glyphCount;
					j++;
				}
			}
			return subtableSize;
		}

		
	}
	
	private static class Type4Generator extends SetGenerator implements CoverageConsumer
	{		
		private Type4Generator(LayoutTable origTable, Subset subset, OTByteArrayBuilder newData, int newSTOffset, int origSTOffset, int glyphCount) throws InvalidFontException
		{
			super(origTable, subset, newData, newSTOffset, origSTOffset, 
					glyphCount, 4, false,true);
			writeHeader(newSTOffset, glyphCount);
		}
		

		private void writeHeader(int newSTOffset, int glyphCount)
		{
			builder.ensureCapacity(newSTOffset + 6 + 2*glyphCount);
			builder.setuint16(newSTOffset, 1);
			builder.setuint16(newSTOffset + 4, glyphCount);
		}
		
		static Type4Generator newInstance(LayoutTable origTable, Subset subset, OTByteArrayBuilder newData, int newSTOffset, int origSTOffset, Map newCoverages) 
		throws InvalidFontException, UnsupportedFontException
		{
			int coverageOffset = origTable.data.getOffset(origSTOffset, 2);
			NewCoverage newCoverage = (NewCoverage)newCoverages.get(new Integer(coverageOffset));
			Type4Generator generator = new Type4Generator(origTable, subset, newData, newSTOffset, origSTOffset, newCoverage.glyphCount);
			origTable.iterateCoverage(coverageOffset, subset, generator);
			return generator;
			
		}
		
		boolean[] computeMembersToKeep(int newLSOffset, int origLSOffset) throws InvalidFontException, UnsupportedFontException
		{
			int origLigaCount = origTable.data.getuint16(origLSOffset);
			boolean[] keep = new boolean[origLigaCount];
			Arrays.fill(keep, true);
			for (int i = 0; i < origLigaCount; i++)
			{
				// if this ligature requires a glyph not in our subset, drop the ligature.
				int ligatureOffset = origTable.data.getOffset(origLSOffset, 2 + 2*i);
				int ligGlyph = origTable.data.getuint16(ligatureOffset);
				if (subset.getExistingSubsetGid(ligGlyph) == -1)
				{
					keep[i] = false;
					continue;
				}
				
				int compCount = origTable.data.getuint16(ligatureOffset + 2);
				for (int j = 0; j < compCount-1; j++)
				{
					ligGlyph = origTable.data.getuint16(ligatureOffset + 4 + 2*j);
					if (subset.getExistingSubsetGid(ligGlyph) == -1)
					{
						keep[i] = false;
						break;
					}
				}
				
			}
			
			return keep;
		}
		
		int writeMember(int newLigaOffset, int origLigaOffset) throws InvalidFontException, UnsupportedFontException
		{
			int componentCount = origTable.data.getuint16(origLigaOffset + 2);
			int lSize = 4 + 2 * (componentCount - 1);
			builder.ensureCapacity(newLigaOffset + lSize);
			
			// write out new ligature gid.
			builder.setuint16(newLigaOffset, subset.getExistingSubsetGid(origTable.data.getuint16(origLigaOffset)));
			builder.setuint16(newLigaOffset + 2, componentCount);
			
			// write out new component gids
			for (int i = 0; i < componentCount - 1; i++)
			{
				builder.setuint16(newLigaOffset + 4 + 2*i, subset.getExistingSubsetGid(origTable.data.getuint16(origLigaOffset + 4 + 2*i)));
			}
			
			return lSize;
		}
		
		int whichRuleSetIndexApplies(int gid, int coverageIndex)
		{
			return coverageIndex;
		}

		int getOrigRecordSize() {
			return 0;
		}
	}
	
	int writeSubtable(int origSTOffset, int newSTOffset, 
			Map newCoverages, int lookupType, Subset subset, LookupSubset lookupSubset)	throws InvalidFontException, UnsupportedFontException
	{
		LookupSubtableGenerator generator;
		
    	switch (lookupType) {
		      case 1: 
		      {
		    	  int format = origTable.data.getuint16(origSTOffset);
		    	  Type1TargetGIDFetcher fetcher;
		    	  switch (format)
		    	  {
		    	  case 1:
		    		  fetcher = new Type1Format1TargetGIDFetcher();
		    		  ((Type1Format1TargetGIDFetcher)fetcher).setDelta(origTable.data.getint16(origSTOffset + 4));
		    		  break;
		    	  case 2:
		    		  fetcher = new Type1Format2TargetGIDFetcher();
		    		  ((Type1Format2TargetGIDFetcher)fetcher).setOrigSTOffset(origSTOffset);
		    		  break;
		    	  default:
		    		  throw new InvalidFontException("Invalid lookup type 1 format:" + format);
		    			  
		    	  }
	    		  DeltaDeterminer consumer = new DeltaDeterminer(subset, fetcher);
	    		  origTable.iterateCoverage(origTable.data.getOffset(origSTOffset, 2), subset, consumer);
	    		  if (consumer.canUseFormat1())
	    			  return writeType1Format1(origSTOffset, newSTOffset, consumer.delta());
		    	  else
		    		  generator = new Type1Format2Generator(origSTOffset, subset, builder, newSTOffset, fetcher, newCoverages);

	    		  break;
		      }
		    	  
		      case 2: 
		      case 3: 
		    	  // 1-to-n lookups have the same format.
		    	  generator = new Type2Or3Generator(subset, builder, newSTOffset, origSTOffset, newCoverages);
		    	  break;
		    	  
		      case 4: 
		    	  generator = Type4Generator.newInstance(origTable, subset, builder, newSTOffset, origSTOffset, newCoverages);
		    	  break;
		  
		      case 5: 
		    	  generator = ContextualGenerator.newContextualGenerator(origTable, builder,
		    			  origSTOffset, newSTOffset, subset, newCoverages, lookupSubset, origNumGlyphs);
		    	  break;
		      
		      case 6: 
		    	  generator = ChainingGenerator.newChainingInstance(origTable, builder, origSTOffset, newSTOffset, subset, 
		    				newCoverages, lookupSubset, origNumGlyphs);
		    	  break;

		      case 7:
		    	  builder.ensureCapacity(newSTOffset + 4);
		    	  builder.setuint16(newSTOffset, 1);
		    	  builder.setuint16(newSTOffset + 2, origTable.data.getuint16(origSTOffset+2));
		    	  return 8;
		      
		      default: {
		        throw new InvalidFontException ("Invalid GSUB lookup type (" + lookupType + ")"); }}
    	
		return generator.writeSubtable();
	}
	
	int getExtensionSubtableSize()
	{
		return 8;
	}
	
	void gatherCoveragesForSubtable(int stOffset, int lookupType, int lookupIndex, Integer subtableIndex, 
			Map coverages, Map extensionCoverages, Subset subset) 
	throws InvalidFontException, UnsupportedFontException
	{

	      Map coverageMapToUse = coverages;
	      
		  if (lookupType == 7)
		  {
			  lookupType = origTable.data.getuint16(stOffset + 2);
			  stOffset += origTable.data.getuint32asint(stOffset + 4, "Unhandled extension offset");
			  coverageMapToUse = extensionCoverages;
		  }
		  
		  
		  switch (lookupType) {
		      case 1: 
		      case 2: 
		      case 3: 
		      case 4: { 
		    	  int coverageOffset = origTable.data.getOffset (stOffset, 2);
		    	  addToCoveragesMap(origTable, coverageOffset, coverageMapToUse, lookupIndex, subtableIndex, subset);
		    	  break; }
		  
		      case 5: { 
		    	  ContextualGenerator.gatherCoveragesForSubtable(origTable, stOffset, lookupType, lookupIndex, 
		    			  subtableIndex, coverageMapToUse, subset);
		    	  break; }
		      
		      case 6: { 
		    	  ChainingGenerator.gatherCoveragesForSubtable(origTable, stOffset, lookupType, 
		    			  lookupIndex, subtableIndex, coverageMapToUse, subset);
		    	  break; 
		      }
		      default: {
		        throw new InvalidFontException ("Invalid GSUB lookup type (" + lookupType + ")"); }}	    	
	}

	int getReferencedLookupType(int origExtSubOffset) throws InvalidFontException {
		return origTable.data.getuint16(origExtSubOffset + 2);
	}

	void patchExtensionOffset(int extensionSubtableOffset, int nonExtensionSubtableOffset) {
  	  builder.setuint32(extensionSubtableOffset + 4, nonExtensionSubtableOffset - extensionSubtableOffset);
	}
	
	int getReferencedSubtableOffset(int origSTOffset) throws InvalidFontException, UnsupportedFontException
	{
		return origSTOffset + origTable.data.getuint32asint(origSTOffset + 4, "Only handle extensions that fit in 31 bits");
	}

	int getExtensionLookupType() {
		return 7;
	}

	int writeExtensionSubtable(int newSTOffset, int subTableLookupType) {
	  builder.setuint16(newSTOffset, 1);
  	  builder.setuint16(newSTOffset + 2, subTableLookupType);
  	  return 8;
	}

	boolean mustKeepFeature(int featureTagOffset) {
		return false;
	}
}
