package com.adobe.fontengine.font.opentype;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.LayoutTable.CoverageConsumer;

public abstract class LookupTableHarvester {

	  protected final int numGlyphs;
	  final LookupTable lookupTable;
	  
	  LookupTableHarvester(LookupTable lookupTable, int numGlyphs)
	  {
		  this.lookupTable = lookupTable;
		  this.numGlyphs = numGlyphs;		  
	  }
	  
	  abstract List harvestSubtables(Subset gids, int lookupOffset, int lookupType, int lookupFlag, Map lookups) 
	  throws InvalidFontException, UnsupportedFontException;
	  
	/**
	   * Gather the lookups that can be retrieved via the feature list. That is the "head" of
	   * what we can harvest. 
	   * @return a sorted list of lookup ids, which may contain duplicates.
	   */
	protected IntGrowableArray getAllFeatureListFeatures() throws InvalidFontException {
		  int featureListOffset = lookupTable.getFeatureListOffset();
		  int featureCount = lookupTable.data.getuint16 (featureListOffset);
		  
		  IntGrowableArray featuresInFeatureList = new IntGrowableArray();
		    
		   for (int f = 0; f < featureCount; f++) {
		    	int featureOffset = featureListOffset 
		               + lookupTable.data.getuint16 (featureListOffset + 2 + f*6 + 4);
		        int lookupCount = lookupTable.data.getuint16 (featureOffset + 2);
		        for (int l = 0; l < lookupCount; l++) {
		          featuresInFeatureList.set(lookupTable.data.getuint16 (featureOffset + 4 + l*2)); }
		    }
		   
		   Arrays.sort(featuresInFeatureList.getBuffer(), 0, featuresInFeatureList.getLength());
		   return featuresInFeatureList;
		   
	  }

	protected void harvestAllFeatures(IntGrowableArray featuresInFeatureList, Map applicableLookups, Subset gids) throws InvalidFontException, UnsupportedFontException {
		  if (featuresInFeatureList.getLength() > 0)
		   {
			   harvest(gids, featuresInFeatureList.getBuffer()[0], applicableLookups);
			   for (int i = 1; i < featuresInFeatureList.getLength(); i++)
			   {
				   if (featuresInFeatureList.get(i) != featuresInFeatureList.get(i-1))
					   harvest(gids, featuresInFeatureList.get(i), applicableLookups);
			   }
		   }
	  }

	protected void harvest(Subset gids, int lookupIndex, Map lookups) throws InvalidFontException, UnsupportedFontException {
	      // Convert to lookup offsets
	      int lookupListOffset = lookupTable.getLookupListOffset ();
	      int offset = lookupTable.data.getOffset (lookupListOffset, 2 + 2*lookupIndex); 
	      int lookupType = lookupTable.data.getuint16 (offset);
		  int lookupFlag = lookupTable.data.getuint16 (offset + 2);
		  List subtables = harvestSubtables(gids, offset, lookupType, lookupFlag, lookups);
		  
		  if (subtables != null)
		  {
			  lookups.put(new Integer(lookupIndex), subtables);
		  }
	  }


	  protected List gatherLookupsForCoverage(int stOffset, int coverageDelta, Subset gids, List subtables, CoverageLookupHarvester harvester, int subtable) throws InvalidFontException, UnsupportedFontException {
		  
		  CoverageHarvester coverageHarvester = new CoverageHarvester(stOffset, gids, subtables, harvester, subtable);
		  lookupTable.iterateCoverage(lookupTable.data.getOffset (stOffset, coverageDelta), gids, coverageHarvester);
		  return coverageHarvester.subtables;
	  }


	protected List gatherLookupsForCoverageAllSubtables(Subset gids, int offset, CoverageLookupHarvester harvester) throws InvalidFontException, UnsupportedFontException {
		  List subtables = null;
		  int subtableCount = lookupTable.data.getuint16 (offset+ 4);
		      
		  for (int i = 0; i < subtableCount; i++)
		  {
		    int stOffset = lookupTable.data.getOffset (offset, 6 + 2*i);
		    subtables = gatherLookupsForCoverage(stOffset, 2, gids, subtables, harvester, i);
		  }
		  return subtables;
	  }


	protected interface CoverageLookupHarvester { 
		  /**
		   * Given a subtable in a lookup, and a glyph (and related coverage index) in the initial coverage, verify that the lookup applies (and do
		   * any processing that may be necessary for an applicable lookup. For example, with GSUB, perhaps gather the substitution glyph ids).
		   * @return true iff the subtable can apply to 'gids'.
		 * @throws UnsupportedFontException 
		   */
		  abstract boolean lookupApplies(int coverageGlyph, int stOffset, int coverageIndex, Subset gids) throws InvalidFontException, UnsupportedFontException;
		  
		  /**
		   * If you've seen that a lookup applies (lookupApplies has returned true), should the caller continue to iterate the relevant coverage?
		   */
		  abstract boolean keepGoing();
	  }
	
	protected class  CoverageHarvester implements CoverageConsumer {
		  final int stOffset;
		  final Subset gids;
		  final CoverageLookupHarvester harvester;
		  final int subtable;
		  List subtables;
		  
		  CoverageHarvester(int stOffset, Subset gids, List subtables, CoverageLookupHarvester harvester, int subtable)
		  {
			  this.stOffset = stOffset;
			  this.gids = gids;
			  this.subtables = subtables;
			  this.harvester = harvester;
			  this.subtable = subtable;
		  }
	  
		  public boolean glyphInfo(int gid, int coverageIndex) throws InvalidFontException, UnsupportedFontException {
			  if (harvester.lookupApplies(gid, stOffset, coverageIndex, gids)) 
			  {
				  if (subtables == null)
		    		  subtables = new ArrayList();
		    	  Integer i = new Integer(subtable);
		    	  if (!subtables.contains(i))
		    		  subtables.add(i);
		    	  
		    	  return harvester.keepGoing();
			  }
			  
			  return true;
		  }
	  }

}
