/*************************************************************************
 * ADOBE CONFIDENTIAL
 * __________________
 *
 * Copyright 2014 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 **************************************************************************/
package com.adobe.granite.translation.api;

import java.util.Map;

import org.apache.sling.api.resource.Resource;

import com.adobe.granite.translation.api.TranslationConstants.TranslationMethod;


/**
 * The TranslationManager maintains a list of installed TranslationServiceFactory instances within the system. These
 * are registered with the Manager at activation and must contain a unique "FactoryName" identifier. The
 * TranslationManager is obtained via the normal OSGi service registry. This service creates TranslationService
 * instances using the respective factories via one of the "createTranslationService()" methods below.
 */
public interface TranslationManager {

    /**
     * Checks if the given resource has any translation service specified, and all the properties are set correctly.
     * it returns false if translation service is not correctly configured or return true otherwise
     * @param resource resource
     * @return boolean
     * @throws TranslationException the translation exception
     */
    boolean isTranslationServiceConfigured(Resource resource) throws TranslationException;

    /**
     * Creates a <code>TranslationService</code> instance registered with the given translation factory name.
     * TranslationManager returns the TranslationService instance From within a JSP for example: TranslationManager tm
     * = sling.getService(TranslationManager.class); TranslationService translationService =
     * tm.createTranslationService(resource); Or from within a Component/Service you would import the
     * TranslationManager Object:
     * @param factoryName the translation factory name
     * @return the <code>TranslationService</code> instance
     * @throws TranslationException the translation exception
     */
    @Deprecated
    TranslationService createTranslationService(String factoryName) throws TranslationException;

    /**
     * @deprecated  since 2.4.0, use {@link #createTranslationService(String, TranslationMethod, String, Resource)} instead.
     * Creates a <code>TranslationService</code> instance registered with the given translation factory name.
     * TranslationManager returns the TranslationService instance From within a JSP for example: TranslationManager tm
     * = sling.getService(TranslationManager.class); TranslationService translationService =
     * tm.createTranslationService(factoryName, translationMethod, strCloudConfigPath); Or from within a
     * Component/Service you would import the TranslationManager Object:
     * @param factoryName the translation factory name
     * @param translationMethod the translation method
     * @param strCloudConfigPath path to cloud config, if this value is null, then factory name is used. This cloud
     *            config is passed to service factory.
     * @return the <code>TranslationService</code> instance
     * @throws TranslationException the translation exception
     */
    @Deprecated
    TranslationService createTranslationService(String factoryName, TranslationMethod translationMethod,
        String strCloudConfigPath) throws TranslationException;

    /**
     * Creates a <code>TranslationService</code> instance registered with the given translation factory name.
     * TranslationManager returns the TranslationService instance. For example: TranslationManager tm
     * = sling.getService(TranslationManager.class); TranslationService translationService =
     * tm.createTranslationService(resource, factoryName, translationMethod, strCloudConfigPath); Or you
     * can import TranslationManager Object:
     * @param factoryName the translation factory name
     * @param translationMethod the translation method
     * @param strCloudConfigPath path to cloud config, if this value is null, then factory name is used. This cloud
     *            config is passed to service factory.
     * @param resource context-aware resource, containing the content to be translated.
     * @return the <code>TranslationService</code> instance
     * @throws TranslationException the translation exception
     */
    default TranslationService createTranslationService(String factoryName, TranslationMethod translationMethod, String strCloudConfigPath,
        Resource resource) throws TranslationException {
        return createTranslationService(factoryName, translationMethod, strCloudConfigPath);
    }

    /**
     * Creates a <code>TranslationService</code> instance based upon logic determined by the properties of the passed
     * in resource. The Translation configuration logic is used to determine which properties to base the decision off
     * of (Content Path, Content Category, Provider Name stored on content). By default a configured factory name
     * (TranslationConstants.TN_FACTORY_NAME) takes precedence. If no Connector can be determined the system default
     * TranslationService will be returned.
     * @param resource The resource of a content node containing the content to be translated.
     * @return the <code>TranslationService</code> instance
     * @throws TranslationException the translation exception
     */
    TranslationService createTranslationService(Resource resource) throws TranslationException;

    /**
     * Creates a <code>TranslationService</code> instance based upon logic determined by the properties of the passed
     * in resource. The Translation configuration logic is used to determine which properties to base the decision off
     * of (Content Path, Content Category, Provider Name stored on content). If no Connector can be determined return
     * null.
     * @param resource The resource of a content node containing the content to be translated.
     * @return the <code>TranslationService</code> instance
     * @throws TranslationException the translation exception
     */
    TranslationService createPostEditService(Resource resource) throws TranslationException;

    /**
     * CreatePostEditService is used as part of an asynchronous translation process.  A Post Edit Service may be
     * included in a translation workflow.  This would normally happen after a synchronous translation service.
     * to improve the quality of a machine translation. The Post Edit service defines the human asyncronous
     * translation service to be used via a connector to this service. A connector using asynchronous translation
     * is required for a Post Edit Service. 
     * @return Map&lt;String, String&gt; containing currently installed factories (&lt;Factory Name, Factory Label&gt;)
     * @throws TranslationException the translation exception
     */
    @Deprecated
    Map<String, String> getAvailableFactoryNames() throws TranslationException;

    /**
     * Returns a Map of the currently available TranslationServiceFactories The service returns both the factory name
     * and a factory label for use in drop downs based on TranslationMethod. This saves us from having to make a
     * second call (per factory) to getLabel(); As noted: the factory names (the ID) is the KEY and the factory label
     * is the VALUE. This is the value that can be displayed when factory will show up in UI. The return is a
     * MAP&lt;NAME, LABEL&gt; of factories within the system.
     * @param translationMethod Use TranslationMethod parameter to filter out services and return only those factories
     *            which support this translation method. Pass null to return all installed factories.
     * @return Map&lt;String, String&gt; containing currently installed factories (&lt;Factory Name, Factory Label&gt;)
     * @throws TranslationException the translation exception
     */
    Map<String, String> getAvailableFactoryNames(TranslationMethod translationMethod) throws TranslationException;
}