/* ****************************************************************************
 *
 *	File: CosDocument.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.cos;

import java.io.DataOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.rmi.server.UID;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import com.adobe.internal.io.ByteReader;
import com.adobe.internal.io.ByteWriter;
import com.adobe.internal.io.ByteWriterFactory.Fixed;
import com.adobe.internal.io.stream.IO;
import com.adobe.internal.io.stream.InputByteStream;
import com.adobe.internal.io.stream.NullOutputByteStream;
import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.io.stream.StreamManager;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosParseException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosParseException.CosParseErrorType;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFInvalidParameterException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFUnsupportedFeatureException;
import com.adobe.internal.pdftoolkit.core.types.ASName;
import com.adobe.internal.pdftoolkit.core.types.ASString;
import com.adobe.internal.pdftoolkit.core.util.PDFDocEncoding;
import com.adobe.internal.pdftoolkit.core.util.StringOps;
import com.adobe.internal.pdftoolkit.core.util.Utility;

/**
 * @author gish
 *
 * Class CosDocument
 * Represents a PDF document at the COS (Carousel Object System) layer. The
 * COS layer is a low-level representation of the PDF document.
 */
public final class CosDocument
{
	private static final String XREF = "xref\n";
	private static final String EOF = "\n%%EOF\n";
	private static final String STARTXREF = "startxref\n";
	// Constants
	private static final int OBJSTM_MAXNUMOBJS = 200;	// Never put more than this many in any one object stream
	private static final byte[] PDF_MARKER = {'%', 'P', 'D', 'F', '-'};
	private static final int[] PDF_MARKER_KMPARRAY = Utility.ComputeKMPNextArray(PDF_MARKER);
	private static final byte[] FDF_MARKER = {'%', 'F', 'D', 'F', '-'};
	private static final int[] FDF_MARKER_KMPARRAY = Utility.ComputeKMPNextArray(FDF_MARKER);
	
	static final HashSet<ASName> dictionariesNotToBeCompressed = new HashSet<ASName>();
	static {
		dictionariesNotToBeCompressed.add(ASName.k_Sig);
		dictionariesNotToBeCompressed.add(ASName.k_DocTimeStamp);
	}
	
	private CosOpenOptions mOptions;
	private StreamManager mStreamManager;
	private ByteReader mByteReader;

	private CosEncryption mEncryption;
	private InputByteStream mBuf;				// Underlying PDF data for the document - can be null if no backing store
	private XRefTable mXRef;				// Cross reference table
	private Object[] mUserData = new Object[2];	// Used by the higher-level PDF layer
	private CosList mObjectInfos;				// List of all ObjectInfos by object number
	private CosList mCompObjInfos;				// List of output compressed ObjectInfos by object number
	private int mNumObjects;				// Number of objects in the document (max obj number + 1)
	private int mOrigNumObjects;				// Original mNumObjects value before modifications
	private CosLinearization mCosLin;			// For use during linearization
	private String mToSaveVersion;				// Client set version to use at save time
	private Map mToSaveExtensions;				// Client set /Catalog/Extensions to use at save time
	private boolean mToSaveExtensionsInit;			// True if mToSaveExtensions was initialized.

	private static final String MAX_VERSION = "1.7";
	private String mHeader = "%PDF-1.7\r\n";
	private String mHeaderToken;

	private CosDictionary mTrailer;
	private CosDictionary mRoot;
	private boolean mIsFDF;
	private boolean mCacheEnabled = true;			// No longer an option - always on
	private int mHeaderTrashCount;				// Count of trash bytes before header
	private int mTrailerTrashCount;				// Count of trash bytes after trailer
	private boolean mDocIsDirty;				// Per-document dirty flag
	private boolean mSaveToCopy;				// Last (or current) save is copy or FDF
	private boolean mIsLinearized;				// Document is linearized
	private boolean mWasLinearized;				// Document was linearized, may now have updates
	private boolean mForceCompress;				// Force Flate compression on uncompressed streams
	private long garbageLength = 0;				//this is the garbage which is found just after last EOF marker of document.
	private CosRepairList cosRepairList;        // It holds a parallel cache of changes done to cos objects for repair.  
	private boolean documentCosLevelRepaired;   // Has cos level repair done on this document ?
	private boolean useRepairList = true;       
	private long nextIncrementalSectionOffset = -1;// This is the offset where next incremental section shall be written if
												   // the document is saved incrementally.
	// The logging output path
	private static String mLoggingPath = null;
	
	/**
	 * Enum set holding types of repair done for this document.
	 */
	private EnumSet<REPAIRTYPE> repairTypes = EnumSet.noneOf(REPAIRTYPE.class);;

	/**
	 * Constructs an empty, uninitialized COS document.
	 * @throws PDFSecurityException
	 * @throws PDFIOException
	 * @throws PDFCosParseException
	 */
	private CosDocument(CosOpenOptions options)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		mOptions = options;
		mStreamManager = StreamManager.newInstance(options.getByteWriterFactory(), null);
		mEncryption = new CosEncryption(null);
		init();
		mEncryption.setupDecryption();
	}

	/**
	 * CosSaveParams.XREFTABLE is specified and compressed objects are present
	 * encryption parameter is provided
	 * @throws PDFSecurityException
	 */
	private CosDocument(ByteReader byteReader, CosOpenOptions options)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		mOptions = options;
		mByteReader = byteReader;
		mStreamManager = StreamManager.newInstance(options.getByteWriterFactory(), byteReader);
		init(byteReader);
	}

	/**
	 * 
	 * Returns an empty document with some initialized data structures so that
	 * you can build a new CosDocument.
	 * @param byteReader	the PDF data for the document
	 * @param options the options to provide for the new document
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public static CosDocument newDocument(ByteReader byteReader, CosOpenOptions options)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return new CosDocument(byteReader, options);
	}

	/**
	 * 
	 * Returns an empty document with some initialized data structures so that
	 * you can build a new CosDocument.
	 * @param options the options to provide for the new document
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public static CosDocument newDocument(CosOpenOptions options)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return new CosDocument(options);
	}

	/**
	 * 
	 * Once a CosDocument was instanciated from a PDFCore object, that PDFCore object
	 * can't be used again. You can generate a new PDFCore object by calling
	 * CosDocument.finish().
	 *
	 * @param pdfCore
	 * @return CosDocument from the input PDFCore.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public static CosDocument newDocument(PDFCore pdfCore)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		if (pdfCore == null) {
			throw new PDFUnsupportedFeatureException("PDFCore must be non-null");
		}
		return pdfCore.getCosDoc();
	}

	/**
	 * Get the <code>StreamManager</code> to use for this document.
	 *
	 * @return the stream manager for this document
	 */
	public StreamManager getStreamManager()
	{
		return mStreamManager;
	}

	/**
	 * Figure out the widths values for an xref stream.
	 */
	private int[] setWidthsArray(Object list, CosStream xrefStream, long xrefStmPos)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		ArrayList secList;
		int[] widths = new int[3];
		long maxW2 = xrefStmPos;
		long thisW2;
		int maxW3 = 0;
		int thisW3;
		if (list instanceof ArrayList) {
			secList = (ArrayList)list;
		} else {
			secList = new ArrayList();
			secList.add(list);
		}
		Iterator secIter = secList.iterator();
		while (secIter.hasNext()) {
			CosList infosList = (CosList)secIter.next();
			Iterator infoIter = infosList.iterator();
			while (infoIter.hasNext()) {
				CosObjectInfo info = (CosObjectInfo)infoIter.next();
				if (!info.isFree()) {
					if (info.isCompressed()) {
						thisW2 = info.getStreamInfo().getObjNum();
						thisW3 = info.getStreamNdx();
					} else {
						thisW2 = info.getPos();
						thisW3 = info.getObjGen();
					}
					if (thisW2 > maxW2)
						maxW2 = thisW2;
					if (thisW3 > maxW3)
						maxW3 = thisW3;
				}
			}
		}
		widths[0] = 1;
		int widthW2 = 1;
		while (maxW2 > 255) {
			widthW2++;
			maxW2 >>= 8;
		}
		widths[1] = widthW2;
		int widthW3 = 1;
		while (maxW3 > 255) {
			widthW3++;
			maxW3 >>= 8;
		}
		widths[2] = widthW3;
		CosArray wValues = createCosArray();
		wValues.addInt(widths[0]);
		wValues.addInt(widths[1]);
		wValues.addInt(widths[2]);
		xrefStream.put(ASName.k_W, wValues);
		CosDictionary parmsDict = createDirectCosDictionary();
		parmsDict.put(ASName.k_Columns, 1 + widthW2 + widthW3);
		parmsDict.put(ASName.k_Predictor, 12);
		xrefStream.put(ASName.k_DecodeParms, parmsDict);
		return widths;
	}

	/**
	 * 
	 * @return new PDFCore which can be used to reinstantiate a CosDocument
	 */
	public PDFCore finish() {
		return new PDFCore(this);
	}

	/**
	 * 
	 * Close the <code>CosDocument</code> and free any resources associated
	 * with it.
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 * @throws PDFCosParseException
	 */
	public void close()
		throws PDFIOException, PDFCosParseException, PDFSecurityException
	{
		try {
			try {
				if (mBuf != null) {
					mBuf.close();
				}
			} finally {
				try {
					if (mObjectInfos != null) {
						closeAllCosObjects();
					}
				} finally {
					try {
						if (mXRef != null) {
							mXRef.close();
						}
					} finally {
						if (mStreamManager != null) {
							mStreamManager.close();
						}
					}
				}
			}
		} catch (IOException e) {
			throw new PDFIOException(e);
		}

		// now let's null out stuff to encourage garbage collection
		mBuf = null;
		mEncryption = null;
		mObjectInfos = null;
		mRoot = null;
		mTrailer = null;
		mXRef = null;
		mStreamManager = null;
	}

	private long headerPos(InputByteStream byteStream)
		throws IOException
	{
		// Is this stream a PDF data stream?
		// See if you can find the PDF marker in the first 1024 bytes.
		// see PDF 1.5, Appendix H, note 13
		byte[]pdfMarker = {'%', 'P', 'D', 'F', '-'};
		int size = 1024;
		if (size > byteStream.length()) size = (int)(byteStream.length());
		byte[] header = new byte[size];
		byteStream.seek(0);
		byteStream.read(header);
		long result = Utility.KMPFindFirst(pdfMarker, Utility.ComputeKMPNextArray(pdfMarker), header);
		if (result >= 0) {
			byte[] linearizationMarker = {'/', 'L', 'i', 'n', 'e', 'a', 'r', 'i', 'z', 'e', 'd', ' ', '1'};
			mWasLinearized = mIsLinearized = Utility.KMPFindFirst(linearizationMarker, Utility.ComputeKMPNextArray(linearizationMarker), header) >= 0;
		}
		return result;
	}

	private long fdfHeaderPos(InputByteStream byteStream)
		throws IOException
	{
		// Is this stream an FDF data stream?
		// See if you can find the FDF marker in the first 1024 bytes.
		// see PDF 1.5, Appendix H, note 13
		// SLG - modified to read FDF files, too.
		byte[]fdfMarker = {'%', 'F', 'D', 'F', '-'};
		int size = 1024;
		if (size > byteStream.length()) size = (int)(byteStream.length());
		byte[] header = new byte[size];
		byteStream.seek(0);
		byteStream.read(header);
		return Utility.KMPFindFirst(fdfMarker, Utility.ComputeKMPNextArray(fdfMarker), header);
	}

	/**
	 * Sets up and initializes the various data structures for a
	 * CosDocument being created from input data
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	private void init(ByteReader byteReader)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			InputByteStream byteStream = mStreamManager.getInputByteStream(byteReader);
			initObjectList();
			long hdr = headerPos(byteStream);
			mHeaderTrashCount = (int)hdr;
			if (hdr < 0) {
				hdr = fdfHeaderPos(byteStream);
				if (hdr >= 0)
					mIsFDF = true;
			}
			long eof = mOptions.getEofValue();
			if (eof > byteStream.length()) {
				eof = byteStream.length();
				mOptions.setEofValue(eof);
			}
			if (hdr < 0)
				throw new PDFCosParseException("Stream does not represent a PDF document.");
			if (hdr == 0 && eof == byteStream.length())
				mBuf = byteStream;
			else {
				/*
				 * Bug 1017945 pointed out a fact that is not obvious from reading
				 * the PDF reference manual. The manual states in appendix H, note
				 * 13 that the header can appear anywhere in the first 1024 bytes
				 * of the file. It does not add that the offsets in the xref
				 * section (and the startxref) are relative to the header, not
				 * relative to the start of the file. In fact, in section 3.4.3,
				 * the manual states incorrectly that the byte offset is "the number
				 * of bytes from the beginning of the file to the beginning of the
				 * object."
				 */

				/*
				 * Must close the unused stream but also need to be careful to not
				 * close the last/only InputByteStream referring to the ByteReader.
				 * We use the getInputByteStream() method rather than slice()
				 * because we don't want a slice but the one, true stream to
				 * back the document using the original ByteReader. During a
				 * slice operation a ByteReader implementation may give us a
				 * completely different ByteReader to use. We don't want that.
				 */
				mBuf = mStreamManager.getInputByteStream(byteReader, hdr, eof - hdr);
				byteStream.close();
			}
			mEncryption = new CosEncryption(this);
			mXRef = new XRefTable(this, mBuf.slice(), mOptions.getNoPreloadXRef(), mIsFDF);
			try {
				mXRef.loadMainInfos();
				mXRef.loadUpdateInfos();
				if(mOptions != null && mOptions.skipCorruptObjects())
					linkInfos();
				if(garbageLength > 0)
					repairTypes.add(REPAIRTYPE.garbageRepair);					
			} catch (PDFCosParseException e) {
				if (mOptions.getRepairEnabled()) {
					mXRef.rebuild();
					repairTypes.add(REPAIRTYPE.xrefRepair);
				} else {
					throw new PDFCosParseException("XRef repair required but not enabled", e);
				}
			}
			mXRef.setupTrailerEncryption();
			CosDictionary trailer = mXRef.getTrailer();
			if (!mIsFDF) {
				CosObject sz = trailer.get(ASName.k_Size);
				int size = ((CosNumeric)sz).intValue();
				int numObjectsDefinedInXRefEntries = mXRef.getNumObjectsDefinedInXRefEntries();
				if(mOptions.getRepairEnabled())
				{
					if(size >= numObjectsDefinedInXRefEntries)
					{
						mOrigNumObjects = mNumObjects = size;
					}
					else
					{
						mOrigNumObjects = mNumObjects = numObjectsDefinedInXRefEntries;						
						repairTypes.add(REPAIRTYPE.sizeEntryRepair);
					}
				}
				else
				{
					mOrigNumObjects = mNumObjects = size;
				}
					
				getEncryption().setupDecryption();
			}
			mTrailerTrashCount = (int)(mBuf.length() - mXRef.getRevisionEOF(mXRef.getNumRevisions() - 1));
			if (mXRef.getNumRevisions() != 1)
				mIsLinearized = false;
			mTrailer = trailer;
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 * This method populates for each cos object info, the offset of the next object 
	 * as present in the document stream.
	 */
	private void linkInfos(){
		Iterator<Object> itr = this.mObjectInfos.iterator();
		CosObjectInfo[] infos = new CosObjectInfo[mObjectInfos.count()];
		int index = 0;
		// get the array of cosobjects
		while(itr.hasNext()){
			infos[index++] = (CosObjectInfo) itr.next();
		}
		// sort the infos on the basis of their positioning in document stream.
		Arrays.sort(infos, new Comparator<CosObjectInfo>() {

			public int compare(CosObjectInfo o1, CosObjectInfo o2) {
				long pos1 = o1.getPos();
				if(pos1 == 0 && o1.getStreamInfo() != null){
					pos1 = o1.getStreamInfo().getPos();
				}
				long pos2 = o2.getPos();
				if(pos2 == 0 && o2.getStreamInfo() != null){
					pos2 = o2.getStreamInfo().getPos();
				}
				//For objects at the same position , compare should return 0.
				//return (pos1 < pos2) ? -1 : 1;
		        return Long.valueOf(pos1).compareTo(pos2);
			}
		});
		// set the offset of next object in each info.
		for(int i=0;i<infos.length-1;i++){
			if(infos[i+1].getPos() == 0){
				if(infos[i+1].getStreamInfo() != null)
					infos[i].setNextObjPos(infos[i+1].getStreamInfo().getPos());
			}else
				infos[i].setNextObjPos(infos[i+1].getPos());
		}
	}
	/**
	 * Sets up and initializes the various data structures for a new
	 * CosDocument
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	private void init()
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		initObjectList();
		mBuf = null;
		mXRef = new XRefTable(this);

		// Create the catalog
		mNumObjects++;
		CosDictionary cosCatalog = createCosDictionary(CosObject.INDIRECT);
		cosCatalog.put(ASName.k_Type, ASName.k_Catalog);

		// Put the catalog into the trailer as /Root
		mXRef.getTrailer().put(ASName.k_Root, cosCatalog);
		mEncryption = new CosEncryption(this);
		mTrailer = mXRef.getTrailer();		
	}

	private void initObjectList()
	{
		mNumObjects = 0;
		mObjectInfos = new CosList();
		CosObjectInfo freeInfo = getObjectInfo(0, 65535);
		freeInfo.markFree();
	}

	/**
	 * 
	 * Returns the version string of the PDF file (e.g. "1.5"). This string is
	 * typically found in the header of the file. However, the
	 * version also can be an entry in the catalog; if the version
	 * is more recent and the file has been saved incrementally,
	 * the version entry in the catalog overrides the header. A
	 * newly created document will not have a header yet, and the
	 * version information will either have to be put in the catalog
	 * or supplied explicitly to the save routine when the document
	 * is first written. The version string is discussed in
	 * sections 3.4.1 and H.1 of the PDF Reference Manual version 1.4.
	 *
	 * @return PDF file version string.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public String getOriginalVersion()
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		String result = null;
		String catalogVersion = null;
		String headerVersion = null;

		// First, check for version entry in the catalog
		CosObject version = getRoot().get(ASName.k_Version);
		if (version instanceof CosName) {///PDFRef 1.7 table 3.25 says this entry can only be name object
			catalogVersion = ((CosName)version).nameValue().asString(true);
		}
		
		//Bug 4003978:- Need to gracefully handle an incorrect version
		//Acrobat gracefully neglects incorrect version no. without throwing any error
		try{
			if(catalogVersion != null){
				Float.parseFloat(catalogVersion);
				if(catalogVersion.charAt(0) != '1') //If this condition satisfies then also the version of PDF is invalid in the catalog Dictionary
					catalogVersion = null;
			}
		}
		catch(NumberFormatException e){
			catalogVersion = null;  //As version no. starting with V is invalid, it should be of the form M.m
		}
		try {
			// Now, get the version from the header if there is an input stream
			if (mBuf != null) {
				long size = 1024;
				if (size > mBuf.length())
					size = mBuf.length();
				InputByteStream firstK = mBuf.slice(0, size);

				// The init routine already verified that pdfMarker occurs in the first 1024 bytes
				long verpos = Utility.KMPFindFirst(PDF_MARKER, PDF_MARKER_KMPARRAY, firstK);
				if (verpos < 0) {
					firstK.seek(0);
					verpos = Utility.KMPFindFirst(FDF_MARKER, FDF_MARKER_KMPARRAY, firstK);
				}
				// Bug#3569217 : Header of some PDF files do not end with a new line char.
				// To handle those cases, a space character is deemed sufficient to signify line end.
				String headerVersionLine = CosToken.readLine(firstK.seek(verpos), true);

				// Get the version string out of the line if it's there
				int startindx = headerVersionLine.indexOf("1.");
				if (startindx > 0)
					headerVersion = headerVersionLine.substring(startindx, startindx+3);
			}
		} catch (IOException e) {
			throw new PDFIOException(e);
		}

		// If we have a catalog version and a header version, return the most recent.
		// Otherwise, return the version from either the catalog or the header.
		// Return null if no version specified.
		if (catalogVersion != null && headerVersion != null) {
			int decision = Float.compare(Float.parseFloat(catalogVersion), Float.parseFloat(headerVersion));
			if (decision > 0)
				result = catalogVersion;
			else
				result = headerVersion;
		}
		else if (catalogVersion != null)
			result = catalogVersion;
		else if (headerVersion != null)
			result = headerVersion;
		return result;
	}

	public String procureOriginalVersion()
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		String version = getOriginalVersion();
		if (version == null)
			version = MAX_VERSION;
		return version;
	}

	public Map getToSaveExtensions()
	{
		return mToSaveExtensions;
	}

	public void setToSaveExtensions(Map extensions)
	{
		mToSaveExtensions = extensions;
		mToSaveExtensionsInit = true;
	}

	public boolean isToSaveExtensionsInitialized() {
		return mToSaveExtensionsInit;
	}

	public String getToSaveVersion()
	{
		return mToSaveVersion;
	}

	public void setToSaveVersion(String version)
	{
		mToSaveVersion = version;
	}

	public String procureToSaveVersion()
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		String version = getToSaveVersion();
		if (version == null)
			version = getOriginalVersion();
		if (version == null)
			version = MAX_VERSION;
		return version;
	}

	/**
	 * Check the parameters so they can be used without further processing
	 * in all three save styles.
	 */
	private CosSaveParams normalizeParams(CosSaveParams params)
		throws PDFCosParseException, PDFInvalidParameterException, PDFIOException, PDFSecurityException
	{
		params = (CosSaveParams)params.clone();
		String originalVersion = getOriginalVersion();
		if (originalVersion == null)
			originalVersion = MAX_VERSION;
		String toSaveVersion = (mToSaveVersion == null) ? params.getVersion() : mToSaveVersion;
		if (toSaveVersion == null)
			toSaveVersion = originalVersion;
		int xrefStyle = params.getXrefStyle();
		int saveStyle = params.getSaveStyle();
		boolean saveToCopy = params.getSaveToCopy();
		if (saveStyle == CosSaveParams.SAVE_STYLE_DEFAULT)
			saveStyle = CosSaveParams.SAVE_STYLE_FULL;
		if (saveStyle == CosSaveParams.SAVE_STYLE_INCREMENTAL) {
			// XRef style must remain the same with incremental save
			int xrefType = mXRef.getType();
			if (xrefType == XRefTable.tTable)
				xrefStyle = CosSaveParams.XREFTABLE;
			else if (xrefType == XRefTable.tStream)
				xrefStyle = CosSaveParams.XREFSTREAM;
			else
				xrefStyle = CosSaveParams.XREFHYBRID;
			if (!originalVersion.equals(toSaveVersion)) {
				// Version changes are strictly limited with incremental save
				String[] originalVersionElements = originalVersion.split("\\.");
				int originalVersionMajorVal = Integer.parseInt(originalVersionElements[0]);
				int originalVersionMinorVal = 0;
				if (originalVersionElements.length > 1)
					originalVersionMinorVal = Integer.parseInt(originalVersionElements[1]);
				String[] toSaveVersionElements = toSaveVersion.split("\\.");
				int toSaveVersionMajorVal = Integer.parseInt(toSaveVersionElements[0]);
				int toSaveVersionMinorVal = 0;
				if (toSaveVersionElements.length > 1)
					toSaveVersionMinorVal = Integer.parseInt(toSaveVersionElements[1]);
				if (toSaveVersionMajorVal == 1 && toSaveVersionMinorVal < 4)
					throw new PDFInvalidParameterException("Cannot change the version during an incremental save to PDF 1.3 and earlier");
				if (toSaveVersionMajorVal < originalVersionMajorVal)
					throw new PDFInvalidParameterException("Cannot lower version in incremental save");
				if ((toSaveVersionMajorVal == originalVersionMajorVal) && (toSaveVersionMinorVal < originalVersionMinorVal))
					throw new PDFInvalidParameterException("Cannot lower version in incremental save");
				getRoot().put(ASName.k_Version, ASName.create(toSaveVersion));
			}
		} else {
			if (saveToCopy && saveStyle == CosSaveParams.SAVE_STYLE_LINEAR)
				throw new PDFInvalidParameterException("SaveToCopy mode does not support linear save");
			if (params.getHeader() != null)
				mHeader = params.getHeader();
			else {
				mHeaderToken = params.getHeaderToken();
				if (mHeaderToken != null) {
					if (mHeaderToken.length() > 10)
						throw new PDFInvalidParameterException("Header token too long");
					for (int i = 0; i < mHeaderToken.length(); i++) {
						char c = mHeaderToken.charAt(i);
						if (c == '\r' || c == '\n' || (c & 0xFF00) != 0)
							throw new PDFInvalidParameterException("Illegal header token");
					}
				}
				buildHeaderString(toSaveVersion);
			}
			if (xrefStyle == CosSaveParams.CURRENT_XREF) {
				if (mXRef.isNew()) {
					xrefStyle = CosSaveParams.XREFTABLE;
				} else {
					int xrefType = mXRef.getType();
					if (xrefType == XRefTable.tTable)
						xrefStyle = CosSaveParams.XREFTABLE;
					else if (xrefType == XRefTable.tStream)
						xrefStyle = CosSaveParams.XREFSTREAM;
					else
						xrefStyle = CosSaveParams.XREFHYBRID;
				}
			}
			if (xrefStyle == CosSaveParams.XREFSTREAM) {
				float versionVal = 0;
				versionVal = Float.parseFloat(toSaveVersion);
				if (versionVal < 1.5)
					xrefStyle = CosSaveParams.XREFHYBRID;
			}
			if (xrefStyle == CosSaveParams.XREFHYBRID && getRoot().get(ASName.k_StructTreeRoot) == null)
				xrefStyle = CosSaveParams.XREFTABLE;
			getRoot().remove(ASName.k_Version);
		}
		if (mHeader.indexOf("%FDF-") >= 0)
			mIsFDF = true;
		if (mIsFDF) {
			if (saveStyle != CosSaveParams.SAVE_STYLE_FULL)
				throw new PDFInvalidParameterException("FDF mode supports full save only");
			xrefStyle = CosSaveParams.XREFTABLE;
		}
		params.setHeader(mHeader);
		params.setVersion(toSaveVersion);
		params.setXrefStyle(xrefStyle);
		params.setSaveStyle(saveStyle);
		mForceCompress = params.getForceCompress();
		mSaveToCopy = saveToCopy;
		return params;
	}

	/**
	 * 
	 * Retrieves the trailer from the document.
	 *
	 * @return Document's trailer dictionary.
	 */
	public CosDictionary getTrailer()
	{
		// Guard against NULL document
		if (mTrailer == null) {
			if (mXRef != null)
				mTrailer = mXRef.getTrailer();
		}
		return mTrailer;
	}

	public CosDictionary[] getTrailerList()
	{
		CosDictionary[] list = null;
		if (mXRef != null)
			list = mXRef.getTrailerList();
		return list;
	}

	/**
	 * 
	 * Retrieves the root CosDictionary of the document if it's not already
	 * cached; otherwise, returns the cached root.
	 * @return Document's Root dictionary
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosDictionary getRoot()
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject catalog;
		if (mRoot != null)
			catalog = mRoot;
		else {
			CosDictionary trailer = getTrailer();
			catalog = trailer.get(ASName.k_Root);
			if (! (catalog instanceof CosDictionary))
				throw new PDFCosParseException("Document does not have a catalog of type dictionary.");
			mRoot = (CosDictionary) catalog;
		}
		return (CosDictionary) catalog; // here catalog will always be CosDictionary
	}

	/**
	 * 
	 * Retrieves the Information CosDictionary of the document, may be null.
	 * @return Document's Information dictionary
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosDictionary getInfo()
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject docInfo = getTrailer().get(ASName.k_Info);
		if (docInfo instanceof CosDictionary)
			return (CosDictionary)docInfo;
		return null;
	}

	/**
	 * Obtain a <i>slice</i> of the underlying InputByteStream for the document.
	 * This <i>slice</i> belongs to the caller of this method and <b>must</b>
	 * be closed by them when they are finished with it.
	 *
	 * @param start the offset in the <code>CosDocument</code>'s <code>InputByteStream</code>
	 * to use as the start of the returned <i>slice</i>
	 * @param length the length of the slice to create
	 * @return A slice of the underlying InputByteStream for the document.
	 * @throws PDFIOException
	 * @exclude
	 */
	public InputByteStream getStream(long start, long length)
		throws PDFIOException
	{
		try {
			return mBuf.slice(start, length);
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 * Obtain a slice of the underlying InputByteStream for the document.
	 * This <i>slice</i> belongs to the caller of this method and <b>must</b>
	 * be closed by them when they are finished with it.
	 *
	 * @return A slice of the underlying InputByteStream for the document.
	 * @throws PDFIOException
	 * @exclude
	 */
	public InputByteStream getStream()
		throws PDFIOException
	{
		try {
			return mBuf.slice();
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 * An internal-use-only version that avoids slicing
	 */
	InputByteStream getStreamRaw()
	{
		return mBuf;
	}

	public int getNumObjects()
	{
		return mNumObjects;
	}

	int getNumObjectsInternal()
	{
		return mObjectInfos.size();
	}

	public CosOpenOptions getOptions()
	{
		return mOptions;
	}
	
	/**
	 * Returns true iff the document is currently linearized.
	 *
	 * To say that a document is "linearized" in PDF means it has been put into the special order
	 * and has NOT been modified since then.
	 */
	public boolean isLinearized()
	{
		return mIsLinearized;
	}

	/**
	 * Returns true iff the BASE document was a linearized document,
	 * but later suffered incremental updates that spoil its perfect order.
	 *
	 * So if isLinearized() returns false but wasLinearized() returns true you are
	 * dealing with a document that was linear saved and has not been full saved since
	 * then but that HAS had one or more incremental update sections applied to it.
	 */
	public boolean wasLinearized()
	{
		return mWasLinearized;
	}

	public boolean wasRepaired()
	{
		return ! repairTypes.isEmpty() || documentCosLevelRepaired;
	}
	
	public EnumSet<REPAIRTYPE> getRepairTypes()
	{
		return repairTypes;
	}
	
	/**
	 *  this is the garbage which is found just after last EOF marker of document.
	 * @param garbageLength
	 */
	void setGarbageLength(long garbageLength)
	{
		this.garbageLength = garbageLength;
	}

	boolean forceCompress()
	{
		return mForceCompress;
	}

	XRefTable getXRef()
	{
		return mXRef;
	}

	CosObjectInfo getIndexedInfo(int index)
	{
		return (CosObjectInfo)mObjectInfos.get(index);
	}

	void putIndexedInfo(int index, CosObjectInfo info)
	{
		mObjectInfos.add(index, info);
	}

	void putRebuiltInfo(int index, CosObjectInfo info)
	{
		if (index >= mNumObjects)
			mNumObjects = index + 1;
		CosObjectInfo oldInfo = getIndexedInfo(index);
		if (oldInfo == null) {
			putIndexedInfo(index, info);
		} else {
			if (!(oldInfo.isLoaded() || oldInfo.isDirty())) {
				oldInfo.markAddressed();
				oldInfo.setObjGen(info.getObjGen());
				oldInfo.setPos(info.getPos());
			}
		}
	}

	/**
	 * 
	 *
	 * Assign a new ObjectInfo on this document or get the one
	 * we previously assigned for this object number.
	 * @return assigned CosObjectInfo.
	 */
	CosObjectInfo getObjectInfo(int objNum, int objGen)
	{
		if (objNum == 0 && objGen != 65535)
			return null;
		CosObjectInfo info = getIndexedInfo(objNum);
		if (objGen < 0)
			return info;
		if (info != null) {
			if (objNum != info.getObjNum() || objGen != info.getObjGen())
				info = null;
		} else {
			info = new CosObjectInfo(this, objNum, objGen);
			putIndexedInfo(objNum, info);
		}
		return info;
	}

	/**
	 * 
	 *
	 * Assign a new ObjectInfo on this document or get the one
	 * we previously assigned for this object info.
	 * @return assigned CosObjectInfo.
	 */
	CosObjectRef getObjectRef(CosObjectInfo info)
	{
		CosObjectRef ref = info.getRef();
		if (ref == null) {
			ref = new CosObjectRef(this, info);
			info.setRef(ref);
		}
		return ref;
	}

	/**
	 * 
	 *
	 * Generate a new object info for an indirect object that will be in the
	 * new xref table.
	 */
	CosObjectInfo newObjectInfo()
	{
		return getObjectInfo(mNumObjects++, 0);
	}

	/**
	 * Sets pointer to FDFDocument
	 *
	 * @param fdfDocument	FDFDocument to set
	 */
	public void setFDFDocument(Object fdfDocument)
	{
		mUserData[1] = fdfDocument;
	}
	
	/**
	 * Returns pointer to FDFDocument
	 * @return Object
	 */
	public Object getFdfDocument()
	{
		return mUserData[1];
	}
	
	/**
	 * Sets pointer to PDFDocument
	 *
	 * @param pdfdocument	PDFDocument to set
	 */
	public void setPDFDocument(Object pdfdocument)
	{
		mUserData[0] = pdfdocument;
	}
	
	/**
	 * Gets pointer to PDFDocument
	 * @return Object
	 */
	public Object getPdfDocument()
	{
		return mUserData[0];
	}

	/**
	 * @author gish
	 *
	 * Return the CosObject associated with a CosObjectRef
	 *
	 * @param ref - CosObjectRef associated with an indirect object
	 * @return - CosObject associated with the reference
	 * @throws PDFCosParseException
	 * @throws IOException
	 * @throws PDFSecurityException
	 * @throws PDFIOException
	 */
	CosObject resolveReference(CosObjectRef ref)
		throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		CosObjectInfo refInfo = ref.getInfo();
		CosObject result = refInfo.getObject();
		if (result == null)
			result = getIndirectObject(refInfo);
		if (result == null)
			result = createCosNull();
		return result;
	}

	/**
	 * @author gish
	 * This creates the header string for a file being saved. The PDF version
	 * must be supplied in order for the document to be saved. The version can
	 * be supplied explicitly or be derived from the Cos objects.
	 *
	 * @param version String representation of PDF version. If this value is null,
	 * the version will be retrieved from either the pre-existing header (if present)
	 * or the Version entry in the catalog.
	 */
	void buildHeaderString(String version)
	{
		// Create the extra comment line (PDF 1.5 Section 3.4.1)
		StringBuilder result = new StringBuilder("%PDF-");
		result.append(version);
		result.append("\r%");
		result.append('\342').append('\343').append('\317').append('\323').append("\r\n");
		if (mHeaderToken != null)
			result.append('%').append(mHeaderToken).append("\r\n");
		int vindx = result.indexOf("1.");
		if (vindx > 0)
			mHeader = (result.toString());
	}
	/**
	 * Parse the document to identify and remove duplicate resources
	 * referenced in the page resource object trees.</P>
	 *
	 * <P><b>Note</b>: This method must be <i>explicitly</i> invoked by the client prior
	 * to save. It should only be invoked before full save
	 * and linear save.</P>
	 * @throws PDFSecurityException
	 * @throws PDFIOException
	 * @throws PDFCosParseException
	 */
	public void freeDuplicateResources()
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosPDFOptimizer pdfOptimizer = CosPDFOptimizer.newInstance(this);
		try {
			pdfOptimizer.freeDuplicateResources();
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 * Explicit early cleanup of unreferenced objects as done in full save.
	 * returns true if at least one object is freed.
	 */
	public boolean freeUnreferencedObjects()
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosContainer[] roots = new CosContainer[]{getRoot(), getInfo(), getEncryptionDictionary(), getTrailer()};
		try {
			return CosOptimizer.freeUnreferencedObjects(this, roots, false);
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 * Return an array of unreferenced indirect objects in this document.
	 */
	public CosObject[] getUnreferencedObjects()
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosContainer[] roots = new CosContainer[]{getRoot(), getInfo(), getEncryptionDictionary(), getTrailer()};
		try {
			return CosOptimizer.getUnreferencedObjects(this, roots);
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 * 
	 *
	 * Perform an save with previously specified save style
	 *
	 * @param byteWriter for writing the PDF document to
	 * @param params indicating save style
	 * @throws PDFCosParseException
	 * @throws PDFInvalidParameterException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public void save(ByteWriter byteWriter, CosSaveParams params)
		throws PDFCosParseException, PDFInvalidParameterException, PDFIOException, PDFSecurityException
	{
		OutputByteStream outputByteStream = null;
		try {
			long timer = 0;
			if (mLoggingPath != null)
				timer = System.currentTimeMillis();
			long base = 0;
			String style = null;
			CosList saveObjectInfos = null;
			mCompObjInfos = new CosList();
			if (byteWriter == mByteReader)
				byteWriter = null;
			params = normalizeParams(params);
			int saveStyle = params.getSaveStyle();
			if (mSaveToCopy || mIsFDF)
				saveObjectInfos = cloneObjectInfos();
			if (byteWriter != null)
				outputByteStream = getStreamManager().getOutputByteStream(byteWriter);
			if (saveStyle == CosSaveParams.SAVE_STYLE_INCREMENTAL) {
				if (byteWriter == null) {
					byteWriter = getInPlaceByteWriter();
					if (byteWriter == null)
						throw new PDFInvalidParameterException("In-place incremental save requires ByteWriter as source ByteReader");
					if (mBuf != null && mLoggingPath != null)
						base = mBuf.length();
					style = "In-place incremental save";
					outputByteStream = getStreamManager().getOutputByteStream(byteWriter);
				} else {
					style = "Incremental save";
				}
				if (repairTypes.contains(REPAIRTYPE.xrefRepair))
					throw new PDFInvalidParameterException("File cannot be saved incrementally if xref in original PDF stream was damaged.");
				incrementalSave(outputByteStream, params, byteWriter);
			} else if (saveStyle == CosSaveParams.SAVE_STYLE_FULL) {
				style = "Full save";
				fullSave(outputByteStream, params);
			} else if (saveStyle == CosSaveParams.SAVE_STYLE_LINEAR) {
				linearSave(outputByteStream, params);
				if (params.getTempByteWriter() != null)
					style = "Linear save with temp file";
				else
					style = "Linear save without temp file";
			} else {
				style = "Full save";
				fullSave(outputByteStream, params);
			}
			if (!(params.getCloseAfterSave() || mSaveToCopy || mIsFDF)) {
				CosList objStmInfos = getObjStmInfos();
				Iterator iter = objStmInfos.iterator();
				while (iter.hasNext()) {
					CosObjectInfo stmInfo = (CosObjectInfo)iter.next();
					CosObjectStream objStm = (CosObjectStream)stmInfo.getObject();
					objStm.update();
				}
				outputByteStream.flush();
				postSaveCleanup(outputByteStream, byteWriter);
				getEncryption().setDecryptionAsEncryption();
			}
			if (mSaveToCopy || mIsFDF) {
				restoreObjectInfos(saveObjectInfos);
				CosList objStmInfos = getObjStmInfos();
				Iterator iter = objStmInfos.iterator();
				while (iter.hasNext()) {
					CosObjectInfo stmInfo = (CosObjectInfo)iter.next();
					CosObjectStream objStm = (CosObjectStream)stmInfo.getObject();
					objStm.update();
				}
			}
			if (mLoggingPath != null) {
				timer = System.currentTimeMillis() - timer;
				StringBuilder styleBuilder = new StringBuilder(style);
				styleBuilder.append(" took ").append(timer).append("ms");
				if (mBuf != null)
					styleBuilder.append(", ").append(mBuf.length() - base).append(" bytes");
				LogString(styleBuilder.toString());
			}
		} catch (IOException e) {
			throw new PDFIOException(e);
		} finally {
			mIsLinearized = (params.getSaveStyle() == CosSaveParams.SAVE_STYLE_LINEAR) ? true : false;
			if (params.getSaveStyle() != CosSaveParams.SAVE_STYLE_INCREMENTAL)
				mWasLinearized = mIsLinearized;
			mCompObjInfos = null;
			params.setTempByteWriter(null);
			if (params.getCloseAfterSave()) {
				close();
			}
			if (params.getCloseAfterSave() || mSaveToCopy) {
				if (outputByteStream != null) {
					try {
						outputByteStream.close();
					} catch (IOException e) {
						throw new PDFIOException(e);
					}
				}
			}
		}
	}

	static void LogString(String logItem)
	{
		if (mLoggingPath != null) {
			RandomAccessFile logFile = null;
			try {
				logFile = new RandomAccessFile(new File(mLoggingPath + "CosDocument.log"), "rw");
				logFile.seek(logFile.length());
				logFile.writeBytes(logItem + "\r\n");
			} catch (Exception e) {
				throw new RuntimeException("Cannot make logger entry", e);
			}
			finally
			{
				if(logFile != null)
					try {
						logFile.close();
					} catch (IOException e) {
						throw new RuntimeException("Error closing log file", e);
					}
			}
		}
	}

	/**
	 * Clone the master CosObjectInfo list for SaveToCopy
	 */
	private CosList cloneObjectInfos()
	{
		CosList cloneList = new CosList();
		int end = mObjectInfos.size();
		for (int i = 0; i < end; i++) {
			CosObjectInfo info = (CosObjectInfo)mObjectInfos.get(i);
			if (info != null) {
				CosObjectInfo clone = new CosObjectInfo(this, 0, 0);
				clone.copyValuesFrom(info);
				cloneList.add(i, clone);
			}
		}
		return cloneList;
	}

	/**
	 * Restore the previously cloned master CosObjectInfo
	 */
	private void restoreObjectInfos(CosList savedInfos)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		int end = savedInfos.size();
		while (mObjectInfos.size() > end)
			mObjectInfos.delete(mObjectInfos.size() - 1);
		for (int i = 0; i < end; i++) {
			CosObjectInfo info = (CosObjectInfo)mObjectInfos.get(i);
			if (info != null) {
				CosObjectInfo clone = (CosObjectInfo)savedInfos.get(i);
				info.copyValuesFrom(clone);
			}
		}
		mNumObjects = end;
		mRoot = null;
	}

	/**
	 * Get an object info from the document object stream list
	 */
	CosList getObjStmInfos()
	{
		CosList objStmInfos = new CosList();
		Iterator iter = mObjectInfos.iterator();
		while (iter.hasNext()) {
			CosObjectInfo info = (CosObjectInfo)iter.next();
			if (info.isObjStm() && !info.isFree())
				objStmInfos.add(info);
		}
		return objStmInfos;
	}

	/**
	 * See if a full save object ref shold reset the generation to zero
	 */
	boolean willBeCompressed(int objNum)
	{
		if (mCompObjInfos == null)
			return false;
		return mCompObjInfos.containsIndex(objNum);
	}

	/**
	 * 
	 */
	public static void setLoggingPath(String loggingPath)
	{
		mLoggingPath = loggingPath;
	}

	/**
	 * 
	 */
	public static String getLoggingPath()
	{
		return mLoggingPath;
	}

	public ByteWriter getInPlaceByteWriter()
	{
		if (mByteReader instanceof ByteWriter && mOptions.getSaveInPlace())
			return (ByteWriter)mByteReader;
		return null;
	}

	/**
	 * 
	 * @return {@link ByteReader}
	 */
	public ByteReader getByteReader(){
		return this.mByteReader;
	}
	
	/**
	 * Sets the offset where next incremental section shall be written if
	 * the document is saved incrementally.
	 * @param nextIncrementalSectionOffset
	 */
	public void setNextIncrementalSectionOffset(long nextIncrementalSectionOffset) {
		this.nextIncrementalSectionOffset = nextIncrementalSectionOffset;
	}
	
	/**
	 * This method is called when the document is being saved incrementally but the new incremental section
	 * is not to be written at the end of document but after one of the already existing sections.
	 * @param outputByteStream
	 * @param inPlaceSave
	 * @throws PDFCosParseException
	 * @throws PDFSecurityException
	 * @throws PDFIOException
	 */
	private void updateDocumentStream(OutputByteStream outputByteStream, boolean inPlaceSave) throws PDFCosParseException, PDFSecurityException, PDFIOException{
		try{
			if(inPlaceSave){// document being saved in place. We need to replace the content of the stream 
							// after the specified offset with whitespace characters. If we don't do this then there may remain some
							// data un-overwritten after save operation causing invalid document. 
							// On the other hand, If we wait till the save operation completes, and then do this activity on remaining un-overwritten content,
							// then also it will be treated as garbage after EOF.
				outputByteStream.seek(nextIncrementalSectionOffset);
			}else// copy the document stream upto the offset specified. New incremental section shall be written right after this.
				IO.copy(mBuf, 0, nextIncrementalSectionOffset, outputByteStream);
			InputByteStream mBufCopy = mBuf.slice(0, nextIncrementalSectionOffset);// cut down the document stream.
			mBuf.close();
			mBuf = mBufCopy;
			mXRef.resetXRef(mBuf);// reload the xrefs.
			nextIncrementalSectionOffset = -1;
		}catch(IOException e){
			throw new PDFIOException(e);
		}
	}
	/**
	 * Perform an incremental save. This will use the xref style of the input
	 * document.
	 * @throws PDFCosParseException
	 * @throws PDFInvalidParameterException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	private void incrementalSave(OutputByteStream outputByteStream, CosSaveParams params, ByteWriter byteWriter)
		throws PDFCosParseException, PDFInvalidParameterException, IOException, PDFIOException, PDFSecurityException
	{
		long curpos;
		if (mBuf == null)
			throw new PDFInvalidParameterException("Incremental save can be used with existing files only");
		
		if(documentCosLevelRepaired)
		{
			// While incremental save no changes from repair list are to be picked from 
			// repair list, so resetting repair list and related flag.
			documentCosLevelRepaired = false;
			if(cosRepairList != null)
				cosRepairList.clear(); // clear the repair list.
		}
		int xrefstyle = params.getXrefStyle();
		
		// If the document is encrypted, setup encryption before we write
		getEncryption().setupEncryption();
		if (byteWriter != mByteReader) {
			// Copy the input file to the output stream
			mBuf.seek(0);	// copy from the beginning of the file
			if(nextIncrementalSectionOffset != -1){
				updateDocumentStream(outputByteStream, false);
			}
			else{
				IO.copy(mBuf, 0, mBuf.length()-garbageLength, outputByteStream);
			}
		} else {
			// Seek to end of pre-existing file
			if(nextIncrementalSectionOffset != -1){
				updateDocumentStream(outputByteStream, true);				
			}else
				outputByteStream.seek(outputByteStream.length()-garbageLength);
		}
		/// writing a carriage return(10) and line feed(13) characters after EOF marker if there was
		/// some garbage just after EOF which has been removed now.
		if(garbageLength > 0){
			outputByteStream.write(13);
			outputByteStream.write(10);
		}

		// If the doc has an ID, update the ID in the original trailer and
		// if it is not there generate a new one
		CosArray updateID = createUpdateDocID(false);
		getTrailer().put(ASName.k_ID, updateID);

		// Remove orphaned objects by marking all unreferenced dirty objects free.
		CosOptimizer.freeUnreferencedObjectsIncremental(this, mOrigNumObjects);

		// If there are any dirty compressed objects, make sure that the
		// objects are written out to the object's data stream
		CosList dirtyCompObjInfos = buildCompObjList(xrefstyle);
		Iterator iter = dirtyCompObjInfos.iterator();
		while (iter.hasNext()) {
			CosList stmObjList = (CosList)iter.next();
			CosObjectStream newStm = createCosObjectStream();
			newStm.setObjectStreamList(stmObjList);
		}

		// Then process dirty uncompressed objects including the
		// object streams we may have created above
		CosList dirtyObjectInfos = buildObjectList(true);
		if (dirtyObjectInfos.isEmpty())
			return;
		CosList stripped = null;
		CosList updateInfos = null;
		curpos = outputByteStream.getPosition();
		if (xrefstyle == CosSaveParams.XREFHYBRID) {
			updateInfos = prepareHybridInfoList(dirtyObjectInfos, null, null);
			stripped = dirtyObjectInfos;
		} else {
			stripped = dirtyObjectInfos.copy();
			iter = stripped.iterator();
			while (iter.hasNext()) {
				CosObjectInfo info = (CosObjectInfo)iter.next();
				int objNum = info.getObjNum();
				if (info.isCompressed() || mCompObjInfos.containsIndex(objNum))
					iter.remove();
			}
		}
		curpos = writeIndirectObjects(stripped, outputByteStream, params.getSaveToCopy());
		long paddedDataLength = outputByteStream.length() - outputByteStream.getPosition();
		if(paddedDataLength > 0){// This is the case when document is being saved in place. We need to replace the content of the stream 
								// after here with whitespace characters. If we don't do this then there may remain some
								// data un-overwritten after save operation causing invalid document. 
								// On the other hand, If we wait till the save operation completes, and then do this activity on remaining un-overwritten content,
								// then also it will be treated as garbage after EOF.
			byte[] spacePadding = new byte[(int) paddedDataLength];
			Arrays.fill(spacePadding, (byte)' ');
			outputByteStream.write(spacePadding);
			curpos = outputByteStream.getPosition();
		}
		long xrefPos = curpos;

		// Write the xref
		long lastXrefPos = 0;
		ArrayList sections = prepareXRef(dirtyObjectInfos, false);
		if (xrefstyle == CosSaveParams.XREFTABLE) {
			writeXRefTable(sections, outputByteStream);

			// Create the trailer and copy over the old trailer values
			CosDictionary trailer = createCosDictionary(CosObject.DIRECT);
			copyTrailerFields(trailer, getNumObjects());

			// Put a /Prev entry into the new trailer
			lastXrefPos = mXRef.getLastXRefSectionPosition();
			trailer.put(ASName.k_Prev, lastXrefPos);
			writeTrailer(trailer, outputByteStream);
			if (!mSaveToCopy)
				mTrailer = trailer;
			writeEOF(xrefPos, outputByteStream);
		} else if (xrefstyle == CosSaveParams.XREFSTREAM) {
			buildXRefStream(sections, outputByteStream, null);
			try {
				outputByteStream.write(StringOps.toByteArray(STARTXREF));
				outputByteStream.write(StringOps.toByteArray(Long.toString(xrefPos)));
				outputByteStream.write(StringOps.toByteArray(EOF));
			} catch (IOException e) {
				throw new PDFIOException("Unable to write the XRef.", e);
			}
		} else if (xrefstyle == CosSaveParams.XREFHYBRID) {
			writeXRefTable(sections, outputByteStream);

			// Create the trailer and copy over the old trailer values
			CosDictionary trailer = createCosDictionary(CosObject.DIRECT);
			copyTrailerFields(trailer, getNumObjects());

			// Put a /Prev entry into the new trailer
			lastXrefPos = mXRef.getLastXRefSectionPosition();
			trailer.put(ASName.k_Prev, lastXrefPos);
			writeTrailer(trailer, outputByteStream);

			// Write the update objects - if there are none, you're done
			if (updateInfos.isEmpty()) {
				writeEOF(xrefPos, outputByteStream);
				return;
			}

			// Get the xref stream object (and add the /Prev entry before writing it)
			int xindx = updateInfos.size() - 1;
			CosObjectInfo xinfo = (CosObjectInfo)updateInfos.get(xindx);
			CosStream xrefstm = (CosStream)xinfo.getObject();
			xrefstm.put(ASName.k_Prev, xrefPos);
			stripped = updateInfos.copy();
			iter = stripped.iterator();
			while (iter.hasNext()) {
				CosObjectInfo info = (CosObjectInfo)iter.next();
				if (info.isCompressed() || info == xinfo)
					iter.remove();
			}

			// Write the indirect objects in the update section (but not the xref stream)
			long updatexrefpos = writeIndirectObjects(stripped, outputByteStream, params.getSaveToCopy());

			// Now, create XRef sections if the compressed object numbers aren't consecutive
			// Scan the array list, adding entries to a holding array until you reach
			// a break. Then, start a new holding array list.
			ArrayList hybridSections = new ArrayList();
			int sectionindx = 0;
			hybridSections.add(sectionindx, new CosList());
			int lastObjNum = 0;
			int thisObjNum;
			CosObjectInfo info;
			CosList seclist = (CosList)hybridSections.get(sectionindx);
			iter = updateInfos.iterator();
			while (iter.hasNext()) {
				info = (CosObjectInfo)iter.next();
				thisObjNum = info.getObjNum();
				if (thisObjNum - lastObjNum != 1 && lastObjNum != 0) {
					seclist = new CosList();
					hybridSections.add(++sectionindx, seclist);
				}
				seclist.add(info);
				lastObjNum = thisObjNum;
			}

			// Calculate the /Index value for the xref stream
			CosArray indexVal = createCosArray();
			Iterator seciter = hybridSections.iterator();
			CosObjectInfo firstObjInfo;
			int firstObjNum, secsize;
			while (seciter.hasNext()) {
				CosList nextsec = (CosList)seciter.next();
				firstObjInfo = (CosObjectInfo)nextsec.get(0);
				firstObjNum = firstObjInfo.getObjNum();
				secsize = nextsec.size();

				// Add the pair of the first obj number and section size to the array
				indexVal.addInt(firstObjNum);
				indexVal.addInt(secsize);
			}
			xrefstm.put(ASName.k_Index, indexVal);

			// Write the xref stream entries to the object's data stream
			int[] widths = setWidthsArray(updateInfos, xrefstm, outputByteStream.getPosition());
			OutputByteStream buf = getStreamManager().getOutputByteStreamClearDocument(Fixed.GROWABLE, -1);
			Iterator iterator = updateInfos.iterator();
			while (iterator.hasNext()) {
				info = (CosObjectInfo)iterator.next();
				info.writeXRefStreamEntry(buf, widths);
			}
			xrefstm.newDataDecoded(buf.closeAndConvert());

			// Write the xrefstream object to the file
			try {
				updatexrefpos = xinfo.writeIndirectObject(outputByteStream, updatexrefpos);
				outputByteStream.write(StringOps.toByteArray(XREF));
			} catch (PDFIOException e) {
				throw new PDFIOException("Unable to write the XRef stream.", e);
			}

			// Now write the header info for the empty xref section
			try {
				outputByteStream.write('0');
				outputByteStream.write(' ');
				outputByteStream.write('0');
				outputByteStream.write(' ');
				outputByteStream.write('\n');
			} catch (IOException e) {
				throw new PDFIOException("Unable to write the header info.", e);
			}

			// Get the byte offset for the xref stream (last element on the list)
			long xstmPos = xinfo.getPos();

			// Create the update trailer and write it
			CosDictionary updateTrailer = createCosDictionary(CosObject.DIRECT);
			copyTrailerFields(updateTrailer, getNumObjects());
			updateTrailer.put(ASName.k_Prev, xrefPos);
			updateTrailer.put(ASName.k_XRefStm, xstmPos);
			writeTrailer(updateTrailer, outputByteStream);
			if (!mSaveToCopy)
				mTrailer = updateTrailer;
			writeEOF(updatexrefpos, outputByteStream);
		}
	}

	/**
	 * Perform a full save using custom parameter settings. Set parameters
	 * by creating a CosSaveParams instance.
	 *
	 * CosSaveParams.XREFTABLE is specified and compressed objects are present
	 * encryption parameter is provided
	 *
	 * @param outputByteStream output for the document
	 * @param params CosSaveParams instance containing parameter settings
	 * @throws PDFCosParseException
	 * @throws PDFInvalidParameterException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 * @see com.adobe.internal.pdftoolkit.core.cos.CosSaveParams
	 */
	private void fullSave(OutputByteStream outputByteStream, CosSaveParams params)
		throws PDFCosParseException, PDFInvalidParameterException, IOException, PDFIOException, PDFSecurityException
	{
		// Get the header and write it
		try {
			outputByteStream.write(StringOps.toByteArray(mHeader));
		} catch (IOException e) {
			throw new PDFIOException("Unable to write the header.", e);
		}
		if (mIsFDF)
			getRoot().remove(ASName.k_Type);

		// Update the document's ID and add it to the old trailer
		CosArray updatedID = createUpdateDocID(false);
		getTrailer().put(ASName.k_ID, updatedID);
		int xrefstyle = params.getXrefStyle();

		// If the document needs encryption, setup encryption before we write
		getEncryption().setupEncryption();

		// Remove orphaned objects by marking all unreferenced dirty objects free.
		CosContainer[] roots = new CosContainer[]{getRoot(), getInfo(), getEncryptionDictionary(), null};
		long lastXrefPos = 0;
		if (!mXRef.isNew()) {
			if (mXRef.getType() == XRefTable.tHybrid) {
				CosDictionary trailer = getTrailer();
				if (trailer.containsKey(ASName.k_XRefStm))
					lastXrefPos = trailer.getInt(ASName.k_XRefStm);
			} else if (mXRef.getType() == XRefTable.tStream) {
				lastXrefPos = mXRef.getLastXRefSectionPosition();
			}
		}
		if (lastXrefPos == 0)
			roots[3] = getTrailer();
		CosOptimizer.freeUnreferencedObjects(this, roots, true);

		// Build the new compressed object list
		mCompObjInfos = buildFullSaveCompList(xrefstyle);
		CosList objStmInfos = getObjStmInfos();
		Iterator iter = objStmInfos.iterator();
		Iterator compIter = mCompObjInfos.iterator();
		CosObjectStream curObjStm = null;
		int curIndex = OBJSTM_MAXNUMOBJS;
		while (compIter.hasNext()) {
			CosObjectInfo info = (CosObjectInfo)compIter.next();
			CosObject obj = info.getObject();
			if (curIndex == OBJSTM_MAXNUMOBJS) {
				curIndex = 0;
				if (iter != null && iter.hasNext()) {
					CosObjectInfo stmInfo = (CosObjectInfo)iter.next();
					curObjStm = (CosObjectStream)stmInfo.getObject();
					curObjStm.remove(ASName.k_Extends);
				} else {
					iter = null;
					curObjStm = createCosObjectStream();
				}
			}
			curObjStm.addObjectToStream(obj);
			curIndex++;
		}
		CosList objStmsToBeDeleted = new CosList();
		if (iter != null) {
			while (iter.hasNext()) {
				CosObjectInfo stmInfo = (CosObjectInfo)iter.next();
				objStmsToBeDeleted.add(stmInfo.getObjNum(), stmInfo);
			}
		}

		// Then process uncompressed objects including the object streams we may have collected above
		CosList stripped = null;
		CosList updateInfos = null;
		CosList objectInfos = buildObjectList(false);
		if (xrefstyle == CosSaveParams.XREFHYBRID) {
			updateInfos = prepareHybridInfoList(objectInfos, mCompObjInfos, objStmsToBeDeleted);
			stripped = objectInfos;
		} else {
			stripped = objectInfos.copy();
			iter = stripped.iterator();
			while (iter.hasNext()) {
				CosObjectInfo info = (CosObjectInfo)iter.next();
				int objNum = info.getObjNum();
				if (mCompObjInfos.containsIndex(objNum) || objStmsToBeDeleted.containsIndex(objNum))
					iter.remove();
			}
		}
		long xrefPos;
		xrefPos = writeIndirectObjects(stripped, outputByteStream, params.getSaveToCopy());

		// If document was encrypted, create a new encryption dictionary
		// and write it out
		CosDictionary encryptDict = null;
		if (needsEncryption()) {
			Map encryptData = getEncryption().getEncryptionMap();
			if (encryptData == null)
				throw new PDFInvalidParameterException("New encryption data not provided.");
			encryptDict = createCosDictionaryFromNonCosData(encryptData);

			// Mark all the dict's CosString and CosStream values not to be encrypted for write
			encryptDict.setEncryptionState(false);
			CosObjectInfo encryptInfo = encryptDict.getInfo();
			if (encryptInfo != null) {
				xrefPos = encryptInfo.writeIndirectObject(outputByteStream, xrefPos);
				objectInfos.add(encryptInfo.getObjNum(), encryptInfo);
			}
		}
		if (objStmsToBeDeleted.isEmpty()) {
			stripped = objectInfos;
		} else {
			stripped = objectInfos.copy();
			iter = stripped.iterator();
			while (iter.hasNext()) {
				CosObjectInfo info = (CosObjectInfo)iter.next();
				int objNum = info.getObjNum();
				if (objStmsToBeDeleted.containsIndex(objNum))
					iter.remove();
			}
		}
		ArrayList sections = prepareXRef(stripped, true);

		// Modify the encryption entry in the old trailer by either creating
		// a new encryption dictionary or removing the old one
		if (!needsEncryption()) {
			// Remove encryption dictionary from the old trailer
			removeEncryptionDictionary();
		} else {
			// Replace encryption dictionary in the old trailer
			setEncryptionDictionary(encryptDict);
		}

		// Write the appropriate xref and new trailer for the doc
		if (xrefstyle == CosSaveParams.XREFTABLE) {
			// FDF only makes sense on old style files
			if (!mIsFDF)
				writeXRefTable(sections, outputByteStream);

			// Create the trailer and copy over the old trailer values
			CosDictionary trailer = createCosDictionary(CosObject.DIRECT);
			copyTrailerFields(trailer, getNumObjects());
			writeTrailer(trailer, outputByteStream);
			if (!mSaveToCopy)
				mTrailer = trailer;
			if (mIsFDF)
				outputByteStream.write(StringOps.toByteArray("%%EOF\n"));
			else
				writeEOF(xrefPos, outputByteStream);
		} else if (xrefstyle == CosSaveParams.XREFSTREAM) {
			buildXRefStream(sections, outputByteStream, objStmsToBeDeleted);
			try {
				outputByteStream.write(StringOps.toByteArray(STARTXREF));
				outputByteStream.write(StringOps.toByteArray(Long.toString(xrefPos)));
				outputByteStream.write(StringOps.toByteArray(EOF));
			} catch (IOException e) {
				throw new PDFIOException("Unable to write XRef.", e);
			}
		} else if (xrefstyle == CosSaveParams.XREFHYBRID) {
			writeXRefTable(sections, outputByteStream);

			// Create the trailer and copy over the old trailer values
			CosDictionary trailer = createCosDictionary(CosObject.DIRECT);
			copyTrailerFields(trailer, getNumObjects());
			writeTrailer(trailer, outputByteStream);

			// Write the update objects - if there are none, you're done
			if (updateInfos.isEmpty()) {
				iter = objStmsToBeDeleted.iterator();
				while (iter.hasNext()) {
					CosObjectInfo stmInfo = (CosObjectInfo)iter.next();
					stmInfo.markFree();
				}
				writeEOF(xrefPos, outputByteStream);
				return;
			}

			// Get the xref stream object (and add the /Prev entry before writing it)
			int xindx = updateInfos.size() - 1;
			CosObjectInfo xinfo = (CosObjectInfo)updateInfos.get(xindx);
			CosStream xrefstm = (CosStream)xinfo.getObject();
			xrefstm.put(ASName.k_Prev, xrefPos);
			stripped = updateInfos.copy();
			iter = stripped.iterator();
			while (iter.hasNext()) {
				CosObjectInfo info = (CosObjectInfo)iter.next();
				if (!info.isObjStm())
					iter.remove();
			}

			// Write the indirect objects in the update section (but not the xref stream)
			long updatexrefpos = writeIndirectObjects(stripped, outputByteStream, params.getSaveToCopy());

			// Now, create XRef sections if the compressed object numbers aren't consecutive
			// Scan the array list, adding entries to a holding array until you reach
			// a break. Then, start a new holding array list.
			ArrayList hybridSections = new ArrayList();
			int sectionindx = 0;
			hybridSections.add(sectionindx, new CosList());
			int lastObjNum = 0;
			int thisObjNum;
			CosObjectInfo info;
			CosList seclist = (CosList)hybridSections.get(sectionindx);
			iter = updateInfos.iterator();
			while (iter.hasNext()) {
				info = (CosObjectInfo)iter.next();
				thisObjNum = info.getObjNum();
				if (thisObjNum - lastObjNum != 1 && lastObjNum != 0) {
					seclist = new CosList();
					hybridSections.add(++sectionindx, seclist);
				}
				seclist.add(info);
				lastObjNum = thisObjNum;
			}

			// Calculate the /Index value for the xref stream
			CosArray indexVal = createCosArray();
			Iterator seciter = hybridSections.iterator();
			CosObjectInfo firstObjInfo;
			int firstObjNum, secsize;
			while (seciter.hasNext()) {
				CosList nextsec = (CosList)seciter.next();
				firstObjInfo = (CosObjectInfo)nextsec.get(0);
				firstObjNum = firstObjInfo.getObjNum();
				secsize = nextsec.size();

				// Add the pair of the first obj number and section size to the array
				indexVal.addInt(firstObjNum);
				indexVal.addInt(secsize);
			}
			xrefstm.put(ASName.k_Index, indexVal);

			// Write the xref stream entries to the object's data stream
			xrefstm.getInfo().setPos(outputByteStream.getPosition());
			int[] widths = setWidthsArray(updateInfos, xrefstm, outputByteStream.getPosition());
			OutputByteStream buf = getStreamManager().getOutputByteStreamClearDocument(Fixed.GROWABLE, -1);
			Iterator iterator = updateInfos.iterator();
			while (iterator.hasNext()) {
				info = (CosObjectInfo)iterator.next();
				info.writeXRefStreamEntry(buf, widths);
			}
			xrefstm.newDataDecoded(buf.closeAndConvert());

			// Write the xrefstream object to the file
			try {
				updatexrefpos = xinfo.writeIndirectObject(outputByteStream, updatexrefpos);
				outputByteStream.write(StringOps.toByteArray(XREF));
			} catch (PDFIOException e) {
				throw new PDFIOException("Unable to write the XRef Stream.", e);
			}

			// Now write the header info for the empty xref section
			try {
				outputByteStream.write('0');
				outputByteStream.write(' ');
				outputByteStream.write('0');
				outputByteStream.write(' ');
				outputByteStream.write('\n');
			} catch (IOException e) {
				throw new PDFIOException("Unable to write the empty XRef section.", e);
			}

			// Get the byte offset for the xref stream (last element on the list)
			long xstmPos = xinfo.getPos();

			// Create the update trailer and write it
			CosDictionary updateTrailer = createCosDictionary(CosObject.DIRECT);
			copyTrailerFields(updateTrailer, getNumObjects());
			updateTrailer.put(ASName.k_Prev, xrefPos);
			updateTrailer.put(ASName.k_XRefStm, xstmPos);
			writeTrailer(updateTrailer, outputByteStream);
			if (!mSaveToCopy)
				mTrailer = updateTrailer;
			writeEOF(updatexrefpos, outputByteStream);
		}
		iter = objStmsToBeDeleted.iterator();
		while (iter.hasNext()) {
			CosObjectInfo stmInfo = (CosObjectInfo)iter.next();
			stmInfo.markFree();
		}
	}

	/**
	 * Figure out what will go into object streams for full save
	 * @throws IOException
	 */
	private CosList buildFullSaveCompList(int xrefStyle)
		throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		CosList compObjList = new CosList();
		if (xrefStyle == CosSaveParams.XREFHYBRID) {
			CosObject obj = getRoot().get(ASName.k_StructTreeRoot);
			if (obj != null) {
				compObjList = new CosList();
				int stateStackPtr = 0;
				Object[] stateStack = new Object[65536];
				CosContainer container = null;
				Iterator containerIter = null;
				while (true) {
					while (true) {
						if (obj == null || obj instanceof CosNull || obj instanceof CosStream)
							break;
						int objNum = obj.getObjNum();
						if (objNum != 0) {
							if (compObjList.get(objNum) != null)
								break;
							compObjList.add(objNum, obj.getInfo());
						}
						if (!(obj instanceof CosContainer))
							break;
						if (containerIter != null) {
							stateStack[stateStackPtr++] = container;
							stateStack[stateStackPtr++] = containerIter;
						}
						container = (CosContainer)obj;
						if (container instanceof CosArray) {
							containerIter = ((CosArray)container).iterator();
						} else {
							List<ASName> keys = ((CosDictionary)container).getKeys();
							containerIter = keys.iterator();
						}
						break;
					}
					if (containerIter == null)
						return compObjList;
					while (!containerIter.hasNext()) {
						if (stateStackPtr == 0)
							return compObjList;
						containerIter = (Iterator)stateStack[--stateStackPtr];
						container = (CosContainer)stateStack[--stateStackPtr];
					}
					if (container instanceof CosArray) {
						obj = (CosObject)containerIter.next();
					} else {
						ASName key = (ASName)containerIter.next();
						obj = null;
						if (key == ASName.k_Pg || key == ASName.k_P || key == ASName.k_Stm || key == ASName.k_StmOwn || key == ASName.k_Obj)
							continue;
						obj = ((CosDictionary)container).get(key);
					}
				}
			}
		} else if (xrefStyle == CosSaveParams.XREFSTREAM) {
			compObjList = new CosList();
			CosDictionary catalog = getRoot();
			CosDictionary encrypt = getEncryptionDictionary();
			Iterator iter = mObjectInfos.iterator();
			while (iter.hasNext()) {
				CosObjectInfo info = (CosObjectInfo)iter.next();
				CosObject obj = info.getObject();
				if (obj == null || obj instanceof CosStream)
					continue;
				if (obj == catalog || obj == encrypt)
					continue;
				if (obj instanceof CosDictionary)
				{
					CosObject type = ((CosDictionary)obj).get(ASName.k_Type);
					if(type instanceof CosName && (dictionariesNotToBeCompressed.contains(type.nameValue())))
					{
						continue;
					}
				}	
				compObjList.add(info.getObjNum(), info);
			}
		}
		return compObjList;
	}

	/**
	 * Perform a linear save using custom parameter settings. Set parameters
	 * by creating a CosSaveParams instance.
	 *
	 * CosSaveParams.XREFTABLE is specified and compressed objects are present
	 * encryption parameter is provided
	 *
	 * @param outputByteStream output destination for the save
	 * @param params - CosSaveParams instance containing parameter settings
	 * @throws PDFCosParseException
	 * @throws PDFInvalidParameterException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 * @see com.adobe.internal.pdftoolkit.core.cos.CosSaveParams
	 */
	private void linearSave(OutputByteStream outputByteStream, CosSaveParams params)
		throws PDFCosParseException, PDFInvalidParameterException, IOException, PDFIOException, PDFSecurityException
	{
		// Linearization constants
		final int HEADER_OVERHEAD_TABLE = 400;
		final int HEADER_OVERHEAD_STREAM = 500;
		final int HEADER_OVERHEAD_HYBRID = 500;
		
		// Update the document's ID and add it to the old trailer
		CosArray updatedID = createUpdateDocID(false);
		getTrailer().put(ASName.k_ID, updatedID);

		// Now we know the style, do the first part of linearization
		int xrefstyle = params.getXrefStyle();
		mCosLin = new CosLinearization(this);
		mCosLin.buildLinearizationData(xrefstyle);

		// If the document needs encryption, setup encryption before we write but after
		// we registered the fact that it is linearized.
		getEncryption().setupEncryption();

		// Do the main section first
		// We may need a main section XRef stream
		CosStream mainXRefStm = null;
		CosObjectInfo mainXRefStmObjInfo = null;

		// Build new lists of CosObjectInfos, renumbering as we go
		CosList mainObjInfos = mCosLin.getMainSectionList();
		CosList mainObjInfosCompressed = mCosLin.getMainSectionCompressedList();
		CosList updateObjInfos = mCosLin.getUpdateSectionList();
		CosList updateObjInfosCompressed = mCosLin.getUpdateSectionCompressedList();
		if (xrefstyle == CosSaveParams.XREFHYBRID && mainObjInfosCompressed.isEmpty())
			xrefstyle = CosSaveParams.XREFTABLE;
		boolean haveXRefStms = (xrefstyle == CosSaveParams.XREFSTREAM || xrefstyle == CosSaveParams.XREFHYBRID);
		CosList oldToOldObjStmMap = null;
		if (mXRef.getType() == XRefTable.tStream || mXRef.getType() == XRefTable.tHybrid) {
			oldToOldObjStmMap = new CosList();
			Iterator iter = mObjectInfos.iterator();
			while (iter.hasNext()) {
				CosObjectInfo objInfo = (CosObjectInfo)iter.next();
				CosObjectInfo objStmInfo = objInfo.getStreamInfo();
				if (objStmInfo != null)
					oldToOldObjStmMap.add(objStmInfo.getObjNum(), objStmInfo.getObject());
			}
		}

		// Renumber main section objects first
		CosList newInfosList = new CosList();
		newInfosList.addAll(mainObjInfos);
		if (haveXRefStms) {
			newInfosList.addAll(mainObjInfosCompressed);
			if (xrefstyle == CosSaveParams.XREFSTREAM) {
				mainXRefStm = createCosStream();
				mainXRefStmObjInfo = mainXRefStm.getInfo();
				newInfosList.add(mainXRefStmObjInfo);
			}
		}

		// Now the update section
		// Create the first "special" object, the linearization dictionary
		CosDictionary linDict = createCosDictionary();
		CosObjectInfo linDictObjInfo = linDict.getInfo();

		// Second possible "special" object, the update section XRef stream
		CosStream updateXRefStm = null;
		CosObjectInfo updateXRefStmObjInfo = null;
		if (haveXRefStms) {
			updateXRefStm = createCosStream();
			updateXRefStmObjInfo = updateXRefStm.getInfo();
		}

		// The third "special" object, the hint stream
		CosStream hintStm = createCosStream();
		CosObjectInfo hintStmObjInfo = hintStm.getInfo();

		// The forth "special" object, the encryption dictionary
		CosDictionary encryptDict = null;
		if (needsEncryption()) {
			Map encryptData = getEncryption().getEncryptionMap();
			if (encryptData == null)
				throw new PDFInvalidParameterException("New encryption data not provided.");
			encryptDict = createCosDictionaryFromNonCosData(encryptData);

			// Mark all the dict's CosString and CosStream values not to be encrypted for write
			encryptDict.setEncryptionState(false);
			setEncryptionDictionary(encryptDict);
		} else {
			removeEncryptionDictionary();
		}

		// Figure count of special objects, always have linDict and hintStm
		int numSpecialObjs = 2;
		if (updateXRefStm != null)
			numSpecialObjs++;
		if (encryptDict != null)
			numSpecialObjs++;

		// Now the update section, magic objects first
		newInfosList.add(linDictObjInfo);
		if (updateXRefStm != null)
			newInfosList.add(updateXRefStmObjInfo);
		updateObjInfos.add(0, (encryptDict == null) ? null: encryptDict.getInfo());
		newInfosList.addAll(updateObjInfos);
		newInfosList.addAll(updateObjInfosCompressed);
		newInfosList.add(hintStmObjInfo);

		// All writing must be done AFTER objects have been renumbered
		CosListInt newToOldObjNumMap = new CosListInt();
		CosListInt newToOldObjGenMap = new CosListInt();
		CosListInt oldToNewObjNumMap = new CosListInt();
		mNumObjects = newInfosList.size();
		
		// This stores mapping of old object numbers to new object numbers after reordering (only for objects in repair list) 
		HashMap<Integer, Integer> objectNumbersMap = new HashMap<Integer, Integer>(); 
		for (int i = 1; i < mNumObjects; i++) {
			CosObjectInfo objInfo = (CosObjectInfo)newInfosList.get(i);
			newToOldObjNumMap.add(i, objInfo.getObjNum());
			newToOldObjGenMap.add(i, objInfo.getObjGen());
			oldToNewObjNumMap.add(objInfo.getObjNum(), i);
			if(documentCosLevelRepaired && cosRepairList.isObjectRepaired(objInfo.getObjNum()))
			{
				objectNumbersMap.put(objInfo.getObjNum(), i);
			}
			
			objInfo.setObjNum(i);
			objInfo.setObjGen(0);
		}
		// Update the repair list with new object numbers
		if(documentCosLevelRepaired && ! objectNumbersMap.isEmpty())
			cosRepairList.updateObjectNumbers(objectNumbersMap);
		
		
		mCosLin.setNewToOldObjNumMap(newToOldObjNumMap);
		mCosLin.setNewToOldObjGenMap(newToOldObjGenMap);
		mCosLin.setOldToOldObjStmMap(oldToOldObjStmMap);
		mCosLin.setOldToNewObjNumMap(oldToNewObjNumMap);
		mObjectInfos = newInfosList;

		// Create a temp file stream for outputting objects and collecting positions
		OutputByteStream tempSaveOutStream = null;
		ByteWriter tempByteWriter = params.getTempByteWriter();
		if (tempByteWriter == null)
			tempSaveOutStream = getStreamManager().getOutputByteStreamClearDocument(Fixed.GROWABLE, -1);
		else
			tempSaveOutStream = getStreamManager().getOutputByteStream(tempByteWriter);

		// Start writing out objects
		long curPos;
		curPos = writeIndirectObjects(updateObjInfos, tempSaveOutStream, params.getSaveToCopy());
		int updateLastObjNum = ((CosObjectInfo)updateObjInfos.last()).getObjNum();
		curPos = writeIndirectObjects(mainObjInfos, tempSaveOutStream, params.getSaveToCopy());
		int mainLastObjNum = ((CosObjectInfo)mainObjInfos.last()).getObjNum();
		long objEndPos = curPos;

		// Figure out where the hint stream will have to start
		long hintStmPos;
		if (xrefstyle == CosSaveParams.XREFTABLE)
			hintStmPos = HEADER_OVERHEAD_TABLE + ((updateObjInfos.size() + numSpecialObjs) * 20);
		else if (xrefstyle == CosSaveParams.XREFSTREAM) {
			hintStmPos = HEADER_OVERHEAD_STREAM + ((updateObjInfos.size() + numSpecialObjs) * 4);
			hintStmPos += updateObjInfosCompressed.size() * 4;
		} else {
			hintStmPos = HEADER_OVERHEAD_HYBRID + ((updateObjInfos.size() + numSpecialObjs) * 20);
			hintStmPos += mainObjInfosCompressed.size() * 4;
		}

		// Now build the hint stream
		long hintStmLen = 0;
		mCosLin.buildHintStream(mObjectInfos, hintStm, mainLastObjNum, updateLastObjNum, hintStmPos, objEndPos);
		OutputByteStream hintStmOutBuf = getStreamManager().getOutputByteStreamClearDocument(Fixed.GROWABLE, -1);
		hintStmLen = hintStmObjInfo.writeIndirectObject(hintStmOutBuf, 0);

		// Adjust write positions
		Iterator linIter = updateObjInfos.iterator();
		while (linIter.hasNext()) {
			CosObjectInfo objInfo = (CosObjectInfo)linIter.next();
			objInfo.setPos(objInfo.getPos() + hintStmPos + hintStmLen);
		}
		linIter = mainObjInfos.iterator();
		while (linIter.hasNext()) {
			CosObjectInfo objInfo = (CosObjectInfo)linIter.next();
			objInfo.setPos(objInfo.getPos() + hintStmPos + hintStmLen);
		}

		// Create a counted stream for writing the output "file".
		// We can't back this stream up, so we only write to it once
		// when we're sure what we're writing.
		outputByteStream.write(StringOps.toByteArray(mHeader));

		// Build the linearization dictionary, we should have all the parts
		long linDictPos = 0;
		long updateXRefPos = 0;
		linDictPos = outputByteStream.getPosition();
		linDict.put(ASName.k_Linearized, 1);
		linDict.put(ASName.k_L, 1234567890);
		CosArray hintValues = createCosArray();
		hintValues.addInt((int)hintStmPos);
		hintValues.addInt((int)hintStmLen);
		linDict.put(ASName.k_H, hintValues);
		int specialPageObjNum = mCosLin.getSpecialPageObj().getObjNum();
		linDict.put(ASName.k_O, specialPageObjNum);
		int specialPageEnd = (int)((CosObjectInfo)mainObjInfos.get(1)).getPos();
		linDict.put(ASName.k_E, specialPageEnd);
		linDict.put(ASName.k_N, mCosLin.getNumPages());
		linDict.put(ASName.k_T, 1234567890);
		OutputByteStream scratchStm = new NullOutputByteStream();
		updateXRefPos = linDictPos + linDictObjInfo.writeIndirectObject(scratchStm, linDictPos);

		// Write the main xref section
		long linDictT = 0;
		long mainXRefPos = 0;
		CosObjectInfo freeInfo = getObjectInfo(0, 65535);
		mainXRefPos = hintStmPos + hintStmLen + curPos;
		if (xrefstyle == CosSaveParams.XREFTABLE) {
			linDictT = linXRefTable(tempSaveOutStream, mainObjInfos);
			CosDictionary trailer = createCosDictionary(CosObject.DIRECT);
			trailer.put(ASName.k_Size, mainObjInfos.size());
			trailer.put(ASName.k_ID, updatedID);
			if (encryptDict != null)
				trailer.put(ASName.k_Encrypt, encryptDict);
			writeTrailer(trailer, tempSaveOutStream);
			writeEOF(updateXRefPos, tempSaveOutStream);
		} else if (xrefstyle == CosSaveParams.XREFSTREAM) {
			mainObjInfosCompressed.add(mainXRefStmObjInfo);
			linIter = mainObjInfosCompressed.iterator();
			while (linIter.hasNext()) {
				CosObjectInfo info = (CosObjectInfo)linIter.next();
				mainObjInfos.add(info.getObjNum(), info);
			}
			linDictT = linXRefStream(tempSaveOutStream, mainXRefStmObjInfo, mainObjInfos, mainXRefPos, encryptDict, null, null, updatedID, 0, false);
			tempSaveOutStream.write(StringOps.toByteArray(STARTXREF));
			tempSaveOutStream.write(StringOps.toByteArray(Long.toString(updateXRefPos)));
			tempSaveOutStream.write(StringOps.toByteArray(EOF));
		} else {
			int freeCount = mainObjInfosCompressed.size();
			while (freeCount-- != 0)
				mainObjInfos.add(freeInfo);
			linDictT = linXRefTable(tempSaveOutStream, mainObjInfos);
			CosDictionary trailer = createCosDictionary(CosObject.DIRECT);
			trailer.put(ASName.k_Size, mainObjInfos.size());
			trailer.put(ASName.k_ID, updatedID);
			if (encryptDict != null)
				trailer.put(ASName.k_Encrypt, encryptDict);
			writeTrailer(trailer, tempSaveOutStream);
			writeEOF(updateXRefPos, tempSaveOutStream);
		}

		// Fix up the linearization dictionary and write it out
		linDict.put(ASName.k_L, hintStmPos + hintStmLen + tempSaveOutStream.getPosition());
		linDict.put(ASName.k_T, hintStmPos + hintStmLen + linDictT);
		curPos = linDictObjInfo.writeIndirectObject(outputByteStream, linDictPos);
		while (curPos < updateXRefPos - 1) {
			outputByteStream.write(' ');
			curPos++;
		}
		outputByteStream.write('\n');

		// Write the update xref section
		updateObjInfos = copyInfoList(updateObjInfos);
		updateObjInfos.add(linDictObjInfo.getObjNum(), linDictObjInfo);
		linDictObjInfo.setPos(linDictPos);
		hintStmObjInfo.setPos(hintStmPos);
		mainObjInfosCompressed = copyInfoList(mainObjInfosCompressed);
		if (xrefstyle == CosSaveParams.XREFTABLE) {
			updateObjInfos.add(hintStmObjInfo.getObjNum(), hintStmObjInfo);
			CosDictionary trailer = createCosDictionary(CosObject.DIRECT);
			copyTrailerFields(trailer, mNumObjects);
			trailer.put(ASName.k_Prev, mainXRefPos);
			linXRefTable(outputByteStream, updateObjInfos);
			writeTrailer(trailer, outputByteStream);
			mTrailer = trailer;
			writeEOF(0, outputByteStream);
		} else if (xrefstyle == CosSaveParams.XREFSTREAM) {
			updateObjInfos.add(updateXRefStmObjInfo.getObjNum(), updateXRefStmObjInfo);
			updateXRefStmObjInfo.setPos(updateXRefPos);
			updateObjInfos.addAll(updateObjInfosCompressed);
			updateObjInfos.add(hintStmObjInfo.getObjNum(), hintStmObjInfo);
			linXRefStream(outputByteStream, updateXRefStmObjInfo, updateObjInfos, updateXRefPos, encryptDict, getRoot(), getInfo(), updatedID, mainXRefPos, true);
		} else {
			updateObjInfos.add(updateXRefStmObjInfo.getObjNum(), updateXRefStmObjInfo);
			updateObjInfos.add(hintStmObjInfo.getObjNum(), hintStmObjInfo);
			CosDictionary trailer = createCosDictionary(CosObject.DIRECT);
			copyTrailerFields(trailer, mNumObjects);
			trailer.put(ASName.k_Prev, mainXRefPos);
			scratchStm.seek(0);	// reset the scratch stream
			linXRefTable(scratchStm, updateObjInfos);
			curPos = scratchStm.getPosition();
			trailer.put(ASName.k_XRefStm, updateXRefPos + curPos + HEADER_OVERHEAD_HYBRID);
			writeTrailer(trailer, scratchStm);
			writeEOF(0, scratchStm);
			curPos = scratchStm.getPosition();
			trailer.put(ASName.k_XRefStm, updateXRefPos + curPos);
			updateXRefStmObjInfo.setPos(updateXRefPos + curPos);
			linXRefTable(outputByteStream, updateObjInfos);
			writeTrailer(trailer, outputByteStream);
			mTrailer = trailer;
			writeEOF(0, outputByteStream);
			if (outputByteStream.getPosition() < updateXRefPos + curPos)
				outputByteStream.write('\n');
			linXRefStream(outputByteStream, updateXRefStmObjInfo, mainObjInfosCompressed, updateXRefPos + curPos, encryptDict, null, null, null, 0, false);
		}

		// Fill the ugly gap with spaces
		curPos = outputByteStream.getPosition();
		while (curPos < hintStmPos - 1) {
			outputByteStream.write(' ');
			curPos++;
		}
		outputByteStream.write('\n');

		// Now the saved hint stream
		InputByteStream hintStmInBuf = null;
		try {
			// First convert the OBS into an IBS
			hintStmInBuf = hintStmOutBuf.closeAndConvert();
			hintStmOutBuf = null;
			IO.copy(hintStmInBuf, outputByteStream);
		} finally {
			if (hintStmInBuf != null)
				hintStmInBuf.close();
		}

		// And finaly the body of the file
		InputByteStream tempSaveInStream = tempSaveOutStream.closeAndConvert();
		tempSaveOutStream = null;
		IO.copy(tempSaveInStream, outputByteStream);
		tempSaveInStream.close();
		tempSaveInStream = null;
	}

	/**
	 * Simplified XRef table builder particular to the linearizer.
	 * The main simplification is that there is exactly ONE section,
	 * since the object numbers in the CosObjectInfo list that is
	 * passed in will be contiguous.
	 * @throws IOException
	 */
	private long linXRefTable(OutputByteStream os, CosList objInfos)
		throws IOException
	{
		long retVal = 0;
		os.write(StringOps.toByteArray(XREF));
		CosObjectInfo objInfo;
		int first = 0;
		int size = objInfos.size();
		if (size != 0) {
			objInfo = (CosObjectInfo)objInfos.first();
			first = objInfo.getObjNum();
			size -= first;
		}
		os.write(StringOps.toByteArray(Integer.toString(first)));
		os.write(' ');
		os.write(StringOps.toByteArray(Integer.toString(size)));
		retVal = os.getPosition();
		os.write('\n');
		Iterator iter = objInfos.iterator();
		while (iter.hasNext()) {
			objInfo = (CosObjectInfo)iter.next();
			objInfo.writeXRefTableEntry(os);
		}
		return retVal;
	}

	/**
	 * Simplified XRef stream builder particular to the linearizer.
	 * The main simplification is that there is exactly ONE section,
	 * since the object numbers in the CosObjectInfo list that is
	 * passed in will be contiguous. The XRef stream object will
	 * have been previously created.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	private long linXRefStream(OutputByteStream os, CosObjectInfo xrStmObjInfo,
				   CosList objInfos, long curPos, CosDictionary encrypt, CosDictionary root,
				   CosDictionary info, CosArray id, long prev, boolean setTrailer)
		throws PDFCosParseException, IOException, PDFIOException, PDFSecurityException
	{
		long retVal = os.getPosition() - 1;
		xrStmObjInfo.setPos(curPos);
		CosStream xrefStm = (CosStream)xrStmObjInfo.getObject();
		CosObjectInfo objInfo;
		int first = 0;
		int size = objInfos.size();
		if (size != 0) {
			objInfo = (CosObjectInfo)objInfos.first();
			first = objInfo.getObjNum();
			size -= first;
		}
		xrefStm.put(ASName.k_Type, ASName.k_XRef);
		xrefStm.put(ASName.k_Size, first + size);
		xrefStm.put(ASName.k_Filter, ASName.k_FlateDecode);
		CosArray iValues = createCosArray();
		iValues.addInt(first);
		iValues.addInt(size);
		xrefStm.put(ASName.k_Index, iValues);
		if (encrypt != null)
			xrefStm.put(ASName.k_Encrypt, encrypt);
		if (root != null)
			xrefStm.put(ASName.k_Root, root);
		if (info != null)
			xrefStm.put(ASName.k_Info, info);
		if (id != null)
			xrefStm.put(ASName.k_ID, id);
		if (prev != 0)
			xrefStm.put(ASName.k_Prev, prev);
		int [] widths = setWidthsArray(objInfos, xrefStm, curPos);
		OutputByteStream buf = getStreamManager().getOutputByteStreamClearDocument(Fixed.GROWABLE, -1);
		Iterator iter = objInfos.iterator();
		while (iter.hasNext()) {
			objInfo = (CosObjectInfo)iter.next();
			objInfo.writeXRefStreamEntry(buf, widths);
		}
		xrefStm.newDataDecoded(buf.closeAndConvert());
		buf = null;
		xrStmObjInfo.writeIndirectObject(os, curPos);
		if (setTrailer)
			mTrailer = xrefStm;
		return retVal;
	}

	/**
	 * 
	 */
	public boolean isCacheEnabled()
	{
		return mCacheEnabled;
	}

	/**
	 * @author gish
	 *
	 * Utility method to write out a set of indirect objects
	 *
	 * @param infos - CosList of CosObjectInfos representing objects to write
	 * @param os - CountingOutputByteStream to write to
	 * @return - long current position (byte offset from beginning) in the output stream
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	long writeIndirectObjects(CosList infos, OutputByteStream os, boolean saveToCopy)
		throws PDFCosParseException, IOException, PDFIOException, PDFSecurityException
	{
		long curpos = os.getPosition();
		if (infos.isEmpty())
			return curpos;
		Iterator iter = infos.iterator();
		while (iter.hasNext()) {
			CosObjectInfo info = (CosObjectInfo)iter.next();
			if (info.isDirty()) {
				CosObject obj = info.getObject();
				if (obj instanceof CosObjectStream)
					((CosObjectStream)obj).writeObjectsToStream();
			}
			curpos = info.writeIndirectObject(os, curpos, saveToCopy);
		}
		return curpos;
	}

	/**
	 * Utility to copy a CosList that is in fact a list of CosObjectInfos.
	 * Used to go from an unordered list to an ordered list or when wee need
	 * to renumber objects, like during linearization.
	 */
	CosList copyInfoList(CosList src)
	{
		CosList dest = new CosList();
		Iterator iter = src.iterator();
		while (iter.hasNext()) {
			CosObjectInfo info = (CosObjectInfo)iter.next();
			dest.add(info.getObjNum(), info);
		}
		return dest;
	}

	/**
	 * Build XRef sections list
	 *
	 * @param infos - CosList of CosObjectInfos of all of the dirty objects
	 * @param full - boolean to pad free entries (full save) or make sections (incremental)
	 * @return ArrayList of xref sections
	 */
	ArrayList prepareXRef(CosList infos, boolean full) {
		// Now, create XRef sections if the object numbers aren't consecutive
		// Scan the array list, adding entries to a holding array until you reach
		// a break. Then, start a new holding array list.
		ArrayList sections = new ArrayList();
		CosList secList = new CosList();
		sections.add(secList);
		CosObjectInfo freeInfo = getObjectInfo(0, 65535);
		if (infos.isEmpty()) {
			secList.add(freeInfo);
		} else {
			int nextObjNum;
			int infoCounter = 0;
			Iterator infoIter = infos.iterator();
			nextObjNum = ((CosObjectInfo)infos.iterator().next()).getObjNum();
			if (full) {
				while (infoCounter < nextObjNum) {
					secList.add(freeInfo);
					infoCounter++;
				}
			} else {
				infoCounter = nextObjNum;
			}
			while (infoIter.hasNext()) {
				CosObjectInfo nextInfo = (CosObjectInfo)infoIter.next();
				nextObjNum = nextInfo.getObjNum();
				if (infoCounter < nextObjNum) {
					if (full) {
						while (infoCounter < nextObjNum) {
							secList.add(freeInfo);
							infoCounter++;
						}
					} else {
						secList = new CosList();
						sections.add(secList);
						infoCounter = nextObjNum;
					}
				}
				secList.add(nextInfo);
				infoCounter++;
			}
		}
		return sections;
	}

	/**
	 * @author gish
	 *
	 * This writes a new xref table for a full save from the objects on
	 * the document's dirty list.
	 *
	 * @param sections - ArrayList of sections of CosObjectInfos of all of the dirty objects
	 * @param os - CountingOutputByteStream to which document save is being written
	 * @return long position after write (offset from 0) in the CountingOutputByteStream
	 * @throws IOException
	 */
	long writeXRefTable(ArrayList sections, OutputByteStream os)
		throws IOException
	{
		// For each section list, get the object count
		// and the number of the first object and write that first. Then, write
		// an entry for each object in the section.
		os.write(StringOps.toByteArray(XREF));
		Iterator secitr = sections.iterator();
		while (secitr.hasNext()) {
			CosList nextsec = (CosList)secitr.next();
			CosObjectInfo firstObjInfo = (CosObjectInfo)nextsec.get(0);
			int firstObjNum = firstObjInfo.getObjNum();

			// Now write the header info for the section
			os.write(StringOps.toByteArray(Integer.toString(firstObjNum)));
			os.write(' ');
			os.write(StringOps.toByteArray(Integer.toString(nextsec.size())));
			os.write(' ');
			os.write('\n');

			// Now write the xref entries for all the objects in the section
			Iterator secobjitr = nextsec.iterator();
			while (secobjitr.hasNext()) {
				CosObjectInfo secinfo = (CosObjectInfo)secobjitr.next();
				secinfo.writeXRefTableEntry(os);
			}
		}
		return os.getPosition();
	}

	long buildXRefStream(ArrayList sections, OutputByteStream os, CosList objStmsToBeDeleted)
		throws PDFCosParseException, IOException, PDFIOException, PDFSecurityException
	{
		// First, cache the current position since we'll write the
		// xref stream object here
		long xrefpos = os.getPosition();

		// Now create a CosStream instance and initialize the dictionary
		CosStream xrefStream = createCosStream();
		xrefStream.put(ASName.k_Type, ASName.k_XRef);
		
		// Put the trailer info into the dictionary part
		copyTrailerFields(xrefStream, getNumObjects());

		// If an incremental save, put the last Xref Stream pos in as /Prev
		if (objStmsToBeDeleted == null)
			xrefStream.put(ASName.k_Prev, mXRef.getLastXRefSectionPosition());
		CosArray indexVal = createCosArray();
		Iterator seciter = sections.iterator();
		CosObjectInfo firstObjInfo;
		int firstObjNum, secsize;
		while (seciter.hasNext()) {
			CosList nextsec = (CosList)seciter.next();
			firstObjInfo = (CosObjectInfo)nextsec.get(0);
			firstObjNum = firstObjInfo.getObjNum();
			secsize = nextsec.size();

			// Add the pair of the first obj number and section size to the array
			indexVal.addInt(firstObjNum);
			if (!seciter.hasNext()) {	// If this is the last pair
				int thisObjNum = xrefStream.getObjNum();
				if (thisObjNum == firstObjNum + secsize) {
					secsize++;
				} else {
					if (objStmsToBeDeleted != null) {
						CosObjectInfo freeInfo = getObjectInfo(0, 65535);
						while (thisObjNum > firstObjNum + secsize++)
							nextsec.add(freeInfo);
					} else {
						indexVal.addInt(secsize);
						indexVal.addInt(thisObjNum);
						secsize = 1;
					}
				}
			}
			indexVal.addInt(secsize);
		}
		xrefStream.put(ASName.k_Index, indexVal);

		// W array
		int [] widths = setWidthsArray(sections, xrefStream, xrefpos);

		// Set up the stream filter
		xrefStream.put(ASName.k_Filter, ASName.k_FlateDecode);

		// Now write all the entries to an output byte buffer
		OutputByteStream buf = getStreamManager().getOutputByteStreamClearDocument(Fixed.GROWABLE, -1);
		Iterator secitr = sections.iterator();
		while (secitr.hasNext()) {
			CosList nextsec = (CosList)secitr.next();

			// Now write the xref entries for all the objects in the section
			Iterator secobjitr = nextsec.iterator();
			while (secobjitr.hasNext()) {
				CosObjectInfo secinfo = (CosObjectInfo)secobjitr.next();
				if (objStmsToBeDeleted != null && objStmsToBeDeleted.containsIndex(secinfo.getObjNum()))
					secinfo = getObjectInfo(0, 65535);	// Use free entry for to-be-deleted object streams
				secinfo.writeXRefStreamEntry(buf, widths);
			}
		}

		// Add the xrefstream to the dirty list and get its info
		// Then, write its entry in the byte buffer
		CosObjectInfo xrefStreamInfo = xrefStream.getInfo();
		xrefStreamInfo.markDirty();
		xrefStreamInfo.setPos(xrefpos);
		xrefStreamInfo.writeXRefStreamEntry(buf, widths);

		// Add the byte buffer as the xref data stream
		xrefStream.newDataDecoded(buf.closeAndConvert());
		buf = null;

		// Write the xref stream object into the document
		xrefStreamInfo.writeIndirectObject(os, xrefpos);
		if (!mSaveToCopy)
			mTrailer = xrefStream;
		return os.getPosition();
	}

	/**
	 * @author gish
	 *
	 * Extract all the compressed objects and compressed object streams. Add the
	 * compressed object stream to the list so that they can be written in the update.
	 * Create an xref stream for the extracted objects and add it to the update list
	 * plus create a free entry for it on the dirty object list.
	 *
	 * @param dirtyObjectInfos - the objectInfos on the dirty list
	 * @return CosList of compressed objects and compressed object streams to be
	 * written out in the update
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 * @throws IOException
	 */
	CosList prepareHybridInfoList(CosList dirtyObjectInfos, CosList compressedList, CosList objStmsToBeDeleted)
		throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		CosList hybridInfos = new CosList();

		// Extract all object streams and compressed objects to another list
		Iterator objIter = dirtyObjectInfos.iterator();
		CosObject origObj;
		CosObjectInfo origInfo;
		while (objIter.hasNext()) {
			origInfo = (CosObjectInfo)objIter.next();
			int objNum = origInfo.getObjNum();
			if (objStmsToBeDeleted != null && objStmsToBeDeleted.containsIndex(objNum)) {
				objIter.remove();
				continue;
			}
			origObj = origInfo.getObject();
			if (origObj != null) {
				// Only for instantiated CosObjects
				boolean compressed;
				if (compressedList != null)
					compressed = compressedList.containsIndex(objNum);
				else
					compressed = origInfo.isCompressed();
				if (compressed || origObj instanceof CosObjectStream) {
					hybridInfos.add(objNum, origInfo);
					objIter.remove();
				}
			}
		}

		// Build the xref stream object for the hybridInfos
		CosStream xref = createCosStream();
		CosObjectInfo xinfo = xref.getInfo();

		// Create the stream dictionary entries
		xref.put(ASName.k_Type, ASName.k_XRef);
		xref.put(ASName.k_Size, hybridInfos.size() + 1);	//Add the xref stream

		// Set up the stream filter
		xref.put(ASName.k_Filter, ASName.k_FlateDecode);

		// Add the xref stream to the result and add a free entry for it to the
		// dirty list
		hybridInfos.add(xinfo.getObjNum(), xinfo);
		return hybridInfos;
	}

	/**
	 * Writes the first part of the PDF file trailer. The format of the
	 * trailer is defined in section 3.4.4 of the PDF Reference Manual
	 * version 1.4.
	 *
	 * @param trailer		Trailer dictionary
	 * @param os			CountingOutputByteStream to write to
	 * @throws PDFIOException
	 *
	 */
	void writeTrailer(CosDictionary trailer, OutputByteStream os)
		throws PDFCosParseException, PDFIOException, IOException, PDFSecurityException
	{
		os.write(StringOps.toByteArray("trailer\n"));
		trailer.writeOut(os);
		os.write('\n');
	}

	/**
	 * Write the last part of the PDF file trailer. The format of the
	 * trailer is define in section 3.4.4 of the PDF Reference Manual
	 * version 1.4.
	 *
	 * @param xrefPos		Location of last cross reference table
	 * @param os			CountingOutputByteStream stream to write to
	 * @throws IOException
	 *
	 */
	void writeEOF(long xrefPos, OutputByteStream os)
		throws IOException
	{
		os.write(StringOps.toByteArray(STARTXREF));
		os.write(StringOps.toByteArray(Long.toString(xrefPos)));
		os.write(StringOps.toByteArray(EOF));
	}

	// Encryption state methods

	/**
	 * 
	 */
	public boolean isEncrypted()
	{
		CosDictionary trailer = getTrailer();

		// Guard against NULL documents
		return (trailer != null) ? trailer.containsKey(ASName.k_Encrypt) : false;
	}

	/**
	 * 
	 */
	public CosEncryption getEncryption()
	{
		return mEncryption;
	}

	CosDictionary getEncryptionDictionary()
	{
		return mEncryption.getEncryption();
	}

	void setEncryptionDictionary(CosDictionary encrypt)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		if (encrypt != null) {
			int savedState = 0;
			CosDictionary trailer = getTrailer();
			CosObjectInfo trailerInfo = trailer.getInfo();
			if (trailerInfo != null)
				savedState = trailerInfo.getState();
			trailer.put(ASName.k_Encrypt, encrypt);
			if (trailerInfo != null)
				trailerInfo.setState(savedState);
		}
	}

	void removeEncryptionDictionary()
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		int savedState = 0;
		CosDictionary trailer = getTrailer();
		CosObjectInfo trailerInfo = trailer.getInfo();
		if (trailerInfo != null)
			savedState = trailerInfo.getState();
		trailer.remove(ASName.k_Encrypt);
		if (trailerInfo != null)
			trailerInfo.setState(savedState);
	}

	boolean needsEncryption()
	{
		return getEncryption().needsEncryption();
	}

	protected CosLinearization getLinearization()
	{
		return mCosLin;
	}

	/**
	 * 
	 *
	 * Return an indirect CosObject instance corresponding to an object number. If this
	 * CosDocument instance is not already caching the CosObject, use the XRef
	 * table (if there are entries) to locate the object in the input stream and parse
	 * it. Cache it for future references.
	 *
	 * @param objNum	Indirect object number of COS object whose byte location is desired
	 * @return CosObject parsed result. Returns null if object doesn't exist or if
	 * the object entry in the XRef is marked free.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosObject getIndirectObjectByNumber(int objNum)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			if (objNum > mNumObjects)
				return null;
			CosObjectInfo indexedInfo = getIndexedInfo(objNum);
			CosObjectInfo info = indexedInfo;
			if (info == null)
				info = mXRef.getInfo(objNum);
			else if (info.isFree())
				return null;
			if (info == null)
				return null;
			if (indexedInfo == null)
				putIndexedInfo(objNum, info);
			CosObject result = info.getObject();
			if (result == null) {
				try {
					result = mXRef.getIndirectObject(info);
				} catch (PDFCosParseException e) {
					if (mOptions.getLateRepairEnabled()) {
						if (! repairTypes.contains(REPAIRTYPE.xrefRepair)) {
							mXRef.rebuildLate();
							repairTypes.add(REPAIRTYPE.xrefRepair);
							try {
								result = mXRef.getIndirectObject(info);
							} catch (PDFCosParseException ex) {
							}
						}
					} else {
						throw new PDFCosParseException("XRef repair required but not enabled", e);
					}
				}
			}
			return result;
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 * 
	 *
	 * Return an indirect CosObjectInfo instance corresponding to an object number.
	 * If this CosDocument instance is not already caching the CosObjectInfo, use the XRef
	 * table (if there are entries) to look it up and cache it for future references.
	 */
	public CosObjectInfo getIndirectObjectInfoByNumber(int objNum)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			if (objNum > mNumObjects)
				return null;
			CosObjectInfo indexedInfo = getIndexedInfo(objNum);
			CosObjectInfo info = indexedInfo;
			if (info == null)
				info = mXRef.getInfo(objNum);
			else if (info.isFree())
				return null;
			if (info == null)
				return null;
			if (indexedInfo == null)
				putIndexedInfo(objNum, info);
			return info;
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 * 
	 *
	 * Return an indirect CosObject instance corresponding to a CosObjectInfo. If this
	 * CosDocument instance is not already caching the CosObject, use the XRef
	 * table (if there are entries) to locate the object in the input stream and parse
	 * it. Cache it for future references.
	 *
	 * @param info - CosObjectInfo of the indirect object
	 * @return CosObject parsed result. Returns null if object doesn't exist or if
	 * the object entry in the XRef is marked free.
	 * @throws PDFCosParseException
	 * @throws IOException
	 * @throws PDFSecurityException
	 * @throws PDFIOException
	 */
	CosObject getIndirectObject(CosObjectInfo info)
		throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		if (info == null)
			return null;
		CosObject result = info.getObject();
		if (result == null) {
			try {
				result = mXRef.getIndirectObject(info);
			} catch (PDFCosParseException e) {
				if(e.hasErrorType(CosParseErrorType.NumberParseError))
					throw e;//we can't repair if number parsing errors occur like
							// integer is out of range etc. So propogating the exception to caller.
				if (mOptions.getLateRepairEnabled()) {
					if (! repairTypes.contains(REPAIRTYPE.xrefRepair)) {
						repairTypes.add(REPAIRTYPE.xrefRepair);
						mXRef.rebuildLate();
						try {
							result = mXRef.getIndirectObject(info);
						} catch (PDFCosParseException ex) {
						}
					}
				} else {
					throw new PDFCosParseException("XRef repair required but not enabled", e);
				}
			}
		}
		if (result == null)
			result = createCosNull();
		return result;
	}

	/**
	 * Return EOF of object's update section or zero if cannot be determined.
	 * 
	 * @exclude
	 */
	public long getObjEOF(CosObject obj)
	{
		return mXRef.getObjEOF(obj.getInfo());
	}

	/**
	 * Return index of object's update section or -1 if cannot be determined.
	 * 
	 * @exclude
	 */
	public int getObjRevision(CosObject obj)
	{
		return mXRef.getObjRevision(obj.getInfo());
	}

	/**
	 * Return the count of trash bytes that appear before the document header.
	 * 
	 * @exclude
	 */
	public int getHeaderTrashCount()
	{
		return mHeaderTrashCount;
	}

	/**
	 * Return the count of trash bytes that appear after the document trailer.
	 * 
	 * @exclude
	 */
	public int getTrailerTrashCount()
	{
		return mTrailerTrashCount;
	}

	/**
	 * Get total number of revisions for this document.
	 * 
	 * @exclude
	 */
	public int getNumRevisions()
	{
		return mXRef.getNumRevisions();
	}

	/**
	 * 
	 * @exclude
	 * Get the file length for this document. This is the length of the underlying buffer
	 * after header and trailer trash have been removed.
	 * @return the length of the underly buffer in bytes
	 * @throws PDFIOException
	 */
	public long getFileSize()
		throws PDFIOException
	{
		try {
			return this.mBuf.length();
		} catch (IOException e) {
			throw new PDFIOException("Error with stream underlying document.", e);
		}
	}

	/**
	 * Return CosLIst of objects added, deleted, or changed since EOF value.
	 * 
	 * @exclude
	 */
	public CosList getChangedObjects(long eof)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosList dirtyObjectInfos = buildObjectList(true);
		CosList modList = new CosList();
		Iterator iter = dirtyObjectInfos.iterator();
		while (iter.hasNext()) {
			CosObjectInfo info = (CosObjectInfo)iter.next();
			int objNum = info.getObjNum();
			int objGen = info.getObjGen();
			if (info.isFree())
				objGen = -1;
			modList.add(objNum, new CosObjectID(objNum, objGen));
		}
		try {
			mXRef.getChangedObjects(eof, modList);
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
		modList.delete(0);
		return modList;
	}

	/**
	 * Get the position of the object in the PDF file stream.
	 * Returns zero for direct objects.
	 */
	long getObjPos(CosObject obj)
	{
		CosObjectInfo info = obj.getInfo();
		if (info == null)
			return 0;
		if (info.isCompressed())
			return info.getStreamInfo().getPos();
		return info.getPos();
	}

	private void copyTrailerFields(CosDictionary trailer, int newSize)
		throws PDFCosParseException, PDFIOException, PDFSecurityException{
		CosDictionary oldTrailer = getTrailer();

		// copy fields from the old trailer to the new
		if (oldTrailer.containsKey(ASName.k_Root))
			trailer.put(ASName.k_Root, oldTrailer.get(ASName.k_Root));
		if (oldTrailer.containsKey(ASName.k_Info)) {
			CosDictionary dict = (CosDictionary)oldTrailer.get(ASName.k_Info);
			if(! dict.isIndirect()) {
				CosCloneMgr cloneMgr = new CosCloneMgr(this);
				dict = (CosDictionary) cloneMgr.clone(dict);
			}
			trailer.put(ASName.k_Info, dict);
		}
		if (oldTrailer.containsKey(ASName.k_Encrypt))
			trailer.put(ASName.k_Encrypt, oldTrailer.get(ASName.k_Encrypt));

		// SLG - Don't put the /ID entry in the trailer of an FDF Document
		// SLG - Don't put the /Size entry in the trailer of an FDF Document
		if (!mIsFDF) {
			if (oldTrailer.containsKey(ASName.k_ID))
				trailer.put(ASName.k_ID, oldTrailer.get(ASName.k_ID));
			trailer.put(ASName.k_Size, newSize);
		}
	}

	private static class IDProducer extends OutputStream
	{
		MessageDigest mDigest;

		/**
		 * @author sweet
		 * Create an OutputStream so that I can get java.mri.uid written to MD5 digest.
		 * @throws RuntimeException, hard failure, if MD5 does not exist
		 * This represents a "can't happen" situation
		 */
		IDProducer()
		{
			try {
				mDigest = MessageDigest.getInstance("MD5");
			}
			catch (NoSuchAlgorithmException e) {
				throw new RuntimeException("MD5 is not available.", e);
			}
		}

		/* (non-Javadoc)
		 * @see java.io.OutputStream#write(int)
		 */

		@Override
		public void write(int b)
		{
			mDigest.update((byte)b);
		}

		byte[] getDigest()
		{
			return mDigest.digest();
		}
	}

	CosArray createUpdateDocID(boolean both)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		// get the array of two strings - it could be missing
		CosObject idArray = null;
		if (!both) {
			idArray = getTrailer().get(ASName.k_ID);
			if (!(idArray instanceof CosArray) || ((CosArray)idArray).size() != 2) {
				idArray = null;
				both = true;
			}
		}
		IDProducer idproc = new IDProducer();
		DataOutputStream uidOut = new DataOutputStream(idproc);
		UID newUID = new UID();	// get a unique ID
		try {
			newUID.write(uidOut);	// write the UID to the digest
			if (idArray != null) {
				ASString oldID = ((CosArray)idArray).getString(1);	// get the old changeable ID
				idproc.write(oldID.getBytes());	// write it to the digest
			}
		}
		catch (IOException e) {
			throw new PDFIOException(e);
		}
		byte[] newID = idproc.getDigest();
		if (idArray == null) {
			idArray = createCosArray(CosObject.DIRECT);
			((CosArray)idArray).addString(new ASString(newID));
			((CosArray)idArray).add(createCosNull());	// make room for entry at 1
		}
		((CosArray)idArray).setString(1, new ASString(newID));	// write into second element of array
		if (both)
			((CosArray)idArray).setString(0, ((CosArray)idArray).getString(1));

		// now make sure they get written as hex
		for (int i = 0; i < 2; i++) {
			CosString elem = ((CosArray)idArray).getCosString(i);
			elem.setWriteHex(true);
			elem.setToEncrypt(false);
		}
		return ((CosArray)idArray);
	}

	// making this public as javascript API needs it anjoshi 10/22/07
	public void markDirty()
	{
		mDocIsDirty = true;
	}

	/**
	 * 
	 * Is the document dirty?
	 * @return <code>true</code> if the document is dirty; <code>false</code> otherwise
	 */
	public boolean isDirty()
	{
		return mDocIsDirty;
	}

	/**
	 * 
	 * Mark document as not dirty.
	 * @return previous dirty state
	 */
	public boolean markNotDirty()
	{
		boolean isDirty = mDocIsDirty;
		mDocIsDirty = false;
		return isDirty;
	}

	/**
	 * 
	 */
	public long getEOF()
		throws PDFIOException
	{
		if (mBuf == null)
			return 0;
		try {
			return mBuf.length();
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 * 
	 * Get the current XRef type
	 */
	public int getXRefType()
	{
		if (mXRef != null) {
			int xrefType = mXRef.getType();
			if (xrefType == XRefTable.tStream)
				return CosSaveParams.XREFSTREAM;
			if (xrefType == XRefTable.tHybrid)
				return CosSaveParams.XREFHYBRID;
		}
		return CosSaveParams.XREFTABLE;
	}

	CosList buildObjectList(boolean dirtyOnly)
	{
		CosList objectList = new CosList();
		Iterator iter = mObjectInfos.iterator();
		while (iter.hasNext()) {
			CosObjectInfo info = (CosObjectInfo)iter.next();
			if ((dirtyOnly && info.isDirty()) || (!dirtyOnly && !info.isFree())) {
				objectList.add(info.getObjNum(), info);
			}
		}
		return objectList;
	}

	CosList buildCompObjList(int xrefStyle)
		throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		CosList stmList = new CosList();
		Iterator iter = mObjectInfos.iterator();
		CosDictionary catalog = getRoot();
		CosDictionary encrypt = getEncryptionDictionary();
		while (iter.hasNext()) {
			CosObjectInfo info = (CosObjectInfo)iter.next();
			if (info.isDirty()) {
				if (info.isCompressed()) {
					CosObjectInfo stmInfo = info.getStreamInfo();
					CosList stmObjList = (CosList)stmList.get(stmInfo.getObjNum());
					if (stmObjList == null) {
						stmObjList = new CosList();
						stmList.add(stmInfo.getObjNum(), stmObjList);
					}
					stmObjList.add(info.getObjNum(), info);
				} else {
					if (xrefStyle == CosSaveParams.XREFSTREAM) {
						CosObject obj = info.getObject();
						if (obj == null || obj instanceof CosStream)
							continue;
						if (obj == catalog || obj == encrypt)
							continue;
						if (obj instanceof CosDictionary)
						{
							CosObject type = ((CosDictionary)obj).get(ASName.k_Type);
							if(type instanceof CosName && (dictionariesNotToBeCompressed.contains(type.nameValue())))
							{
								continue;
							}
						}
						mCompObjInfos.add(info.getObjNum(), info);
					}
				}
			}
		}
		// Build the new compressed object list
		Iterator compIter = mCompObjInfos.iterator();
		CosObjectStream curObjStm = null;
		CosList stmObjList = null;
		int curIndex = OBJSTM_MAXNUMOBJS;
		while (compIter.hasNext()) {
			CosObjectInfo info = (CosObjectInfo)compIter.next();
			CosObject obj = info.getObject();
			if (curIndex == OBJSTM_MAXNUMOBJS) {
				curIndex = 0;
				if (curObjStm != null) {
					// TODO: This is a hack to get prevent this object stream to be written out to the document.
					// The writing will be taken care of by the calling method. We needed that object just a place
					// holder for dirty objcets while building the list of objects to be written.
					curObjStm.markNotDirty();
				}
				curObjStm = createCosObjectStream();
				stmObjList = new CosList();
				stmList.add(curObjStm.getInfo().getObjNum(), stmObjList);
			}
			curObjStm.addObjectToStream(obj);
			stmObjList.add(info.getObjNum(), info);
			curIndex++;
		}
		if (curObjStm != null) {
			// TODO: This is a hack to get prevent this object stream to be written out to the document.
			// The writing will be taken care of by the calling method. We needed that object just a place
			// holder for dirty objcets while building the list of objects to be written.
			curObjStm.markNotDirty();
		}
		return stmList;
	}

	/**
	 * Clean up existing "in memory" COS objects, turnaround the OutputByteStream, and
	 * prepare the just saved PDF document to be used as the backing store to continue with.
	 * This method takes ownership of the <code>OutputByteStream</code> and it should not be
	 * used by the caller after this method is called.
	 * @param outputByteStream the stream to which the file was just saved
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	protected void postSaveCleanup(OutputByteStream outputByteStream, ByteWriter byteWriter)
		throws PDFCosParseException, IOException, PDFIOException, PDFSecurityException
	{
		if (mBuf != null)
			mBuf.close();		// close the old InputByteStream
		releaseAllCosObjects();
		if (mByteReader != byteWriter) {
			mStreamManager.resetMasterByteReader(byteWriter);
			mByteReader = byteWriter;
		}
		mBuf = outputByteStream.closeAndConvert();
		mXRef.resetXRef(mBuf.slice());
		mOrigNumObjects = mNumObjects;
		mCosLin = null;
		mDocIsDirty = false;
		if(cosRepairList != null)
			cosRepairList.clear(); // clear the repair list.
		repairTypes.clear();
		documentCosLevelRepaired = false; // clear cos level repair flag
		this.nextIncrementalSectionOffset = -1;
	}

	/**
	 * This method calls the close on all CosObjects held by this CosDocument
	 * that are currently in memory. This causes the CosObjects to release all resources
	 * and become invalid.
	 * @throws PDFIOException
	 * @throws PDFCosParseException
	 * @throws PDFSecurityException
	 * @throws IOException
	 */
	private void closeAllCosObjects()
		throws PDFIOException, PDFCosParseException, PDFSecurityException, IOException
	{
		Exception saveEx = null;
		Iterator iter = mObjectInfos.iterator();
		while (iter.hasNext()) {
			CosObjectInfo info = (CosObjectInfo)iter.next();
			try {
				CosObject obj = info.getObject(false /* don't load in released objects */);
				if (obj != null)
					obj.close();
			} catch (Exception e) {
				saveEx = e;
			}
		}
		if (saveEx != null) {
			if (saveEx instanceof PDFIOException)
				throw (PDFIOException)saveEx;
			if (saveEx instanceof PDFCosParseException)
				throw (PDFCosParseException)saveEx;
			if (saveEx instanceof PDFSecurityException)
				throw (PDFSecurityException)saveEx;
			if (saveEx instanceof IOException)
				throw (IOException)saveEx;
			throw new PDFIOException(saveEx);
		}
	}

	/**
	 * This method calls the release on all CosObjects held by this CosDocument that
	 * are currently in meory. This cause the CosObjects to release the their hold on
	 * their current InputByteStream.
	 * @throws PDFIOException
	 * @throws PDFCosParseException
	 * @throws PDFSecurityException
	 * @throws IOException
	 */
	private void releaseAllCosObjects()
		throws PDFIOException, PDFCosParseException, PDFSecurityException, IOException
	{
		Iterator iter = mObjectInfos.iterator();
		while (iter.hasNext()) {
			CosObjectInfo info = (CosObjectInfo)iter.next();
			CosObject obj = info.getObject(false /* don't load in released objects */);
			if (obj != null)
				obj.release();
		}
	}

	// Creation methods

	/**
	 * Creates a new COS array object and populates it with the specified ArrayList
	 * of CosObjects. The created CosArray container can be made direct or
	 * indirect by setting the directStatus flag.
	 * @param array - ArrayList of CosObjects that becomes the data in the new CosArray object
	 * @return A newly created CosArray.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosArray createCosArray(ArrayList array, int directStatus)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosArray newArray = null;
		if (directStatus == CosObject.INDIRECT) {
			CosObjectInfo info = newObjectInfo();
			newArray = new CosArray(this, array, info);
			try {
				info.markDirty();
			} catch (IOException e) {
				throw new PDFIOException(e);
			}
		}
		else
			newArray = new CosArray(this, array, null);
		return newArray;
	}

	/**
	 * 
	 *
	 * Creates a new COS array object and populates it with the specified ArrayList
	 * of CosObjects. The created CosArray container will be INDIRECT.
	 *
	 * @param array - ArrayList of CosObjects that becomes the data in the new CosArray object
	 * @return A newly created CosArray.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosArray createIndirectCosArray(ArrayList array)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return createCosArray(array, CosObject.INDIRECT);
	}

	/**
	 * 
	 *
	 * Creates a new COS array object and populates it with the specified ArrayList
	 * of CosObjects. The created CosArray container will be DIRECT.
	 *
	 * @param array - ArrayList of CosObjects that becomes the data in the new CosArray object
	 * @return A newly created CosArray.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosArray createCosArray(ArrayList array)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return createCosArray(array, CosObject.DIRECT);
	}

	/**
	 * Creates a new COS array object and populates it with the specified ArrayList
	 * of non CosObjects. The created CosArray container can be made direct or
	 * indirect by setting the directStatus flag.
	 * @param array - ArrayList of non CosObjects that becomes the data in the new CosArray object
	 * @return A newly created CosArray.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosArray createCosArrayFromNonCosData(ArrayList array, int directStatus)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosArray newArray = createCosArray(directStatus);
		Iterator arrayIter = array.iterator();
		CosObject obj = null;
		while(arrayIter.hasNext()) {
			obj = convertToCosObject(arrayIter.next());
			if (obj != null)
				newArray.add(obj);
		}
		return newArray;
	}

	/**
	 * 
	 *
	 * Creates a new COS array object and populates it with the specified ArrayList
	 * of non CosObjects. The created CosArray container will be INDIRECT.
	 * @param array - ArrayList of non CosObjects that becomes the data in the new CosArray object
	 * @return A newly created CosArray.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosArray createIndirectCosArrayFromNonCosData(ArrayList array)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return createCosArrayFromNonCosData(array, CosObject.INDIRECT);
	}

	/**
	 * 
	 *
	 * Creates a new COS array object and populates it with the specified ArrayList
	 * of non CosObjects. The created CosArray container will be DIRECT.
	 * @param array - ArrayList of non CosObjects that becomes the data in the new CosArray object
	 * @return A newly created CosArray.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosArray createCosArrayFromNonCosData(ArrayList array)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return createCosArrayFromNonCosData(array, CosObject.DIRECT);
	}

	/**
	 * Creates a new, empty COS array object. Array will be direct or
	 * indirect depending on the value of the directStatus enum.
	 *
	 * @return A newly allocated COS array.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosArray createCosArray(int directStatus)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return createCosArray(new ArrayList(), directStatus);
	}

	/**
	 * 
	 *
	 * Creates a new, empty COS array object. Array will be INDIRECT.
	 *
	 * @return A newly allocated COS array.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosArray createIndirectCosArray()
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return createCosArray(CosObject.INDIRECT);
	}

	/**
	 * 
	 *
	 * Creates a new, empty COS array object. Array will be DIRECT.
	 *
	 * @return A newly allocated COS array.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosArray createCosArray()
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return createCosArray(CosObject.DIRECT);
	}

	/**
	 * 
	 *
	 * Create a new CosBoolean instance (always a direct object)
	 * @param value - boolean value attribute of the instance
	 * @return - CosBoolean new instance
	 */
	public CosBoolean createCosBoolean(boolean value)
	{
		return new CosBoolean(this, value, null);
	}

	/**
	 * 
	 *
	 * Create a new CosBoolean instance (always a direct object)
	 * @param value - Boolean value attribute of the instance
	 * @return - CosBoolean new instance
	 */
	public CosBoolean createCosBoolean(Boolean value)
	{
		return new CosBoolean(this, value, null);
	}

	/**
	 * Creates an empty COS dictionary. Based on direct status enum, this will be created
	 * as either an INDIRECT object (can be referenced and entered in the xref)or as a
	 * DIRECT object.
	 *
	 * @return Newly allocated COS dictionary.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosDictionary createCosDictionary(int directStatus)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosDictionary newDict = null;
		if (directStatus == CosObject.INDIRECT) {
			CosObjectInfo info = newObjectInfo();
			newDict = new CosDictionary(this, new LinkedHashMap(), info);
			try {
				info.markDirty();
			} catch (IOException e) {
				throw new PDFIOException(e);
			}
		}
		else
			newDict = new CosDictionary(this, new LinkedHashMap(), null);
		return newDict;
	}

	/**
	 * 
	 *
	 * Creates an empty COS dictionary. This will be created
	 * as an INDIRECT object (can be referenced and entered in the xref).
	 *
	 * @return Newly allocated COS dictionary.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosDictionary createCosDictionary()
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return createCosDictionary(CosObject.INDIRECT);
	}

	/**
	 * 
	 *
	 * Creates an empty COS dictionary. This will be created
	 * as a DIRECT object (cannot be referenced or entered in the xref).
	 *
	 * @return Newly allocated COS dictionary.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosDictionary createDirectCosDictionary()
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return createCosDictionary(CosObject.DIRECT);
	}

	/**
	 * Creates a COS dictionary from a Map of Cos objects based on the value of
	 * the directStatus enum.
	 *
	 * @param cosData Map of Cos objects with ASName keys
	 * @return Newly allocated COS dictionary.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosDictionary createCosDictionary(Map cosData, int directStatus)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosDictionary newDict = null;
		if (directStatus == CosObject.INDIRECT) {
			CosObjectInfo info = newObjectInfo();
			newDict = new CosDictionary(this, cosData, info);
			try {
				info.markDirty();
			} catch (IOException e) {
				throw new PDFIOException(e);
			}
		}
		else
			newDict = new CosDictionary(this, cosData, null);
		return newDict;
	}

	/**
	 * Creates a COS dictionary from a Map of Cos objects. This will be created
	 * as an INDIRECT object (can be referenced and entered in the xref).
	 *
	 * @param cosData Map of Cos objects with ASName keys
	 * @return Newly allocated COS dictionary.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosDictionary createCosDictionary(Map cosData)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return createCosDictionary(cosData, CosObject.INDIRECT);
	}

	/**
	 * Creates a COS dictionary from a Map of Cos objects. This will be created
	 * as a DIRECT object (cannot be referenced or entered in the xref).
	 *
	 * @param cosData Map of Cos objects with ASName keys
	 * @return Newly allocated COS dictionary.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosDictionary createDirectCosDictionary(Map cosData)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return createCosDictionary(cosData, CosObject.DIRECT);
	}

	/**
	 * Constructs COS dictionary from the HashMap that contains non-Cos object values.
	 * The keys are assumed to be strings and not ASNames. For the values,
	 * Boolean gets translated to CosBoolean, Number to CosNumeric,
	 * Byte Array to CosString, String to CosName, HashMap to CosDictionary
	 * and ArrayList to CosArray. Any other type gets translated to null and
	 * is skipped. The strings are converted to ASNames.
	 * This uses CosDictionary's put method to insure that the direct object values
	 * get the parent dictionary's ID set.
	 * Each entry in HashMap and ArrayList gets recursively translated
	 * under the same rules.
	 * @param data - a Map with string key - non-CosObject value pairs
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosDictionary createCosDictionaryFromNonCosData(Map data, int directStatus)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosDictionary newDict = createCosDictionary(directStatus);
		Iterator<Map.Entry> mapIter = data.entrySet().iterator();
		while(mapIter.hasNext()) {
			Map.Entry entry = mapIter.next();
			if (entry != null && entry.getKey() != null) {
				CosObject cosObj = convertToCosObject(entry.getValue());
				if (cosObj != null) {
					newDict.put(ASName.create((String)entry.getKey()), cosObj);
				}
			}
		}
		return newDict;
	}

	/**
	 * 
	 *
	 * Creates a COS dictionary from non CosObject data.
	 * Boolean gets translated to CosBoolean, Number to CosNumeric,
	 * Byte Array to CosString, String to CosName, HashMap to CosDictionary
	 * and ArrayList to CosArray. Any other type gets translated to null and
	 * is skipped.
	 * Each entry in HashMap and ArrayList gets recursively translated
	 * under the same rules.
	 * This dictionary will be created
	 * as an INDIRECT object (can be referenced and entered in the xref).
	 *
	 * @param javaMap - Map with strings as keys, nonCos objects as data
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosDictionary createCosDictionaryFromNonCosData(Map javaMap)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return createCosDictionaryFromNonCosData(javaMap, CosObject.INDIRECT);
	}

	/**
	 * 
	 *
	 * Creates a COS dictionary from non CosObject data.
	 * Boolean gets translated to CosBoolean, Number to CosNumeric,
	 * Byte Array to CosString, String to CosName, HashMap to CosDictionary
	 * and ArrayList to CosArray. Any other type gets translated to null and
	 * is skipped.
	 * Each entry in HashMap and ArrayList gets recursively translated
	 * under the same rules.
	 * This dictionary will be created
	 * as a DIRECT object (cannot be referenced or entered in the xref).
	 *
	 * @param javaMap - Map with strings as keys, nonCos objects as data
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosDictionary createDirectCosDictionaryFromNonCosData(Map javaMap)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return createCosDictionaryFromNonCosData(javaMap, CosObject.DIRECT);
	}

	/**
	 * 
	 *
	 * Creates a COS name object corresponding to the specified atom.
	 *
	 * @param value	Atom to be wrapped by a COS name object
	 * @return Newly allocated COS name object.
	 */
	public CosName createCosName(ASName value)
	{
		return new CosName(this, value, null);
	}

	/**
	 * 
	 *
	 * Creates a COS null object.
	 *
	 * @return Newly allocated COS null object.
	 */
	public CosNull createCosNull()
	{
		return new CosNull(this);
	}

	/**
	 * 
	 *
	 * Creates a COS numeric value object.
	 *
	 * @param value	Integer value for the object
	 * @return Newly allocated COS numeric object initialized with the specified value.
	 */
	public CosNumeric createCosNumeric(Number value)
	{
		return new CosNumeric(this, value, null);
	}

	/**
	 * 
	 *
	 * Creates a COS numeric value object.
	 *
	 * @param value	Integer value for the object
	 * @return Newly allocated COS numeric object initialized with the specified value.
	 */
	public CosNumeric createCosNumeric(int value)
	{
		return new CosNumeric(this, Integer.valueOf(value), null);
	}

	/**
	 * 
	 *
	 * Creates a COS numeric value object.
	 *
	 * @param value	Integer value for the object
	 * @return Newly allocated COS numeric object initialized with the specified value.
	 */
	public CosNumeric createCosNumeric(long value)
	{
		return new CosNumeric(this, Long.valueOf(value), null);
	}

	/**
	 * 
	 *
	 * Creates a COS numeric value object.
	 *
	 * @param value	Floating point value for the object
	 * @return Newly allocated COS numeric object initialized with the specified value.
	 */
	public CosNumeric createCosNumeric(double value)
	{
		return new CosNumeric(this, new Double(value), null);
	}

	/**
	 * 
	 *
	 * Creates a COS numeric value object.
	 *
	 * @param inputRep byte string value for the object
	 * @return Newly allocated COS numeric object initialized with the specified value.
	 */
	public CosNumeric createCosNumeric(byte[] inputRep)
		throws PDFCosParseException
	{
		return new CosNumeric(this, inputRep, null);
	}

	/**
	 * 
	 *
	 * Creates a COS numeric value object.
	 *
	 * @param source CosNumeric object to copy
	 * @return Newly allocated COS numeric object initialized with the specified value.
	 */
	public CosNumeric createCosNumeric(CosNumeric source)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			return new CosNumeric(this, source);
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 * 
	 *
	 * Create a new, initialized CosObjectStream instance
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosObjectStream createCosObjectStream()
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			return new CosObjectStream(this);
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 * 
	 *
	 * Creates a COS stream with an empty stream data. The filter information
	 * must be set in the dictionary portion of the stream before obtaining
	 * the output stream that writes into the stream data. CosStreams must
	 * always be indirect objects.
	 *
	 * @return newly allocated COS stream.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosStream createCosStream()
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			CosObjectInfo info = newObjectInfo();
			/*
			 * streams are always indirect (per PDF Reference manual), so we
			 * go ahead and generate a new ID for the new stream.
			 */
			info.markDirty();
			CosStream cosObj = new CosStream(this, info);
			cosObj.setIsEncrypted(false);	// The new stream is decrypted but not encrypted
			cosObj.setToEncrypt(true);
			return cosObj;
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 * 
	 * Creates a CosStream object.
	 *
	 * @param data Data for the string
	 * @return Newly allocated CosStream object initialized with the specified data.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosStream createCosStream(InputByteStream data)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			CosObjectInfo info = newObjectInfo();
			info.markDirty();
			CosStream cosObj = new CosStream(this, info, data);
			cosObj.setIsEncrypted(false);	// The new stream is decrypted but not encrypted
			cosObj.setToEncrypt(true);
			return cosObj;
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 * 
	 *
	 * Creates a COS string object.
	 *
	 * @param value	Data for the string
	 * @return Newly allocated COS string object initialized with the specified data.
	 */
	public CosString createCosString(ASString value)
	{
		return createCosString(value.getBytes());
	}

	/**
	 * 
	 *
	 * Creates a COS string object.
	 *
	 * @param value	Data for the string
	 * @return Newly allocated COS string object initialized with the specified data.
	 */
	public CosString createCosString(byte[] value)
	{
		return new CosString(this, value, 0, value.length, false, null);
	}

	/**
	 * 
	 *
	 * Creates a COS string object.
	 *
	 * @param value	String. NOTE this String must be in PDFDocEncoding.
	 * @return Newly allocated COS string object initialized with the specified data.
	 */
	public CosString createCosString(String value)
	{
		byte[] bytes = PDFDocEncoding.fromUnicodeString(value);
		return new CosString(this, bytes, 0, bytes.length, false, null);
	}

	/**
	 * 
	 *
	 * Creates a COS hex string object.
	 *
	 * @param value	Data for the string
	 * @return Newly allocated COS string object initialized with the specified data.
	 */
	public CosString createHexCosString(ASString value)
	{
		return createHexCosString(value.getBytes());
	}

	/**
	 * 
	 *
	 * Creates a COS hex string object.
	 *
	 * @param value	Data for the string
	 * @return Newly allocated COS string object initialized with the specified data.
	 */
	public CosString createHexCosString(byte[] value)
	{
		return new CosString(this, value, 0, value.length, false, null, true);
	}

	/**
	 * 
	 *
	 * Creates a COS hex string object.
	 *
	 * @param value	Data for the string
	 * @return Newly allocated COS string object initialized with the specified data.
	 */
	public CosString createHexCosString(String value)
	{
		byte[] bytes = PDFDocEncoding.fromUnicodeString(value);
		return new CosString(this, bytes, 0, bytes.length, false, null, true);
	}

	/**
	 * Implicitly convert a Java object type to a CosObject. If a CosObject
	 * is input, it's returned. If the object is an unhandled type, null is returned.
	 *
	 * @param val - Instance of Object
	 * @return - CosObject result which is null if the input Object was an
	 * unhandled type.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	protected CosObject convertToCosObject(Object val)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		if (val instanceof CosObject)
			return (CosObject)val;
		CosObject newObj = null;
		if (val instanceof Boolean)
			newObj = createCosBoolean((Boolean)val);
		else if (val instanceof Number)
			newObj = createCosNumeric((Number) val);
		else if (val instanceof byte[])
			newObj = createCosString((byte[]) val);
		else if (val instanceof String)
			newObj = createCosName(ASName.create((String) val));
		else if (val instanceof ASName)
			newObj = createCosName((ASName) val);
		else if (val instanceof HashMap)
			newObj = createCosDictionaryFromNonCosData((HashMap) val, CosObject.DIRECT);
		else if (val instanceof ArrayList)
			newObj = createCosArrayFromNonCosData((ArrayList) val, CosObject.DIRECT);
		return newObj;
	}
	
	/**
	 * Sets the repaired value for the entry inside cosdictionary.
	 * @param obj CosDictionary which is repaired
	 * @param key key in cosdictionary whose value is repaired
	 * @param value repaired value.
	 */
	void setRepairedValue(CosDictionary obj, ASName key, CosObject value)
	{
		if(! documentCosLevelRepaired)
		{
			cosRepairList = new CosRepairList(this);
			documentCosLevelRepaired = true;
		}
		cosRepairList.setRepairedValue(obj, key, value);
	}
	/**
	 * Sets the repaired value of the cos object with the specified object number.
	 * @param objNum object number
	 * @param obj Repaired object.
	 */
	void setRepairedValue(Integer objNum, CosObject obj)
	{
		if(obj == null)
			return;
		if(! documentCosLevelRepaired)
		{
			cosRepairList = new CosRepairList(this);
			documentCosLevelRepaired = true;
		}
		cosRepairList.setRepairedValue(objNum, obj);
	}
	/**
	 * Gets repaired cosobject for the given object number for the key.
	 */
	CosObject getRepairedValue(Integer objNum, ASName key)
	{
		return documentCosLevelRepaired ? cosRepairList.getRepairedValue(objNum, key) : null;
	}
	/**
	 * Get the repaired cos object for the specified object number.
	 * @param objNum
	 * @return {@link CosObject}
	 */
	CosObject getRepairedValue(Integer objNum)
	{
		return documentCosLevelRepaired && cosRepairList != null? cosRepairList.getRepairedValue(objNum) : null;
	}

	/**
	 * Returns if some cos object level repair is performed on this document.
	 */
	boolean isDocumentCosLevelRepaired()
	{
		return documentCosLevelRepaired;
	}
	
	/**
	 * Returns if some cos object level repair is performed on this document.
	 */
	void setDocumentCosLevelRepaired()
	{
		documentCosLevelRepaired = true;
	}
	
	/**
	 * @return return the repair list
	 */
	CosRepairList getRepairList()
	{
		return cosRepairList;
	}
	
	/**
	 * For some API's such as PDF/A validation Gibson needs to work on original
	 * cos objects and ignore repairs made to cos objects. This flag is set to
	 * false for such operations. However this should be reset once that operation
	 * is done, so that for further operations on the document repair list cab be used.
	 * Default value is true.
	 * @param useRepairList
	 */
	public void setUseRepairList(boolean useRepairList) 
	{
		this.useRepairList = useRepairList;
	}
	
	/**
	 * Gets if repair list should be used.
	 */
	boolean getUseRepairList() 
	{
		return useRepairList;
	}

	boolean isFDF() 
	{
		return mIsFDF;
	}
}
