/* ****************************************************************************
 *
 *	File: CosUtils.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.cos;

import java.io.IOException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Collection;
import java.util.Iterator;

import com.adobe.internal.io.stream.InputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosParseException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFInvalidDocumentException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityException;
import com.adobe.internal.pdftoolkit.core.types.ASName;
import com.adobe.internal.pdftoolkit.core.util.Utility;

/**
 *   class:  CosUtils
 *
 * describtion:
 *
 *
 * change history:
 */
public final class CosUtils
{
	private CosUtils(){}
	
	/**
	 * This method clones all the keys of dictionary "dict" except that of present in specialKeys
	 * array passed here.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public static CosDictionary cloneOrdinaryKeys(CosDictionary cloned, CosCloneMgr cloneHandler, CosDictionary dict, ASName[] specialKeys)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		Iterator keyIterator = dict.keyIterator();
		while (keyIterator.hasNext()) {
			ASName curKey = (ASName) keyIterator.next();
			if (!Utility.nameInArray(curKey, specialKeys)) {
				cloned.put(curKey, cloneHandler.clone(dict.get(curKey)));
			}
		}
		return cloned;
	}

	/**
	 * This method clones all the keys of dictionary "dict" except that of present in specialKeys
	 * array passed here.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public static CosDictionary cloneOrdinaryKeys(CosDocument toDoc, CosCloneMgr cloneHandler, CosDictionary dict, ASName[] specialKeys)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosDictionary cloned = toDoc.createCosDictionary();
		return cloneOrdinaryKeys(cloned, cloneHandler, dict, specialKeys);
	}
	
	/**
	 * @param ibs
	 * @param digestAlgorithm
	 * @return byte[]
	 * @throws PDFInvalidDocumentException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	static byte[] digestStream(InputByteStream ibs, String digestAlgorithm) throws PDFInvalidDocumentException, PDFIOException, PDFSecurityException{
		byte[] byteStream = new byte[8192];
		MessageDigest digester=null;
		try {
			digester = MessageDigest.getInstance(digestAlgorithm);
			while(ibs.bytesAvailable() > 0) {
				int numBytesRead = ibs.read(byteStream);
				digester.update(byteStream, 0, numBytesRead);
			}
		} catch (IOException e) {
			throw new PDFIOException(e);
		}catch (NoSuchAlgorithmException e) {
			throw new PDFSecurityException(e);
		}finally{
			try{
				if(ibs != null)
					ibs.close();
			}catch(IOException e){
				throw new PDFIOException("Problem occured while closing the input byte stream.", e);
			}
		}
		return digester.digest();
	}
	
	/**
	 * compares the CosObjects in a collection on the bases of reference only.
	 */
	public static <T extends CosObject> int indexOfCosObjectRef(Collection<T> objectList, T obj){
		int index = 0;
		Iterator<T> itr = objectList.iterator();
		while(itr.hasNext()){
			if(obj == itr.next())
				return index;
			index++;
		}
		return -1;
	}

}
