/* File: JCECredentials.java
*  
* ****************************************************************************
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2007 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
* ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.credentials.impl;

import java.io.IOException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.Security;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.security.spec.InvalidKeySpecException;

import com.adobe.internal.pdftoolkit.core.credentials.CredentialFactory;
import com.adobe.internal.pdftoolkit.core.credentials.Credentials;
import com.adobe.internal.pdftoolkit.core.credentials.PrivateKeyHolder;
import com.adobe.internal.pdftoolkit.core.credentials.impl.utils.CertUtils;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFInvalidParameterException;

/**
 * JCE Credentials with JDK1.5 HSM support.
 * @author mdharan
 *
 */
public class JCECredentials implements Credentials
{
	String providerName;
	boolean isHSM;

	public JCECredentials(PrivateKeyHolder keyHolder, byte[] derEncodedCert, byte[][] derEncodedCertChain) 
	throws PDFInvalidParameterException
	{
		try
		{
		      X509Certificate jceCert = (X509Certificate) CertUtils.importCertificate(derEncodedCert);
		      Certificate[] jceCertChain = CertUtils.importCertificateChain(derEncodedCertChain);	
		      PrivateKeyHolder jcePrivateKeyHolder = CredentialFactory.computeJCEPrivateKey(keyHolder);  // Turns non-JCE private keys into JCE private key
		      this.populateJCEData(jcePrivateKeyHolder, jceCert, jceCertChain);
		      if(keyHolder instanceof JCEKeyHolder)
		    	  this.providerName = ((JCEKeyHolder)keyHolder).getProviderName();
		      else
		    	  this.providerName = "RSA";
		} catch (CertificateException e){
			throw new PDFInvalidParameterException("Could not load the certficate and certificate chain - ",e);
		} catch (IOException e){
			throw new PDFInvalidParameterException("IO Exception when importing JCE keys and certs - ",e);
		} catch (NoSuchAlgorithmException e) {
			throw new PDFInvalidParameterException("Could not determine the Algorithm - ",e);
		} catch (InvalidKeySpecException e) {
			throw new PDFInvalidParameterException("Invalid Key ",e);
		}
	}


	public JCECredentials(PrivateKey privateKey, X509Certificate cert,
			Certificate[] certChain) {
		
		  JCEKeyHolder keyHolder = new JCEKeyHolder(privateKey, "RSA");
		  this.populateJCEData(keyHolder, cert, certChain);
	}
	 
	
	public JCECredentials(PrivateKeyHolder keyHolder, X509Certificate cert, Certificate[] certChain)
		      throws NoSuchAlgorithmException, InvalidKeySpecException, PDFInvalidParameterException {
		    PrivateKeyHolder jcePrivateKeyHolder = CredentialFactory.computeJCEPrivateKey(keyHolder);  // Turns non-JCE private keys into JCE private key
		    this.populateJCEData(jcePrivateKeyHolder, cert, certChain);
	}


	public void setProviderName(String providerName) {
		this.providerName = providerName;
	}

	public boolean isHSM() {
		return isHSM;
	}

	public void setHSM(boolean isHSM) {
		this.isHSM = isHSM;
	}

  private JCEKeyHolder privateKeyHolder;
  private X509Certificate cert;
  private Certificate[] certChain;

  protected void populateJCEData(PrivateKeyHolder privateKeyHolder, X509Certificate cert, Certificate[] certChain)
  {
    this.privateKeyHolder = (JCEKeyHolder)privateKeyHolder; // That's the only kind of keyHolder we implement
    this.cert = cert;
    this.certChain = certChain;
  }

  /**
   * Gets the private key associate with this credential.
   * 
   * @return PrivateKey
   */
  public PrivateKey getPrivateKey()
  {
    return (privateKeyHolder != null)? privateKeyHolder.getJcePrivateKey():null;
  }

  /**
   * Get the name of the key Provider
   * 
   * @return String
   */
  public String getProviderName()
  {
    return  (privateKeyHolder != null)? privateKeyHolder.getProviderName():null;
  }

  /**
   * Get a Security Provider for the private key
   * 
   * @return java.security.Provider
   */
  public Provider getProvider()
  {
    return Security.getProvider(getProviderName());
  }

  /**
   * Gets the x509 certificate associated with this credential.
   * 
   * @return X509Certificate
   */
  public X509Certificate getCertificate()
  {
    return cert;
  }

  /**
   * Gets the certificate chain associated with this credential.
   * This method is useful when client already have an ordered certificate chain.
   * @return Certificate[]
   */
  public Certificate[] getCertificateChain()
  {
    return certChain;
  }

}
