package com.adobe.internal.pdftoolkit.core.filter;

import java.io.IOException;
import java.io.InputStream;

import com.adobe.internal.pdftoolkit.core.exceptions.PDFFilterFormatException;

public class CCITTFaxInputStream
  extends DecodeInputStream
  implements CCITTFaxTables
{
  private static final byte decode[] /* 256-entry groups */ =
  {
    /* White */
    (byte)(XDT+9), (byte)(XDT+8),
                      29,     30,     45,     46,     22,     22,
      23,     23,     47,     48,     13,     13,     13,     13,
      20,     20,     33,     34,     35,     36,     37,     38,
      19,     19,     31,     32,      1,      1,      1,      1,
      12,     12,     12,     12,     53,     54,     26,     26,
      39,     40,     41,     42,     43,     44,     21,     21,
      28,     28,     61,     62,     63,      0,     68,     69,
      10,     10,     10,     10,     10,     10,     10,     10,
      11,     11,     11,     11,     11,     11,     11,     11,
      27,     27,     59,     60,
                   (byte)(XDT+6),  (byte)(XDT+7),     18,     18,
      24,     24,     49,     50,     51,     52,     25,     25,
      55,     56,     57,     58,     66,     66,     66,     66,
      89,     89,     89,     89,     70,     71,
                                           (byte)(XDT+0),     73,
      72, (byte)(XDT+1), (byte)(XDT+2), (byte)(XDT+3),
                   (byte)(XDT+4),  (byte)(XDT+5),     67,     67,
       2,      2,      2,      2,      2,      2,      2,      2,
       2,      2,      2,      2,      2,      2,      2,      2,
       3,      3,      3,      3,      3,      3,      3,      3,
       3,      3,      3,      3,      3,      3,      3,      3,
      65,     65,     65,     65,     65,     65,     65,     65,
       8,      8,      8,      8,      8,      8,      8,      8,
       9,      9,      9,      9,      9,      9,      9,      9,
      16,     16,     16,     16,     17,     17,     17,     17,
       4,      4,      4,      4,      4,      4,      4,      4,
       4,      4,      4,      4,      4,      4,      4,      4,
       5,      5,      5,      5,      5,      5,      5,      5,
       5,      5,      5,      5,      5,      5,      5,      5,
      14,     14,     14,     14,     15,     15,     15,     15,
      64,     64,     64,     64,     64,     64,     64,     64,
       6,      6,      6,      6,      6,      6,      6,      6,
       6,      6,      6,      6,      6,      6,      6,      6,
       7,      7,      7,      7,      7,      7,      7,      7,
       7,      7,      7,      7,      7,      7,      7,      7,

    /* Black */
    (byte)(XDT+7), (byte)(XDT+6), (byte)(XDT+4), (byte)(XDT+5),
                                      13,
                                (byte)(XDT+2), (byte)(XDT+3), 14,
       10,    10,     11,     11,
                   (byte)(XDT+1),  (byte)(XDT+0),     12,     12,
       9,      9,      9,      9,      8,      8,      8,      8,
       7,      7,      7,      7,      7,      7,      7,      7,
       6,      6,      6,      6,      6,      6,      6,      6,
       6,      6,      6,      6,      6,      6,      6,      6,
       5,      5,      5,      5,      5,      5,      5,      5,
       5,      5,      5,      5,      5,      5,      5,      5,
       1,      1,      1,      1,      1,      1,      1,      1,
       1,      1,      1,      1,      1,      1,      1,      1,
       1,      1,      1,      1,      1,      1,      1,      1,
       1,      1,      1,      1,      1,      1,      1,      1,
       4,      4,      4,      4,      4,      4,      4,      4,
       4,      4,      4,      4,      4,      4,      4,      4,
       4,      4,      4,      4,      4,      4,      4,      4,
       4,      4,      4,      4,      4,      4,      4,      4,
       3,      3,      3,      3,      3,      3,      3,      3,
       3,      3,      3,      3,      3,      3,      3,      3,
       3,      3,      3,      3,      3,      3,      3,      3,
       3,      3,      3,      3,      3,      3,      3,      3,
       3,      3,      3,      3,      3,      3,      3,      3,
       3,      3,      3,      3,      3,      3,      3,      3,
       3,      3,      3,      3,      3,      3,      3,      3,
       3,      3,      3,      3,      3,      3,      3,      3,
       2,      2,      2,      2,      2,      2,      2,      2,
       2,      2,      2,      2,      2,      2,      2,      2,
       2,      2,      2,      2,      2,      2,      2,      2,
       2,      2,      2,      2,      2,      2,      2,      2,
       2,      2,      2,      2,      2,      2,      2,      2,
       2,      2,      2,      2,      2,      2,      2,      2,
       2,      2,      2,      2,      2,      2,      2,      2,
       2,      2,      2,      2,      2,      2,      2,      2,

    /* White uncompressed */
    XDT,
               1,   UNCW,   UNCB,   W5_U,   W5_U,      5,      5,
       4,      4,      4,      4,      4,      4,      4,      4,
       3,      3,      3,      3,      3,      3,      3,      3,
       3,      3,      3,      3,      3,      3,      3,      3,
       2,      2,      2,      2,      2,      2,      2,      2,
       2,      2,      2,      2,      2,      2,      2,      2,
       2,      2,      2,      2,      2,      2,      2,      2,
       2,      2,      2,      2,      2,      2,      2,      2,
       1,      1,      1,      1,      1,      1,      1,      1,
       1,      1,      1,      1,      1,      1,      1,      1,
       1,      1,      1,      1,      1,      1,      1,      1,
       1,      1,      1,      1,      1,      1,      1,      1,
       1,      1,      1,      1,      1,      1,      1,      1,
       1,      1,      1,      1,      1,      1,      1,      1,
       1,      1,      1,      1,      1,      1,      1,      1,
       1,      1,      1,      1,      1,      1,      1,      1,

    /* Black uncompressed */
       1,      1,      1,      1,      1,      1,      1,      1,
       1,      1,      1,      1,      1,      1,      1,      1,
       1,      1,      1,      1,      1,      1,      1,      1,
       1,      1,      1,      1,      1,      1,      1,      1,
       1,      1,      1,      1,      1,      1,      1,      1,
       1,      1,      1,      1,      1,      1,      1,      1,
       1,      1,      1,      1,      1,      1,      1,      1,
       1,      1,      1,      1,      1,      1,      1,      1,
       2,      2,      2,      2,      2,      2,      2,      2,
       2,      2,      2,      2,      2,      2,      2,      2,
       2,      2,      2,      2,      2,      2,      2,      2,
       2,      2,      2,      2,      2,      2,      2,      2,
       3,      3,      3,      3,      3,      3,      3,      3,
       3,      3,      3,      3,      3,      3,      3,      3,
       4,      4,      4,      4,      4,      4,      4,      4,
       5,      5,      5,      5,      6,      6,      7,
                                                   (byte)(XDT+0),

    /* 2-D */
    (byte)(XDT+1),    UND,    UND,  (byte)(XDT+0),
                                  VL3_2D, VL3_2D, VR3_2D, VR3_2D,
  VL2_2D, VL2_2D, VL2_2D, VL2_2D, VR2_2D, VR2_2D, VR2_2D, VR2_2D,
    P_2D,   P_2D,   P_2D,   P_2D,   P_2D,   P_2D,   P_2D,   P_2D,
    P_2D,   P_2D,   P_2D,   P_2D,   P_2D,   P_2D,   P_2D,   P_2D,
    H_2D,   H_2D,   H_2D,   H_2D,   H_2D,   H_2D,   H_2D,   H_2D,
    H_2D,   H_2D,   H_2D,   H_2D,   H_2D,   H_2D,   H_2D,   H_2D,
    H_2D,   H_2D,   H_2D,   H_2D,   H_2D,   H_2D,   H_2D,   H_2D,
    H_2D,   H_2D,   H_2D,   H_2D,   H_2D,   H_2D,   H_2D,   H_2D,
  VL1_2D, VL1_2D, VL1_2D, VL1_2D, VL1_2D, VL1_2D, VL1_2D, VL1_2D,
  VL1_2D, VL1_2D, VL1_2D, VL1_2D, VL1_2D, VL1_2D, VL1_2D, VL1_2D,
  VL1_2D, VL1_2D, VL1_2D, VL1_2D, VL1_2D, VL1_2D, VL1_2D, VL1_2D,
  VL1_2D, VL1_2D, VL1_2D, VL1_2D, VL1_2D, VL1_2D, VL1_2D, VL1_2D,
  VR1_2D, VR1_2D, VR1_2D, VR1_2D, VR1_2D, VR1_2D, VR1_2D, VR1_2D,
  VR1_2D, VR1_2D, VR1_2D, VR1_2D, VR1_2D, VR1_2D, VR1_2D, VR1_2D,
  VR1_2D, VR1_2D, VR1_2D, VR1_2D, VR1_2D, VR1_2D, VR1_2D, VR1_2D,
  VR1_2D, VR1_2D, VR1_2D, VR1_2D, VR1_2D, VR1_2D, VR1_2D, VR1_2D,
   V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,
   V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,
   V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,
   V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,
   V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,
   V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,
   V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,
   V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,
   V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,
   V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,
   V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,
   V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,
   V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,
   V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,
   V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,
   V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D,  V0_2D
  };

  private static final byte decode_ext[] /* 32-entry groups */ =
  {
      /* Extension 0 (white) */
          74,     74,     74,     74,     74,     74,     74,     74,
          74,     74,     74,     74,     74,     74,     74,     74,
          75,     75,     75,     75,     75,     75,     75,     75,
          75,     75,     75,     75,     75,     75,     75,     75,
      /* Extension 1 (white) */
          76,     76,     76,     76,     76,     76,     76,     76,
          76,     76,     76,     76,     76,     76,     76,     76,
          77,     77,     77,     77,     77,     77,     77,     77,
          77,     77,     77,     77,     77,     77,     77,     77,
      /* Extension 2 (white) */
          78,     78,     78,     78,     78,     78,     78,     78,
          78,     78,     78,     78,     78,     78,     78,     78,
          79,     79,     79,     79,     79,     79,     79,     79,
          79,     79,     79,     79,     79,     79,     79,     79,
      /* Extension 3 (white) */
          80,     80,     80,     80,     80,     80,     80,     80,
          80,     80,     80,     80,     80,     80,     80,     80,
          81,     81,     81,     81,     81,     81,     81,     81,
          81,     81,     81,     81,     81,     81,     81,     81,
      /* Extension 4 (white) */
          82,     82,     82,     82,     82,     82,     82,     82,
          82,     82,     82,     82,     82,     82,     82,     82,
          83,     83,     83,     83,     83,     83,     83,     83,
          83,     83,     83,     83,     83,     83,     83,     83,
      /* Extension 5 (white) */
          84,     84,     84,     84,     84,     84,     84,     84,
          84,     84,     84,     84,     84,     84,     84,     84,
          85,     85,     85,     85,     85,     85,     85,     85,
          85,     85,     85,     85,     85,     85,     85,     85,
      /* Extension 6 (white) */
          86,     86,     86,     86,     86,     86,     86,     86,
          86,     86,     86,     86,     86,     86,     86,     86,
          87,     87,     87,     87,     87,     87,     87,     87,
          87,     87,     87,     87,     87,     87,     87,     87,
      /* Extension 7 (white) */
          88,     88,     88,     88,     88,     88,     88,     88,
          88,     88,     88,     88,     88,     88,     88,     88,
          90,     90,     90,     90,     90,     90,     90,     90,
          90,     90,     90,     90,     90,     90,     90,     90,
      /* Extension 8 (white) */
          91,     91,     91,     91,     94,     94,     95,     95,
          96,     96,     97,     97,     98,     98,     99,     99,
          92,     92,     92,     92,     93,     93,     93,     93,
         100,    100,    101,    101,    102,    102,    103,    103,
      /* Extension 9 (white) */
        EOLP,   EOLP,   EOLP,   EOLP,    UND,    UND,    UND,    UND,
         UND,    UND,    UND,    UND,    UND,    UND,    UND,    UND,
         UND,    UND,    UND,    UND,    UND,    UND,    UND,    UND,
         UND,    UND,    UND,    UND,    UND,    UND,   UNCW,   UNCB,

      /* Extension 10 (black) */
          20,     20,     20,     20,     34,     34,     35,     35,
          36,     36,     37,     37,     38,     38,     39,     39,
          21,     21,     21,     21,     42,     42,     43,     43,
           0,      0,      0,      0,      0,      0,      0,      0,
      /* Extension 11 (black) */
          15,     15,     15,     15,     15,     15,     15,     15,
          15,     15,     15,     15,     15,     15,     15,     15,
          65,     65,     66,     66,     26,     26,     27,     27,
          28,     28,     29,     29,     19,     19,     19,     19,
      /* Extension 12 (black) */
          23,     23,     23,     23,     50,     50,     51,     51,
          44,     44,     45,     45,     46,     46,     47,     47,
          57,     57,     58,     58,     61,     61,     67,     67,
          16,     16,     16,     16,     16,     16,     16,     16,
      /* Extension 13 (black) */
          17,     17,     17,     17,     17,     17,     17,     17,
          48,     48,     49,     49,     62,     62,     63,     63,
          30,     30,     31,     31,     32,     32,     33,     33,
          40,     40,     41,     41,     22,     22,     22,     22,
      /* Extension 14 (black) */
          18,     18,     18,     18,     18,     18,     18,     18,
          52,     52,     73,     74,     75,     76,     55,     55,
          56,     56,     83,     84,     85,     86,     59,     59,
          60,     60,     87,     88,     24,     24,     24,     24,
      /* Extension 15 (black) */
          25,     25,     25,     25,     89,     90,     68,     68,
          69,     69,     70,     70,     71,     72,     53,     53,
          54,     54,     77,     78,     79,     80,     81,     82,
          64,     64,     64,     64,     64,     64,     64,     64,
      /* Extension 16 (black) */
          91,     91,     91,     91,     94,     94,     95,     95,
          96,     96,     97,     97,     98,     98,     99,     99,
          92,     92,     92,     92,     93,     93,     93,     93,
         100,    100,    101,    101,    102,    102,    103,    103,
      /* Extension 17 (black) */
        EOLP,   EOLP,   EOLP,   EOLP,    UND,    UND,    UND,    UND,
         UND,    UND,    UND,    UND,    UND,    UND,    UND,    UND,
         UND,    UND,    UND,    UND,    UND,    UND,    UND,    UND,
         UND,    UND,    UND,    UND,    UND,    UND,   UNCW,   UNCB,

      /* Extension 18 (Black uncompressed) */
         8,        8,      8,      8,      8,      8,      8,      8,
         8,        8,      8,      8,      8,      8,      8,      8,
         9,        9,      9,      9,      9,      9,      9,      9,
        10,       10,     10,     10,     11,     11,  B11_U,  B11_U,

      /* Extension 19 (2-D) */
         UND,    UND,    UND,    UND,    UND,    UND,    UND,    UND,
         UND,    UND,    UND,    UND,    UND,    UND,    UND,    UND,
         UND,    UND,    UND,    UND,    UND,    UND,    UND,    UND,
        UNCW,   UNCW,   UNCW,   UNCW,   UNCB,   UNCB,   UNCB,   UNCB,
      /* Extension 20 (2-D and white uncompressed) */
        EOLP,   EOLP,   EOLP,   EOLP,    UND,    UND,    UND,    UND,
         UND,    UND,    UND,    UND,    UND,    UND,    UND,    UND,
         UND,    UND,    UND,    UND,    UND,    UND,    UND,    UND,
         UND,    UND,    UND,    UND,    UND,    UND,    UND,    UND
  };

  /* Filter Parameters */
  private int     cols;
  private int     rows;
  private int     k;
  private int     damagedRowsBeforeError;
  private boolean byteAlign;
  private boolean doRTC;
  private boolean doEOL;
  private boolean blackIs1;
  private boolean uncompressedMode;

  private int residue = 0;
  private int residueLen = 0;

  private int curRowNumber = 0;
  private boolean rowIs2D = false;
  private int runs1DIn2D = 0;
  private int appendedInput0s = 0;
  int totalDamagedRows = 0;
  int consecDamagedRows = 0;
//  private boolean pickyError = false;
  private static final boolean CCITTFaxDebugging = false;
    private int pauseBeyondRow = 0; // used only for debugging

  private int stoppedBecause = 0;
    private static final int STILL_RUNNING = 0;
    private static final int CAUSE_INPUT_UNAVAIL = 1;
    private static final int CAUSE_OUTPUT_OVERRUN = 2;
    private static final int CAUSE_UNDEFINED_CODE = 3;
    private static final int CAUSE_ILLEG_CODE = 4;
    private static final int CAUSE_DECODER_BUG = 5;
    private static final int CAUSE_EOL_PREFIX = 6;
    private static final int CAUSE_LINE_DONE = 7;
    private static final int CAUSE_RTC = 8;

  private int curMode = 0;
    private static final int NEXT_W1D = 0; // current color W/B carried in even/odd
    private static final int NEXT_B1D = 1;
    private static final int NEXT_WU = 2;
   // private static final int NEXT_BU = 3;
    private static final int NEXT_W2D = 4;
    //private static final int NEXT_B2D = 5;

  private static final TableBase tBases[] = {
    new TableBase(   BASE_WHITE,   0,  0),
    new TableBase(   BASE_BLACK, 256, 10),
    new TableBase( BASE_U_WHITE, 512, 20),
    new TableBase( BASE_U_BLACK, 512, 18),
    new TableBase(      BASE_2D, 768, 19),
    new TableBase(      BASE_2D, 768, 19)
  }; // indexed by curMode

  private byte [] curRow;
  private byte [] prevRow;
    // xRow entries are run lengths or partial run lengths, in the range
    // [0..127].  Even entry indices are white, odd indices are black.
    //   a) xRow[0] is 0 (preliminary 0-length white run).
    //   b) If i > 0 and xRow[i] and xRow[i + 1] are both zero,
    //      then xRow[0] + ... + xRow[i - 1] is params.cols.
    //   c) If i > 0 and xRow[i + 1] is 0 and xRow[0] + ... + xRow[i] < params.cols,
    //      then xRow[i] is 127.  (This keeps xRow as compact as possible.)
    //   d) If i is minimal such that xRow[0] + ... + xRow[i] is params.cols, then
    //      xRow[i + 1] and xRow[i + 2] are both defined and both zero.

  int curRowPos = 0;
  private int a0 = 0;
    // The number of pixels in the current line whose color is known.
  private int a0x = 0;
    // The number of pixels in the current line strictly before the
    // run of the current color that ends at a0.
  private int a0xx = 0;
    // The number of pixels in the current line strictly before the
    // run of the opposite color that ends at a0x.  a0xx is also the sum
    // of the entries in curRow[0]..curRow[curRowPos-1].
  private int b1 = 0;
    // The number of pixels strictly before the leftmost pixel of
    // a run of opposite color to curMode
    // (black if curMode is even, white if odd)
    // that lies to the right of a0.  As a special case,
    // if this is the first white run of the row
    // (a0 is 0 and curMode is even), b1 is 0 if the first run
    // of prevRow is black.
  private int db = 0;
    // b1 is prevRow[0] + ... + prevRow[db - 1].

  private boolean SufficientResidue(int len)
  {
    while ( residueLen < len )
    {
      // Get more input bits into residue
      int   newChar;

      if ( inCount <= inPos ) {
        if (!fillInputBuffer()) {
            newChar = -1;
        }
      else {
	        newChar = inBuf[inPos++] & 0xFF;
        }
      }
      else {
	    newChar = inBuf[inPos++] & 0xFF;
	  }
      if (newChar < 0) {
        newChar = 0;
        appendedInput0s += 8;
        if ( 32 < appendedInput0s )
        {
          pendingEOF = true;
          stoppedBecause = CAUSE_INPUT_UNAVAIL;
          return false;
        }
      }

      residue = (residue << 8) + newChar;
      residueLen += 8;
    }
    return true;
  }

  private void SetErrorCause(int cause)
  {
    stoppedBecause = cause;
  }

  private void SwitchCurRowColor()
  {
    // curMode and the (potentially 0-length) run ending at a0
    // are the same color.
    if ( a0x < a0 )
    {
      // Final run of color (curMode & 1) ending at a0 has positive length
      if ( a0xx < a0x )
      {
        // Prior run of opposite color ending at a0x has positive length.
        // Dump opposite-color run to the current row vector.
        // Chop into 127's as necessary to fit it into the byte type.
        while ( (a0xx + 128) <= a0x )
        {
          curRow[curRowPos] = 127;
          curRow[curRowPos + 1] = 0;
          a0xx += 127;
          curRowPos += 2;
        }
        curRow[curRowPos++] = (byte) (a0x - a0xx);
        a0xx = a0x;
      }
      else if ( a0xx == 0 ) // 0 == a0xx == a0x
      {
        // The run from [a0x..a0) is white/black if (curMode & 1) is 0/1.
        // The first positive-length run should appear in curRow[1] if black,
        // or curRow[2] if white.
        curRowPos = 2 - (curMode & 1);
      }
      a0x = a0;
    }
    else // a0x == a0
      a0x = a0xx;
    curMode ^= 1; // switch color
  }

  private void SwitchPrevRowColor(boolean rightward)
  {
    // b1 needs to point to the leftmost pixel of
    // a run of the opposite color.
    // The limited element size of prevRow
    // forces us to break runs into 127-pixel pieces,
    // so this move can involve several elements of prevRow.

    // Pre- and post-invariants:
    //   1) b1 is prevRow[0] + ... + prevRow[db - 1].
    //   2) If b1 is in the open interval (0..params.cols), prevRow[db] and
    //      prevRow[db - 1] are both positive.

    // Uses the xRow invariants that
    //   a) xRow[0] is 0, and for i > 0, and
    //   b) if xRow[i] and xRow[i + 1] are both zero, xRow[0] + ... + xRow[i - 1]
    //     is params.cols.

    if ( !rightward && (0 < b1) )
    {
      b1 -= prevRow[--db]; // leftward, always safe if 0 < b1
      while ( (prevRow[db - 1] == 0) && (0 < b1) )
      {
        db -= 2;
        b1 -= prevRow[db];
      }
    }
    else if ( b1 < cols )
    {
      b1 += prevRow[db++]; // rightward
      while ( (prevRow[db] == 0) && (b1 < cols) )
      {
        b1 += prevRow[db + 1];
        db += 2;
      }
    }
    else
    {
      db += (prevRow[db - 1] == 0) ? -1 : 1;
        // idle back and forth between the two zero-length runs at the end
    }
  }

  private void SynchPrevRowColor()
  {
    if ( (db ^ curMode) == 0 )
    {
      SwitchPrevRowColor(true);
        // Move b1 rightward to the first pixel of the next run of
        // opposite color in prevRow.
    }
  }

  private boolean LineContinues(boolean switchColor)
  {
    boolean result = true;
    if ( cols <= a0 )
    {
      if ( cols < a0 )
      {
        SetErrorCause(CAUSE_OUTPUT_OVERRUN);
        a0 = cols;
        result = false;
      }
      else if ( switchColor && !doEOL && (runs1DIn2D < 1) )
      {
        stoppedBecause = CAUSE_LINE_DONE;
        result = false;
      }
    }
    if ( switchColor )
      SwitchCurRowColor();
    return result;
  }

  private boolean UncompressedEscape(int color)
  {
    // Enter or exit uncompressed mode, depending on curMode
    if ( !uncompressedMode || (0 < runs1DIn2D) )
    {
      SetErrorCause(CAUSE_ILLEG_CODE);
      return false;
    }
    if ( !LineContinues((curMode & 1) != color) )
      return false;
    curMode =
      (
        ((curMode >>> 1) != (NEXT_WU >>> 1)) ?
        NEXT_WU :
        (rowIs2D ? NEXT_W2D : NEXT_W1D)
      ) + color;
    if ( NEXT_W2D <= curMode )
      SynchPrevRowColor();
    return true;
  }

  private void InitRow()
  {
    curRow[0] = 0; // preliminary white, must be 0
    curRow[1] = 0; // black
    a0xx = 0;
    a0x = 0;
    a0 = 0;
  }

  private void FinishRow()
  {
    a0 = cols;
    SwitchCurRowColor();
    while ( a0xx < cols )
    {
      // Force a0xx <- a0x <- a0
      if ( a0 == a0x )
        a0++;
      SwitchCurRowColor();
    }
    curRow[curRowPos++] = 0;
    curRow[curRowPos++] = 0;
  }

  private void MakeRowWhite()
  {
    InitRow();
    curMode = NEXT_W1D;
    FinishRow();
  }

  private void ExchangeRowPair()
  {
    byte [] tempLine = curRow;
    curRow = prevRow;
    prevRow = tempLine;
  }

  // Public contructors

    public CCITTFaxInputStream(
        InputStream in,
        int         inSize,
        int         outSize,
        FilterParams params )
    {
        super(in, inSize, outSize, 0, params);
        initFax();
    		//{{INIT_CONTROLS
		//}}
}

    public CCITTFaxInputStream(InputStream  in, FilterParams params)
    {
        this(in, 2048, 0, params);
    }

    public CCITTFaxInputStream(InputStream  in)
	{
		this(in, null);
	}

    private void initFax()
    {
        int width = 0;
        if (diparams.containsKey(FilterParams.Width_K))
            width = ((Integer)diparams.get(FilterParams.Width_K)).intValue();

        cols = 0;
        if (diparams.containsKey(FilterParams.Columns_K))
            cols = ((Integer)diparams.get(FilterParams.Columns_K)).intValue();

        rows = 0;
        if (diparams.containsKey(FilterParams.Rows_K))
            rows = ((Integer)diparams.get(FilterParams.Rows_K)).intValue();
		else if (diparams.containsKey(FilterParams.Height_K))
			rows = ((Integer)diparams.get(FilterParams.Height_K)).intValue();

        k = 0;
        if (diparams.containsKey(FilterParams.K_K))
            k = ((Integer)diparams.get(FilterParams.K_K)).intValue();

        damagedRowsBeforeError = 0;
        if (diparams.containsKey(FilterParams.DamagedRowsBeforeError_K))
            damagedRowsBeforeError = ((Integer)diparams.get(FilterParams.DamagedRowsBeforeError_K)).intValue();

        byteAlign = false;
        if (diparams.containsKey(FilterParams.ByteAlign_K))
            byteAlign = ((Boolean)diparams.get(FilterParams.ByteAlign_K)).booleanValue();

        doRTC = true;
        if (diparams.containsKey(FilterParams.DoRTC_K))
            doRTC = ((Boolean)diparams.get(FilterParams.DoRTC_K)).booleanValue();

        doEOL = false;
        if (diparams.containsKey(FilterParams.DoEOL_K))
            doEOL = ((Boolean)diparams.get(FilterParams.DoEOL_K)).booleanValue();

        blackIs1 = false;
        if (diparams.containsKey(FilterParams.BlackIs1_K))
            blackIs1 = ((Boolean)diparams.get(FilterParams.BlackIs1_K)).booleanValue();

        uncompressedMode = false;
        if (diparams.containsKey(FilterParams.UncompressedMode_K))
            uncompressedMode = ((Boolean)diparams.get(FilterParams.UncompressedMode_K)).booleanValue();

        if (cols < 1)
            cols = 1728;
        if (cols < width)
            cols = width;

        //recompute output buffer size
        int w = (cols + 7)/8;
        if (super.outBuf.length < w)
        	super.outBuf = new byte[w + 1];
        curRow = new byte[cols + 5];
        prevRow = new byte[cols + 5];
        MakeRowWhite();
    }


  // Public instance methods
  @Override
public void close() throws IOException
  {
    curRow = null;
    prevRow = null;
    super.close();
  }

  // See BufferedInputStream.java for rationale.

  @Override
public void fill()
  {
	  if(this.outBuf == null)
		  outBuf = markPos < 0 ? super.outBuf : new byte[super.outBuf.length];
    stoppedBecause = STILL_RUNNING;
    int consecEOLs = 0;
    int leading0s = 0;

    rowIs2D = (k < 0);
    
    if ( CCITTFaxDebugging )
    {
        int colsSoFar = 0;
        for ( int i = 0; i < (prevRow.length - 1); i++ )
        {
            if (
              (prevRow[i] < 0) ||
              ((0 < i) && (colsSoFar < cols) && (prevRow[i] == 0) && (prevRow[i+1] == 0)) ||
              (cols < colsSoFar)
            )
            {
                stoppedBecause = CAUSE_DECODER_BUG;
            }
            colsSoFar += prevRow[i];
        }
        if ( colsSoFar != cols )
        {
            stoppedBecause = CAUSE_DECODER_BUG;
        }
        if ( pauseBeyondRow <= curRowNumber )
        {
            int pointless = 1; // set breakpoint here
        }
    }

    outerLoop:
    for ( stoppedBecause = STILL_RUNNING;; )
    {
      int pickyEOLErrors = 0;
      boolean eolRequiredPreambleOK = 0 < consecDamagedRows;
      
      // If eolRequiredPreambleOK, we expect an arbitrary amount of data without
      // 11 consecutive 0-bits.  Following that, in any case
      // we expect an arbitrary amount of fill (0's), followed by
      // some number (at least one if eolRequiredPreambleOK) of EOL's
      // (11 0's followed by a 1, followed by a single tag
      // bit if 0 < k).
      
      eolLoop:
      for ( ;; )
      {
        if ( (residueLen < 1) && !SufficientResidue(1) )
          break eolLoop;
        if ( (residue & (0xFFFFFFFF >>> (32 - residueLen))) == 0 )
        {
          // The entire residue is 0's, consume it all.
          leading0s += residueLen;
          residueLen = 0;
        }
        else
        {
          // The residue contains at least one 1
          while ( ((residue >>> (residueLen - 1)) & 1) == 0 )
          {
            // Consume leading 0's one by one.
            leading0s++;
            residueLen--;
          }

          // There is at least one bit of residue, and
          // the residue begins with a 1.
          if ( leading0s < 11 )
          {
            // Not enough 0's for an EOL.
            if ( eolRequiredPreambleOK )
            {
              // Discard it all and try again.
              leading0s = 0;
              residueLen--;
            }
            else
            {
              // Not an EOL, so give it all back and continue.
              residueLen += leading0s;
              break eolLoop;
            }
          }
          else
          {
            // It's an EOL
            eolRequiredPreambleOK = false;
            residueLen -= 1;
            if ( 0 < k )
            {
              // It requires a following tag bit for 1-D(1) vs 2-D(0)
              if ( (residueLen < 1) && !SufficientResidue(1) )
                break outerLoop;
              if ( ((residue >>> (residueLen - 1)) & 1) == 0 )
                rowIs2D = true;
              residueLen--;
            }
            if ( 2 <= ++consecEOLs )
            {
              if ( (11 < leading0s) || ((0 < k) && rowIs2D) )
                pickyEOLErrors++;
              if ( (k < 0) || (6 <= consecEOLs) )
              {
                stoppedBecause = CAUSE_RTC;
                break outerLoop;
              }
            }
            leading0s = 0;
          }
        }
      } // eolLoop

//      if ( 0 < pickyEOLErrors )
//        pickyError = true;

      if ( doEOL )
      {
        if (
          (0 <= k) &&
          (0 < damagedRowsBeforeError) &&
          (3 <= consecEOLs) &&
          (pickyEOLErrors == 0)
        )
        {
          stoppedBecause = CAUSE_RTC;
          break outerLoop;
        }
        else if ( consecEOLs != 1 )
        {
          SetErrorCause(CAUSE_ILLEG_CODE);
          break outerLoop;
        }
      }

      if ( (consecEOLs == 0) && (0 < k) )
      {
        // No EOL, but tag bit needed to distinguish 1-D(1) from 2-D(0)
        if ( (residueLen < 1) && !SufficientResidue(1) )
          break outerLoop;
        if ( ((residue >>> (residueLen - 1)) & 1) == 0 )
          rowIs2D = true;
        residueLen--;
      }

      ExchangeRowPair();
      InitRow();

      db = 1; // index of the first possible black run in the previous line
      b1 = 0; // b1 is prevRow[0] + ... + prevRow[db - 1]

      curMode = rowIs2D ? NEXT_W2D : NEXT_W1D;
      runs1DIn2D = 0;

      codeLoop:
//      for ( int prevCodeWords /* for debugging */ = 0;; prevCodeWords++ )
      while(true)
      {
        if ( (residueLen < 8) && !SufficientResidue(8) )
          break outerLoop;

        // Decode the next 8 bits of input.
        int inputChunk = (residue >>> (residueLen - 8)) & 0xFF;
        int decodeIndex = tBases[curMode].decode + inputChunk;
        int code = decode[decodeIndex];
        if (
          (code < TERM_CODES) &&
          ((a0 + code) < cols) // fits in row with room to spare
        )
        {
          // Optimized case for codes up to 8-bits
          // long that append fixed-length
          // terminating (color-switching) runs.
          int priorRunLen;
          residueLen -= (encode[tBases[curMode].encode + code] & 0x0F);
          a0 += code;
          if ( (0 < runs1DIn2D) && (--runs1DIn2D == 0) )
            curMode ^= NEXT_W2D; // return to 2-D decoding
          if (
            (a0x < a0) && // this run has positive length,
            (0 < (priorRunLen = (a0x - a0xx))) &&
              // and prior run has positive length
            (priorRunLen < 127) // but not too long for a positive byte
          )
          {
            // optimized common case
            curRow[curRowPos++] = (byte) priorRunLen; // record prior run
            curMode ^= 1; // switch color
            a0xx = a0x; // shift this run to prior and append 0-length run
            a0x = a0;
          }
          else
            SwitchCurRowColor(); // do it more carefully
        }
        else
        {
          // general case
          if ( EXT <= code )
          {
            // Need 5 more input bits to decode, 13 total
            if ( (residueLen < 13) && !SufficientResidue(13) )
              break outerLoop;
            inputChunk = (residue >>> (residueLen - 13)) & 0x1F;
            decodeIndex = ((tBases[curMode].decodeExt + (code - EXT)) << 5) + inputChunk;
            code = decode_ext[decodeIndex];
          }
          residueLen -= (encode[tBases[curMode].encode + code] & 0x0F);

          if ( NEXT_W2D <= curMode )
          {
            // Invariants:
            //   1) db and curMode have opposite parity.
            //   2) b1 is prevRow[0] + ... + prevRow[db - 1].
            //   3) If a0 is 0, and curMode is even,
            //      b1 is no more than the length of the initial run of white pixels
            //      in prevRow.  Otherwise, b1 is not to the right of the first pixel
            //      in prevRow to the right of a0 and beginning a
            //      run of opposite color to curMode if such a pixel exists,
            //      or cols if not.

            while (
              (b1 < a0) || // b1 lies left of a0
              (
                (b1 < cols) &&
                (
                  (prevRow[db] == 0) ||
                    // isn't a real run, but rather a 0-pixel break in
                    // a run of color curMode
                  (
                    (b1 == 0) ?
                    ((curMode & 1) != 0) : // curMode is black
                      // a white run at the beginning of the row is
                      // the one special case where where b1 == a0 is legal
                    (
                      (b1 == a0) || // b1 must be > a0
                      (prevRow[db - 1] == 0)
                        // b1 lies in the middle of a long monochrome run
                        // that has been broken up by the 127-pixel limit on
                        // each element of prevRow
                    )
                  )
                )
              )
            )
            {
              b1 += prevRow[db] + prevRow[db + 1];
              db += 2;
            }

            // Additional invariants:
            //   4) If b1 is in the open interval (0..cols), prevRow[db] and
            //      prevRow[db - 1] are both positive.
            //   5) If a0 is 0, and curMode is even,
            //      b1 is the length of the initial run of white pixels in prevRow.
            //      Otherwise, b1 is the first pixel in prevRow to the right
            //      of a0 and beginning a run of opposite color to curMode if
            //      such a pixel exists, or cols if not.

            if ( code <= VR3_2D )
            {
              // Vertical code
              a0 = b1 + (code - V0_2D);
              if ( a0 < a0x )
              {
                // A negative run length is illegal (and, except if a0 == 0, a zero
                // run length would be pretty strange, but I suppose it would be
                // legal).
                SetErrorCause(CAUSE_ILLEG_CODE);
                break outerLoop;
              }
              if ( !LineContinues(true) )
                break outerLoop;

              SwitchPrevRowColor(VL2_2D < code);
                // if the code was VL2 or VL3, b1 might need to move leftward
              continue codeLoop;
            }
            else if (code == P_2D )
            {
              // Pass code, color doesn't change
              SwitchPrevRowColor(true);
              a0 = b1;
              SwitchPrevRowColor(true);
              if ( !LineContinues(false) )
                break outerLoop;
              continue codeLoop;
            }
            else if (code == H_2D)
            {
              // Horizontal code
              curMode ^= NEXT_W2D;
                // Switch to 1-D coding for two terminating (< 64 pixel) codes
              runs1DIn2D = 2;
              continue codeLoop;
            }
            // else fall through to very unusual codes
          }
          else if ( code < (TERM_CODES + MAKE_UP_CODES) )
          {
            boolean termRun = (code < TERM_CODES);
            if ( termRun )
            {
              a0 += code;
              if ( (0 < runs1DIn2D) && (--runs1DIn2D == 0) )
                curMode ^= NEXT_W2D; // return to 2-D coding
            }
            else if ( curMode <= NEXT_B1D )
              a0 += 64 * (code - (TERM_CODES - 1));
            else if ( curMode == NEXT_WU )
              a0 += 5; // uncompressed code was 000001, append 5 white pixels
            else // curMode == NEXT_BU
              a0 += 11; // uncompressed code was 11111111111, append 11 black pixels
            if ( !LineContinues(termRun) )
              break outerLoop;
            continue codeLoop;
          }

          switch ( code ) // very unusual codes
          {
            case EOLP:
              residueLen += 11; // add back the length of the EOL prefix
              if ( a0 != cols )
                SetErrorCause(CAUSE_ILLEG_CODE);
              else
                stoppedBecause = CAUSE_EOL_PREFIX;
              LineContinues(true);
              break outerLoop;

            case UNCW:
              if ( !UncompressedEscape(0) )
                break outerLoop;
              break;

            case UNCB:
              if ( !UncompressedEscape(1) )
                break outerLoop;
              break;

            default:
              SetErrorCause(CAUSE_UNDEFINED_CODE);
              break outerLoop;
          }
        }
      } // codeLoop: never terminates, never breaks
    } // outerLoop: never terminates, breaks once per row

    if ( (CAUSE_LINE_DONE <= stoppedBecause) && byteAlign )
      residueLen &= ~7; // discard remaining bits of input byte

    if ( CCITTFaxDebugging )
    {
        if ( stoppedBecause == CAUSE_DECODER_BUG )
        {
            pendingException =
              new PDFFilterFormatException("CCITTFaxDecode: Internal decoder error");
            return;
        }
    }
    switch ( stoppedBecause )
    {
      case CAUSE_RTC:
        pendingEOF = true;
      case CAUSE_INPUT_UNAVAIL:
        return;
    }

    if ( (++curRowNumber == rows) && !doRTC )
      pendingEOF = true;

    if (
      (CAUSE_EOL_PREFIX <= stoppedBecause) &&
      (!rowIs2D || (consecDamagedRows == 0))
    )
    {
      consecDamagedRows = 0;
      FinishRow();
    }
    else
    {
      if ( damagedRowsBeforeError <= totalDamagedRows++ || (k < 0) )
      {
        pendingException =
          new PDFFilterFormatException("CCITTFaxDecode: Too many damaged rows");
        return;
      }
      if ( consecDamagedRows++ < 1 )
        ExchangeRowPair(); // output the previous row again
      else
        MakeRowWhite(); // output a white row
    }

    int outBitPos = outPos << 3; // bit position
    int outBitPosLim = outBitPos + cols;
    for (
      curRowPos = (blackIs1 ? 1 : 0);
      outBitPos < outBitPosLim;
      curRowPos += 2
    )
    {
      // Append run of 1's
      int runLen = curRow[curRowPos];
      int bitsUsedInOutByte = outBitPos & 7;
      if ( bitsUsedInOutByte != 0 )
      {
        outBuf[outBitPos >>> 3] |= (0xFF >>> bitsUsedInOutByte);
        outBitPos += (8 - bitsUsedInOutByte);
        runLen -= (8 - bitsUsedInOutByte);
      }
      if ( 0 < runLen )
      {
        int outBytePos = outBitPos >>> 3;
        int runByteLen = (runLen + 7) >>> 3;
        outBitPos += (runByteLen << 3);
        runLen -= (runByteLen << 3);
        for ( ; 0 < runByteLen; runByteLen-- )
          outBuf[outBytePos++] = -1;
      }
      outBitPos += runLen;

      // Append run of 0's
      runLen = curRow[curRowPos + 1];
      bitsUsedInOutByte = outBitPos & 7;
      if ( bitsUsedInOutByte != 0 )
      {
        outBuf[outBitPos >>> 3] &= ~(0xFF >>> bitsUsedInOutByte);
        outBitPos += (8 - bitsUsedInOutByte);
        runLen -= (8 -bitsUsedInOutByte);
      }
      if ( 0 < runLen )
      {
        int outBytePos = outBitPos >>> 3;
        int runByteLen = (runLen + 7) >>> 3;
        outBitPos += (runByteLen << 3);
        runLen -= (runByteLen << 3);
        for ( ; 0 < runByteLen; runByteLen-- )
          outBuf[outBytePos++] = 0;
      }
      outBitPos += runLen;
    }

    // zero the unused bits in the final byte
    int bitsUsedInOutByte = outBitPos & 7;
    if ( bitsUsedInOutByte != 0 )
    {
      outBuf[outBitPos >>> 3] &= ~(0xFF >>> bitsUsedInOutByte);
      outBitPos += (8 - bitsUsedInOutByte);
    }

    outCount = outBitPos >>> 3;
    outPos = 0;
    
    if(markPos < 0){// stream is not marked so we don't need to preserve the data decoded so far.
    	super.outPos = this.outPos;
    	super.outCount = this.outCount;
    }else{
    	// Expand the parent's buffer and copy decoded data to that.
    	super.outCount+=this.outCount;
    	if(super.outBuf.length < super.outPos +  this.outCount){
    		byte newOutBuf[] = new byte[super.outPos +  this.outCount];
    		// copy the data decoded till previous run.
    		System.arraycopy(super.outBuf, 0, newOutBuf, 0, super.outPos);
    		super.outBuf = newOutBuf;
    	}
    	// copy the data decoded in this run.
    	System.arraycopy(this.outBuf, 0, super.outBuf, super.outPos, this.outCount);
    }
  }
	//{{DECLARE_CONTROLS
	//}}
  
  @Override
  protected boolean increaseBufferSize(long markLimitPlusHeadroom){
	  return false;// we are handling it in fill()
	}
  
  /**
   * Keeping some of the parameters exactly as in parent class because this filter's implementation is written in a way
   * that we have to start with outPos = 0 in the buffer for each row. So if the stream is marked we have to keep copying the rows
   * to parent buffer and we shall use this buffer as current row.
   */
  private byte    outBuf[] = null;
  private int     outCount = 0;   // total valid bytes in output buf
  private int     outPos = 0;
}

class TableBase
{
  int encode;
  int decode;
  int decodeExt;

  TableBase(int encode, int decode, int decodeExt)
  {
    this.encode = encode;
    this.decode = decode;
    this.decodeExt = decodeExt;
  		//{{INIT_CONTROLS
		//}}
}
	//{{DECLARE_CONTROLS
	//}}
}

