/*
 * File: LZWOutputStream.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.filter;

import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.OutputStream;

/**
 * LZWOutputStream
 */
public class LZWOutputStream extends FilterOutputStream
{
	// Variables
	private LZWEngine lzw;

	// Constructors
	/**
	 * Creates an output stream filter.
	 * @param out	the output stream
	 */
	public LZWOutputStream(OutputStream  out, FilterParams p)
	{
		super(out);
		lzw = new LZWEngine(out);
		if (p != null) {
			if (p.containsKey(FilterParams.EarlyChange_K))
				lzw.setEarlyChange(((Integer)p.get(FilterParams.EarlyChange_K)).intValue());
			}
	}

	public LZWOutputStream(OutputStream  out)
	{
		this(out, null);
	}

	// Methods
	/**
	 * Writes a byte. Will block until the byte is actually
	 * written.
	 * @param b the byte
	 * @exception IOException If an I/O error has occurred.
	 */
	@Override
	public void write(int b)
		throws IOException
	{
		lzw.put(b & 0x0ff);
	}

	/**
	 * Writes a subarray of bytes.
	 * @param b	the data to be written
	 * @param off	the start offset in the data
	 * @param len	the number of bytes that are written
	 * @exception IOException If an I/O error has occurred.
	 */
	@Override
	public void write(byte  b[], int  off, int  len)
		throws IOException
	{
		int	maxWrite = b.length-off;

		if (maxWrite > len)
			maxWrite = len;

		while (maxWrite-- > 0) {
			write(b[off++] & 0x0ff);
		}
	}

	/**
	 * Writes an array of bytes. Will block until the bytes
	 * are actually written.
	 * @param b	the data to be written
	 * @exception IOException If an I/O error has occurred.
	 */
	@Override
	public void write(byte  b[])
		throws IOException
	{
		write(b, 0, b.length);
	}

	/**
	 * Closes the stream and writes the stream trailer.
	 * This method must be called to release any resources
	 * associated with the stream.
	 * @exception IOException If an I/O error has occurred.
	 */
	@Override
	public void close()
		throws IOException
	{
		lzw.put(-1);
		super.close();
	}

	/**
	 * Counts the number of bytes written by this filter.
	 * @return	actual number of bytes written
	 */
	public long getTotalOut()
	{
		return lzw.getTotalOut();
	}
}
