/* ****************************************************************************
 *
 *	File: EnableEncryption.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/
package com.adobe.internal.pdftoolkit.core.securityframework.impl;

import java.util.Map;

import com.adobe.internal.io.stream.InputByteStream;
import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityAuthorizationException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityConfigurationException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityException;
import com.adobe.internal.pdftoolkit.core.securityframework.DecryptedState;
import com.adobe.internal.pdftoolkit.core.securityframework.EncryptionHandler;
import com.adobe.internal.pdftoolkit.core.securityframework.SecurityHandler;
import com.adobe.internal.pdftoolkit.core.securityframework.SecurityManager;

/**
 * Provides the interface to the internal implementation of encryption and security in 
 * <I>PDF Java Toolkit</I>.
 */

public interface EnableEncryption
{
	/**
	 * 
	 * Sets the Security Manager used for encryption-related actions
	 * @param securityMgr - Security Manager
	 * @param encryptParams - Encryption Parameters
	 * @throws PDFSecurityException
	 */
	void setDecryptionSecurityManager(SecurityManager securityMgr, Map encryptParams, byte[] id)
    throws PDFSecurityException;

	/**
	 * 
	 */
	void setEncryptionSecurityManager(SecurityManager securityMgr, Map encryptParams, byte[] id);

	/**
	 * 
	 */
	void resetEncryptionSecurityManager();

	/**
	 * 
	 */
	void resetDecryptionSecurityManager();

	/**
	 * 
	 */
	SecurityManager getEncryptionSecurityManager();

	/**
	 * 
	 */
	SecurityManager getDecryptionSecurityManager();

	/**
	 * 
	 */
	Map getEncryptionParameters() throws PDFSecurityAuthorizationException;

	/**
	 * 
	 */
	Map getDecryptionParameters() throws PDFSecurityAuthorizationException;

	/**
	 * 
	 * Authenticates all security handlers specified by the encryption
	 * parameters associated with the stored security manager
	 * @param handler {@link SecurityHandler} to authenticate
	 * @param decryptedState Set null if password or credentials are used for 
	 *                       decryption, else set a valid decryption state.
	 * @throws PDFSecurityAuthorizationException
	 * @throws PDFSecurityConfigurationException
	 */
	boolean authenticateSecurity(SecurityHandler handler, DecryptedState decryptedState)
	throws PDFSecurityAuthorizationException, PDFSecurityConfigurationException;
	
	/**
	 * Authenticates specified security handler
	 * @param params
	 * @param docID
	 * @param handler
	 * @param decryptedState
	 * @return boolean
	 * @throws PDFSecurityAuthorizationException
	 * @throws PDFSecurityConfigurationException
	 */
	boolean authenticateSecurity(Map params, byte[] docID, SecurityHandler handler, DecryptedState decryptedState)
	throws PDFSecurityAuthorizationException, PDFSecurityConfigurationException;

	/**
	 * 
	 * Encrypts CosString's content
	 * @param content byte sequence of the CosString's content
	 * @param addKey CosString-object-specific addition to the encryption key
	 * 					associated with the particular security handler
	 * @return encrypted byte sequence
	 * @throws PDFSecurityException
	 */
	byte[] encryptString(byte[] content, byte[] addKey)
	throws PDFSecurityException;

	/**
	 * 
	 * Decrypts CosString's content
	 * @param content byte sequence of the CosString's content
	 * @param addKey CosString-object-specific addition to the encryption key
	 * 					associated with the particular security handler
	 * @return - decrypted byte sequence
	 * @throws PDFSecurityException
	 */
	byte[] decryptString(byte[] content, byte[] addKey)
	throws PDFSecurityException;

	/**
	 * 
	 * Decrypts CosStream's content
	 * @param src source stream to decrypt
	 * @param dest destination for the decrypted stream
	 * @param addKey CosStream-object-specific addition to the encryption key
	 * 					associated with the particular security handler
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	void decryptStream(String cryptFilter, InputByteStream src, OutputByteStream dest, byte[] addKey)
	throws PDFSecurityException, PDFIOException;

	/**
	 * 
	 * Encrypts CosStream's content
	 * @param src source stream to encrypt
	 * @param dest destination for the encrypted stream
	 * @param addKey CosStream-object-specific addition to the encryption key
	 * 					associated with the particular security handler
	 * @throws PDFSecurityException
	 * @throws PDFIOException
	 */
	void encryptStream(String cryptFilter, InputByteStream src, OutputByteStream dest, byte[] addKey)
	throws PDFSecurityException, PDFIOException;
	
	/**
	 * 
	 * 
	 * @param name
	 * @return Encryption Handler for stream encryption with the specified name
	 */
	EncryptionHandler getStreamEncryptionHandler(String name)
	throws PDFSecurityConfigurationException, PDFSecurityAuthorizationException;
	
	/**
	 * 
	 * 
	 * @return Encryption Handler for string encryption with the specified name
	 */
	EncryptionHandler getStringEncryptionHandler()
    throws PDFSecurityConfigurationException, PDFSecurityAuthorizationException;

	/**
	 * 
	 * 
	 * @param name
	 * @return Encryption Handler for stream decryption with the specified name
	 */
	EncryptionHandler getStreamDecryptionHandler(String name)
    throws PDFSecurityConfigurationException, PDFSecurityAuthorizationException;

	/**
	 * 
	 * 
	 * @return Encryption Handler for string decryption with the specified name
	 */
	EncryptionHandler getStringDecryptionHandler()
    throws PDFSecurityConfigurationException, PDFSecurityAuthorizationException;

	/**
	 * Returns SecurityHandler which has been used to authenticate security. If this 
	 * process is still not done, then this method returns null.
	 * @return {@link SecurityHandler}
	 */
	SecurityHandler getSecurityHandler();

	
}
