/***************************************************************************/
/*                                                                         */
/*                      ADOBE CONFIDENTIAL                                 */
/*                      _ _ _ _ _ _ _ _ _ _                                */
/*                                                                         */
/*  Copyright 2006, Adobe Systems Incorporated                             */
/*  All Rights Reserved.                                                   */
/*                                                                         */
/*  NOTICE: All information contained herein is, and remains the property  */
/*  of Adobe Systems Incorporated and its suppliers, if any.  The          */
/*  intellectual and technical concepts contained herein are proprietary   */
/*  to Adobe Systems Incorporated and its suppliers and may be coveredby   */
/*  U.S. and Foreign Patents, patents in process, and are protected by     */
/*  trade secret or copyright law.  Dissemination of this information or   */
/*  reproduction of this material is strictly forbidden unless prior       */
/*  written permission is obtained from Adobe Systems Incorporated.        */
/*                                                                         */
/***************************************************************************/

package com.adobe.versioncue.nativecomm;

import java.io.File;

import com.adobe.versioncue.internal.nativecomm.IMonitor;
import com.adobe.versioncue.internal.nativecomm.NativeService;
import com.adobe.versioncue.internal.nativecomm.ServiceBuilder;
import com.adobe.versioncue.internal.nativecomm.ServiceConfig;
import com.adobe.versioncue.internal.nativecomm.io.TCPConnection;
import com.adobe.versioncue.nativecomm.io.IConnection;
import com.adobe.versioncue.nativecomm.io.IConnectionFactory;

/**
 * @author <a href="mailto:tnaroska@adobe.com">tnaroska</a>
 * @version $Revision: #1 $
 * @since Mar 8, 2006
 */
public final class NativeComm
{
	// -------------------------------------------------------------------------- Public Enumeration

	/** Host operating system enum */
	public enum OS
	{
		/** Windows 32-bit operating system */
		win32(),
		/** Linux operating system */
		linux(),
		/** MacOSX operating system */
		macosx(),
		/** Solaris/SunOS operating system */
		solaris(),
		/** unknown operating system */
		unknown();

		/** current host operating system */
		public static final OS current;

		/** @return true if running on windows operating system */
		public static boolean isWin32()
		{
			return current == win32;
		}

		/** @return true if running on linux operating system */
		public static boolean isLinux()
		{
			return current == linux;
		}

		/** @return true if running on macosx operating system */
		public static boolean isMacOSX()
		{
			return current == macosx;
		}

		/** @return true if running on solaris operating system */
		public static boolean isSolaris()
		{
			return current == solaris;
		}

		static
		{
			final String os = System.getProperty("os.name").toLowerCase();
			if (os.startsWith("mac os x"))
			{
				current = macosx;
			}
			else if (os.startsWith("win"))
			{
				current = win32;
			}
			else if (os.startsWith("linux"))
			{
				current = linux;
			}
			else if (os.startsWith("sunos"))
			{
				current = solaris;
			}
			else
			{
				current = unknown;
			}
		}
	}

	// -------------------------------------------------------------------- Private Static Variables

	/** SeriveConfig holding default values. */
	private static ServiceConfig defaults = new ServiceConfig("defaults");

	static
	{
		String defaultCommand;
		switch (OS.current)
		{
		case win32:
			defaultCommand = "VCNativeHelper.exe";
			break;
		default:
			defaultCommand = "VCNativeHelper";
			break;
		}
		defaults.command(defaultCommand);

		final File tmpDir = new File(System.getProperty("java.io.tmpdir"));
		defaults.tmpDir(tmpDir);

		defaults.logDir(new File("logs").getAbsoluteFile());
	}

	// ----------------------------------------------------------------------- Public Static Methods

	/**
	 * @param id unique service id
	 * @return IServiceBuilder object
	 */
	public static IServiceBuilder serviceBuilder(String id)
	{
		ServiceConfig cfg = defaults.clone().id(id);
		return new ServiceBuilder(cfg);
	}

	/**
	 * HACKY WHACKY - quick way to connect a running service on a port
	 * 
	 * @param id - arbitrary id
	 * @param port - TCP port to connect
	 * @return INativeService instance
	 */
	public static INativeService service(String id, final int port)
	{
		final ServiceConfig cfg = new ServiceConfig(id);
		cfg.maxConnections(2);

		IConnectionFactory factory = new IConnectionFactory()
		{
			public IConnection[] connect(INativeService service) throws NativeCommException
			{
				TCPConnection con = new TCPConnection(cfg, port);
				return new IConnection[] { con };
			}
		};

		return new NativeService(cfg, factory, IMonitor.DUMMY);
	}

	// ----------------------------------------------------------------------- Public Static Methods

	/**
	 * Returns the configured default executable
	 * 
	 * @return the configured default executable
	 */
	public static String getDefaultCommand()
	{
		return defaults.command();
	}

	/**
	 * Returns the configured default temp directory
	 * 
	 * @return the configured default temp directory
	 */
	public static File getDefaultTmpDir()
	{
		return defaults.tmpDir();
	}

	/**
	 * Returns the configured default log directory
	 * 
	 * @return the configured default log directory
	 */
	public static File getDefaultLogDir()
	{
		return defaults.logDir();
	}

	/**
	 * Returns the default maximum connection pool size
	 * 
	 * @return the default maximum connection pool size
	 */
	public static int getDefaultMaxConnections()
	{
		return defaults.maxConnections();
	}

	/**
	 * Returns the default maximum number of requests per process.
	 * 
	 * @return the default maximum number of requests per process (0 == unlimited)
	 */
	public static int getDefaultMaxRequests()
	{
		return defaults.maxRequests();
	}

	/**
	 * Returns the default request timeout in milliseconds.
	 * @return the default request timeout in milliseconds
	 */
	public static int getDefaultRequestTimeout()
	{
		return defaults.requestTimeout();
	}

	/** @return default maximum number of request retries; 0 == do not retry */
	public static int getDefaultRequestRetries()
	{
		return defaults.requestRetries();
	}

	/** @return the default maximum time to wait for a spawned child process' connection. */
	public static int getDefaultLaunchTimeout()
	{
		return defaults.launchTimeout();
	}


	/**
	 * Sets the default executable
	 * 
	 * @param command default executable
	 */
	public static void setDefaultCommand(String command)
	{
		defaults.command(command);
	}

	/**
	 * Sets the default temp directory
	 * 
	 * @param tmpDir default temp directory
	 */
	public static void setDefaultTmpDir(File tmpDir)
	{
		defaults.tmpDir(tmpDir);
	}

	/**
	 * Sets the default log directory
	 * 
	 * @param logDir default log directory
	 */
	public static void setDefaultLogDir(File logDir)
	{
		defaults.logDir(logDir);
	}

	/**
	 * Sets the default maximum connection pool size
	 * 
	 * @param maxConnections default maximum connection pool size
	 */
	public static void setDefaultMaxConnections(int maxConnections)
	{
		defaults.maxConnections(maxConnections);
	}

	/**
	 * Sets the default maximum number of requests per process.
	 * 
	 * @param maxRequests default maximum number of requests per process (0 == unlimited)
	 */
	public static void setDefaultMaxRequests(int maxRequests)
	{
		defaults.maxRequests(maxRequests);
	}

	/** Sets the default request timeout in milliseconds
	 * @param timeout default request timeout in milliseconds; 0 == no timeout
	 */
	public static void setDefaultRequestTimeout(int timeout)
	{
		defaults.requestTimeout(timeout);
	}

	/**
	 * Sets the default maximum number of request retries
	 * @param retries default maximum number of request retries; 0 == do not retry
	 */
	public static void setDefaultRequestRetries(int retries)
	{
		defaults.requestRetries(retries);
	}

	/**
	 * Sets the default maximum time to wait for a spawned child process' connection.
	 * @param timeout default maximum time to wait for a spawned child process' connection.
	 */
	public static void setDefaultLaunchTimeout(int timeout)
	{
		defaults.launchTimeout(timeout);
	}

	// ------------------------------------------------------------------------- Private Constructor

	/** Private ctor */
	private NativeComm()
	{
		// empty
	}
}