/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa;


import java.io.OutputStream;
import java.io.IOException;

import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.ResId;


/**
 * A class to represent the XML comment nodes in the DOM.
 *
 * <p>A comment's data  is all of the characters between
 * the starting <code>&lt;!--</code> and ending <code>--&gt;</code>.
 */
public final class Comment extends Node {

	private final static String mClassName = "comment";

	private final String mComment;

	/**
	 * The mDoc member will be populated if this is a document-level comment.
	 */
	private final Document mDoc;


	/**
	 * @exclude from published api.
	 */
	Comment(Document doc, String comment) {
		super(null, null);
		mComment = comment;
		mDoc = doc;
		if (doc != null)
			mDoc.appendChild(this);
	}

	/**
	 * Instantiates a node with the given comment.
	 * @param parent the node's parent, if any.
	 * @param prevSibling the node's previous sibling, if any.
	 * @param comment the node's comment.
	 */
	public Comment(Element parent, Node prevSibling, String comment) {
		super(parent, prevSibling);
		mComment = comment;
		mDoc = null;
	}

	/**
	 * @exclude from published api.
	 */
	public Node clone(Element parent) {
		if (parent != null)
			return new Comment(parent, parent.getLastXMLChild(), mComment);
		else
			return new Comment(null, null, mComment);
	}

	/**
	 * Gets this node's atomic class name.
	 * @return the atomic class name.
	 *
	 * @exclude from published api.
	 */
	public String getClassAtom() {
		return mClassName;
	}

	/**
	 * @exclude from published api.
	 */
	public String getClassName() {
		return mClassName;
	}

	/**
	 * @see Node#getData()
	 */
	public String getData() {
		return mComment;
	}

	/**
	 * Gets this node's name.
	 * @return the comment name which is the constant value "#comment".
	 */
	public String getName() {
		return STRS.COMMENTNAME;
	}

	/**
	 * @exclude from published api.
	 */
	public boolean isLeaf() {
		return true;
	}

	/**
	 * @exclude from published api.
	 */
	public void postSave() {
		// do nothing
	}

	/**
	 * @exclude from published api.
	 */
	public void preSave(boolean bSaveXMLScript) {
		// do nothing
	}
	
	/**
	 * @exclude from published api.
	 */
	public void serialize(OutputStream outStream, DOMSaveOptions options, int level, Node prevSibling) throws IOException {
		
		if (level != 0 || prevSibling != null) {
			if (options.getDisplayFormat() == DOMSaveOptions.PRETTY_OUTPUT)
				options.writeIndent(outStream, level);
			else if (level == 0 && (options.getFormatOutside() || options.getDisplayFormat() == DOMSaveOptions.SIMPLE_OUTPUT))
				outStream.write(Document.MarkupReturn);
		}
		outStream.write(Document.MarkupCommentStart);
		String sValue = getData();
		outStream.write(sValue.getBytes(Document.Encoding));
		outStream.write(Document.MarkupCommentEnd);
	}

	/**
	 * @exclude from published api.
	 */
	public void setScriptProperty(String sPropertyName, Arg propertyValue) {
		throw new ExFull(ResId.UNSUPPORTED_OPERATION, "Comment#setScriptProperty");
	}

}
