/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa.template.formatting;

import com.adobe.xfa.AppModel;
import com.adobe.xfa.Attribute;
import com.adobe.xfa.Element;
import com.adobe.xfa.EnumAttr;
import com.adobe.xfa.Measurement;
import com.adobe.xfa.Node;
import com.adobe.xfa.ProtoableNode;
import com.adobe.xfa.XFA;
import com.adobe.xfa.template.containers.Draw;
import com.adobe.xfa.template.containers.Field;
import com.adobe.xfa.ut.Assertions;
import com.adobe.xfa.ut.ObjectHolder;
import com.adobe.xfa.ut.UnitSpan;

/**
 * An element that describes the margins of a container object.
 *
 * @exclude from published api -- Mike Tardif, May 2006.
 */

public final class Margin extends ProtoableNode {
	
	private static class CachedInfo {
		boolean mbThicknessesValid; // = false;
		final UnitSpan moThicknesses[] = new UnitSpan[4];
	}
	
	public Margin(Element parent, Node prevSibling) {
		super(parent, prevSibling, null, XFA.MARGIN, XFA.MARGIN, null,
				XFA.MARGINTAG, XFA.MARGIN);
	}

	public Attribute defaultAttribute(int eTag) {
		return defaultAttributeImpl(eTag, null);
	}
	
	private Attribute defaultAttributeImpl(int eTag, CachedInfo oCachedInfo) {
		// Implement the special rules around caption and widget margins. See
		// the following proposals for more information.
		//		
		// http://xtg.can.adobe.com/twiki/bin/view/XFA/WidgetMarginsXFALangProposal
		// http://xtg.can.adobe.com/twiki/bin/view/XFA/CaptionMarginsXFALangProposal
		//		
		if (eTag == XFA.LEFTINSETTAG || eTag == XFA.TOPINSETTAG
				|| eTag == XFA.RIGHTINSETTAG || eTag == XFA.BOTTOMINSETTAG) {
			Element oParent = getXFAParent();

			if ((oParent != null)
					&& (oParent.getClassTag() == XFA.TEXTEDITTAG
							|| oParent.getClassTag() == XFA.NUMERICEDITTAG
							|| oParent.getClassTag() == XFA.DATETIMEEDITTAG
							|| oParent.getClassTag() == XFA.SIGNATURETAG
							|| oParent.getClassTag() == XFA.PASSWORDEDITTAG 
							|| oParent.getClassTag() == XFA.CHOICELISTTAG)) {
				//If this function is called it means we need to return the default margin value.
				//If <margin/> was specified we return an auto calculated value, see links above.
				//Otherwise the omission of a <margin/> element (re: we are a flagged as default) means the margin is considered [0, 0, 0, 0]
				boolean bAutoCalculateMargins = !isDefault(true);
				if (bAutoCalculateMargins) {
					// The presence of an empty <margin/> element is an instruction
					// to the processing application that it should
					// automatically
					// inset the content of the widget according to the
					// following
					// Rules:

					// 1. If there is no border on the widget, the margin shall
					// be 0,0,0,0
					// 2. If there is a border on the widget, the margin shall
					// be
					// calculated using the same insetting formula as Acrobat 6.
					// For Acrobat 6, the inset is twice the border thickness
					// and
					// the top inset is the descent of the font.
					Element poBorder = oParent.getElement(XFA.BORDERTAG, true,
							0, false, false);

					// Watson 1136189 TODO Should we check for all four edges
					// being visible before
					// we do of this? Or will that just slow us down for a few
					// fringe cases?
					// This code is supposed to mimic Acrobat widget formatting,
					// and Acrobat only
					// allows all four edges or none at all.

					// only apply the fix for watson bug 1136189 for Designer 7+ docs.
					
					if (poBorder != null) {
						boolean bIgnorePresence = getAppModel().getLegacySetting(AppModel.XFA_LEGACY_POSITIONING);
						int eBorderPresence = bIgnorePresence ? EnumAttr.PRESENCE_VISIBLE : poBorder.getEnum(XFA.PRESENCETAG);
						if (eBorderPresence != EnumAttr.PRESENCE_HIDDEN && eBorderPresence != EnumAttr.PRESENCE_INACTIVE) {
							// It it's a 3D border, then we need to multiply by
							// 2 as a
							// 3D border consists of 2 strokes to give the 3D
							// effect.
							boolean bIs3D = is3D(poBorder);
							boolean bParentHasComb = oParent.isSpecified(XFA.COMBTAG, true, 0);

							UnitSpan oThickness = UnitSpan.ZERO;
							int nEdge = 0;
							if (eTag == XFA.TOPINSETTAG)
								nEdge = 0;
							else if (eTag == XFA.RIGHTINSETTAG)
								nEdge = 1;
							else if (eTag == XFA.BOTTOMINSETTAG)
								nEdge = 2;
							else if (eTag == XFA.LEFTINSETTAG)
								nEdge = 3;
							
							// Note that we always start at the requested edge.  If we're caching (poCachedInfo not NULL)
							// then we'll populate from nEdge up to and including the 4th edge (no need to populate
							// any edges lower than nEdge because caching is tightly coupled with getInsets, which
							// calls in the 0,1,2,3 order).
							// If not caching (oCachedInfo NULL), then we will return out of this loop on the
							// first iteration (i.e. it won't really loop).
							for (int nCacheLoop = nEdge; nCacheLoop < 4; nCacheLoop++)
							{
								oThickness = UnitSpan.ZERO;	// Reset on each loop.
								
								// 2* the border thickness of edge.
								Element poEdge = poBorder.getElement(
										XFA.EDGETAG, true, nCacheLoop, true, false);

								// Watson 1136189 If the edge is hidden/inactive, do not use it's thickness
								int ePresence = bIgnorePresence ? EnumAttr.PRESENCE_VISIBLE : poEdge.getEnum(XFA.PRESENCETAG);
								if (ePresence != EnumAttr.PRESENCE_HIDDEN && ePresence != EnumAttr.PRESENCE_INACTIVE) {
									oThickness = ((Measurement) poEdge
												.getAttribute(XFA.THICKNESSTAG))
												.getUnitSpan();
									if (!bParentHasComb) // this border processing is not for combs
										oThickness = oThickness.multiply(2.0f);

									if (bIs3D)
										oThickness = oThickness.multiply(2.0f);

									if (nCacheLoop == 0) {
										// eTag == XFA.TOPINSETTAG
										// Need to get descent of the font. Since we
										// don't have access to the
										// font service we will make an
										// approximation based on the font size.
										// Kludgey, but deemed to be acceptable.
										// Based on a sampling of a number
										// of fonts, the descent ranges between
										// 150-300pt for a 1000pt font, so
										// we will approximate it at 1/4 or and
										// average of a 225pt descent for
										// a 1000pt font.
										Node poUI = oParent.getXFAParent();
										if (poUI != null) {
											Element poContainer = poUI.getXFAParent();
											if (poContainer instanceof Field || poContainer instanceof Draw) {
												Element poFont = poContainer.getElement(XFA.FONTTAG,
																						true, 0, true,
																						false);
												UnitSpan oFontSize = ((Measurement) poFont.getAttribute(XFA.SIZETAG)).getUnitSpan();
												oFontSize = oFontSize.divide(4);
												oThickness = oThickness.add(oFontSize);
											}
										}
									}
								}


								// If we're not caching, there's only a single iteration of this loop, with the value nEdge.
								if (oCachedInfo == null)
									return new Measurement(oThickness);

								oCachedInfo.mbThicknessesValid = true;
								oCachedInfo.moThicknesses[nCacheLoop] = oThickness;
							}
							// Must be in the cache case (oCachedInfo non NULL) since the above loop
							// returns on the first iteration otherwise.								
							return new Measurement(oCachedInfo.moThicknesses[nEdge]);
						}// endif (eBorderPresence != EnumAttr.PRESENCE_HIDDEN && eBorderPresence != EnumAttr.PRESENCE_INACTIVE)
					}
				}
			} else if (oParent != null
					&& oParent.getClassTag() == XFA.CAPTIONTAG) {
				// Is there a UI widget that exerts a content margin?
				// If there is then we use the corresponding inset from
				// the UI margin, subject to the captions placement
				// attribute described in the table below.
				//				

				// We are only concerned with fields or draws
				Element poContainer = oParent.getXFAParent();
				if (poContainer instanceof Field || poContainer instanceof Draw) {
					// Get the UI
					Element poUI = poContainer.getElement(XFA.UITAG, true, 0,
							false, false);
					if (poUI != null) {
						// Is it a UI that can have a margin?
						Element poCurrentUI = (Element) poUI.getOneOfChild(
								true, false);
						if ((poCurrentUI != null)
								&& (poCurrentUI.getClassTag() == XFA.TEXTEDITTAG
										|| poCurrentUI.getClassTag() == XFA.NUMERICEDITTAG
										|| poCurrentUI.getClassTag() == XFA.DATETIMEEDITTAG
										|| poCurrentUI.getClassTag() == XFA.SIGNATURETAG
										|| poCurrentUI.getClassTag() == XFA.PASSWORDEDITTAG || poCurrentUI
										.getClassTag() == XFA.CHOICELISTTAG)) {
							// Do we have a margin?
							Element poMargin = poCurrentUI.getElement(
									XFA.MARGINTAG, true, 0, false, false);
							if (poMargin != null) {
								// Wow, we made it! So lets get the placement
								// and return
								// the appropriate attribute based on the table
								// below:
								//
								// Placement UI Margin (Insets)
								// Top Left Right Bottom
								// --------------------------------------------
								// left X X
								// right X X
								// top X X
								// bottom X X
								//

								int ePlacement = oParent.getEnum(XFA.PLACEMENTTAG);

								if (((ePlacement == EnumAttr.PLACEMENT_LEFT || ePlacement == EnumAttr.PLACEMENT_RIGHT) && (eTag == XFA.TOPINSETTAG || eTag == XFA.BOTTOMINSETTAG))
										|| ((ePlacement == EnumAttr.PLACEMENT_TOP || ePlacement == EnumAttr.PLACEMENT_BOTTOM) && (eTag == XFA.LEFTINSETTAG || eTag == XFA.RIGHTINSETTAG))) {
									return poMargin.getAttribute(eTag);
								}
							}
						}
					}
				}
			}
		}

		// Not special case, so let the base class handle it
		return super.defaultAttribute(eTag);
	}

	boolean is3D(Element oNode) {
		//
		// Note: If you change this code, change the is3D method
		// in devicedriverImpl.cpp to match.
		//

		// Peek at the edge/corner properties of oNode. The node could be one of
		// border, arc or rectangle. The only thing that makes sense
		// for something to be 3D is if all of it's specified edges and
		// corners have the same 3D stroke type.
		// Note: You cannot simply query for oNode's children, since the
		// corner/edge properties
		// may exist on a proto'd node.
		boolean bFailed = false;
		boolean bFirst = true;
		int eEdgeStroke = EnumAttr.STROKE_SOLID;
		int eCurrentStroke = eEdgeStroke;
		int ePresence = EnumAttr.PRESENCE_VISIBLE;

		for (int i = 0; i < 4 && !bFailed; i++) {
			Element oEdge = oNode.peekElement(XFA.EDGETAG, false, i);
			Element oCorner = oNode.peekElement(XFA.CORNERTAG, false, i);
			Element oTest = oEdge; // start with edge
			for (int j = 0; j < 2; j++) {
				if (oTest != null) {
					ePresence = oTest.getEnum(XFA.PRESENCETAG);
					if (ePresence == EnumAttr.PRESENCE_VISIBLE) {
						eEdgeStroke = oTest.getEnum(XFA.STROKETAG);
						if (bFirst) {
							eCurrentStroke = eEdgeStroke;
							bFirst = false;
						} else {
							if (eEdgeStroke != eCurrentStroke) {
								bFailed = true;
								break;
							}
						}
					}
				}
				// Tested the i'th edge (if any). Now test i'th corner (if any)
				if (oCorner != null) {
					// watson 1463877 - don't bother checking the corner if it 
					// doesn't have a radius
					Measurement oRadius = (Measurement)oCorner.peekAttribute(XFA.RADIUSTAG);	
					if (oRadius == null || oRadius.getValue() == 0)
						break;			
				}

				oTest = oCorner;
			}
		}
		// Get the stroke, we may be drawing one of our special borders.
		if (!bFailed
				&& (eEdgeStroke == EnumAttr.STROKE_EMBOSSED
						|| eEdgeStroke == EnumAttr.STROKE_ETCHED
						|| eEdgeStroke == EnumAttr.STROKE_RAISED || eEdgeStroke == EnumAttr.STROKE_LOWERED)) {
			return true;
		}

		return false;
	}
	
	/**
	 * @see ProtoableNode#isContextSensitiveAttribute(int)
	 */
	public boolean isContextSensitiveAttribute(int eTag) {
		return eTag == XFA.LEFTINSETTAG || eTag == XFA.RIGHTINSETTAG ||
				  eTag == XFA.TOPINSETTAG || eTag == XFA.BOTTOMINSETTAG ||
				  super.isContextSensitiveAttribute(eTag);
	}
	

	/**
	 * Get the four inset values corresponding to TOPINSETTAG, RIGHTINSETTAG, BOTTOMINSETTAG, LEFTINSETTAG
	 * in that order (same order as edges are defined in XFA).  The goal is to improve
	 * performance over 4 individual queries.
	 */
	public void getInsets(ObjectHolder<UnitSpan> oTHolder, ObjectHolder<UnitSpan> oRHolder, 
						  ObjectHolder<UnitSpan> oBHolder, ObjectHolder<UnitSpan> oLHolder) {
		CachedInfo oCache = new CachedInfo();

		int eTags[] =  new int[] { XFA.TOPINSETTAG, XFA.RIGHTINSETTAG, XFA.BOTTOMINSETTAG, XFA.LEFTINSETTAG };		
		UnitSpan oResults[] = new UnitSpan[] { UnitSpan.ZERO, UnitSpan.ZERO, UnitSpan.ZERO, UnitSpan.ZERO };

		for (int nEdge = 0; nEdge < 4; nEdge++) {
			// Peek at value.  Don't "get" it because we want to handle the
			// default-value case ourselves.
			Attribute oAttr = getAttribute(eTags[nEdge], true, false);
			Measurement oM = null;
			if (oAttr == null) {
				if (oCache.mbThicknessesValid) {
					oResults[nEdge] = oCache.moThicknesses[nEdge];
					continue;
				}
				oM = new Measurement(defaultAttributeImpl(eTags[nEdge], oCache));
			}
			else
				oM = new Measurement(oAttr);

			oResults[nEdge] = oM.getUnitSpan();
		}
		
		if (Assertions.isEnabled == true) {
			// Verify that cached values are correct (don't bother if no caching was done).
			if (oCache.mbThicknessesValid) {
				for (int nEdge = 0; nEdge < 4; nEdge++) {
					// oReference is correct by definition.  Verify that the cached value matches.
					UnitSpan oReference = new Measurement(getAttribute(eTags[nEdge])).getUnitSpan();
					assert(oResults[nEdge].value() == oReference.value());
				}
			}
		}
		
		oTHolder.value = oResults[0];
		oRHolder.value = oResults[1]; 
		oBHolder.value = oResults[2];
		oLHolder.value = oResults[3];
	}
}