/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */

package com.adobe.xfa.ut;

/**
 * The class <b>ISODateTime</b> is a convenience class grouping the
 * {@link ISODate} and {@link ISOTime} classes, to define objects
 * in support of  ISO8601/XFA date time patterns.
 * 
 * <p>A valid ISO8601/XFA date time string is the concatenation 
 * of a valid ISO8601/XFA date string and a valid ISO8601/XFA time string
 * with the letter 'T' acting as a separator between the date
 * and time elements of the string.
 * See {@link ISODate} and {@link ISOTime} for the definitions of
 * valid ISO8601/XFA date and time patterns.
 *
 * Here's a snippet of code illustrating the use of
 * {@link ISODateTime} to reformat an ISO8601/XFA datetime string
 * <pre><code>
 *      #include &lt;pub/ut/isodatetime.h>       // for defn of ISODateTime.
 *      #include &lt;pub/ut/string.h>            // for defn of String.
 *      ...
 *      ISODateTime dt("2000-02-28T11:12:13-05:00");
 *      String s("Unknown");
 *      if (dt.isValid())
 *          s = dt.format(ISODateTime.getXFA_DATETIME_FMT1());
 *      printf("%S\n", (const wchar_t *) s);
 * </code></pre>
 *
 * @author Mike P. Tardif
 *
 * @exclude from published api.
 */
public class ISODateTime {

    /**
     * An ISO8601/XFA date time pattern
     * string: <b>YYYYMMDDTHHMMSS.FFFz</b>.
     */
	public static final String XFA_DATETIME_FMT1 = "YYYYMMDDTHHMMSS.FFFz";

    /**
     * An alternate ISO8601/XFA date time pattern
     * string: <b>YYYY-MM-DDTHH:MM:SS.FFFzz</b>.
     */
	public static final String XFA_DATETIME_FMT2 = "YYYY-MM-DDTHH:MM:SS.FFFzz";

    /**
     * Instantiate an ISODateTime object from today's date and time.
     */
	public ISODateTime() {
		mDate = new ISODate();
		mTime = new ISOTime();
		mDate.setLocalDate();
		mTime.setLocalTime();
	}

    /**
     * Instantiate an ISODateTime object from the given date time string.
     * @param datetime an ISO8601/XFA date time string.
     * @param datelocale a date locale name.
     * @param timelocale a time locale name.
     */
    public ISODateTime(String datetime,
				String datelocale /* = "" */,
					String timelocale /* = "" */) {
		//
		// Extract any ISO date component from the date time string.
		//
		String date = datetime;
		int tee = datetime.indexOf('T');
		if (tee >= 0)
			date = datetime.substring(0, tee);
		//
		// Extract any ISO time component from the date time string.
		//
		String time = datetime;
		tee = datetime.indexOf('T', tee);
		if (tee >= 0)
			time = datetime.substring(tee + 1);
		mDate = new ISODate(date, datelocale);
		mTime = new ISOTime(time, timelocale);
		mDate.setLocalDate();
		mTime.setLocalTime();
	}

    /**
     * Set this object to operate on local (date and) time
     * as opposed to the default, which is Greenwich Mean (date and) time.
     * Any subsequent calls to the format method will result in
     * date time strings that are expressed in local time. 
     */
    public void setLocalTime() {
		mDate.setLocalDate();
		mTime.setLocalTime();
	}

    /**
     * Set this object to operate
     * on Greenwich Mean (date and) time, which is the default.
     * Any subsequent calls to the format method will result in
     * date time strings that are expressed in Greenwich Mean time. 
     */
    public void setGMTime() {
		mDate.setGMDate();
		mTime.setGMTime();
	}

    /**
     * Format this object given the ISO datetime format string. 
     * @param isoformat an ISO8601/XFA date time format string.
     * @return the date time string formatted according to the given format
     * string, upon success, or the empty string, upon error.
     */
	public String format(String isoformat) {
		if (! isValid())
			return "";
		int tee = isoformat.indexOf('T');
		if (tee < 0)
			return "";
		//
		// Extract the ISO date and time components from the date time string.
		//
		String dateformat = isoformat.substring(0, tee);
		String timeformat = isoformat.substring(tee + 1);
		//
		// Individually format the date and time components and concatenate
		// everything together with a 'T' separator.
		//
		return mDate.format(dateformat) + 'T' + mTime.format(timeformat);
	}

    /**
     * Determine if this object is valid.
     * @return boolean true if valid, and false otherwise.
     */
    public boolean isValid() {
		return (mDate.isValid() && mTime.isValid());
	}

	/**
	* Get date component of this object
	* @return date
	*/
	public ISODate getDate() { return mDate; }

	/**
	* Get time component of this object
	* @return time
	*/
	public ISOTime getTime() { return mTime; }


	protected    ISODate mDate;  // This object's date element.
	protected    ISOTime mTime;  // This object's time element.

}
