// =================================================================================================
// ADOBE SYSTEMS INCORPORATED
// Copyright 2011 Adobe Systems Incorporated
// All Rights Reserved
//
// NOTICE:  Adobe permits you to use, modify, and distribute this file in accordance with the terms
// of the Adobe license agreement accompanying it.
// =================================================================================================

package com.adobe.xmp.schema.service;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.xml.namespace.QName;

import com.adobe.xmp.path.XMPPath;
import com.adobe.xmp.schema.service.impl.SchemaManifestParser;
import com.adobe.xmp.schema.service.impl.SchemaManifestParser.SchemaManifest;


/**
 * The standard RelaxNG-provider provides access to standard Relax-NG resources 
 * present inside this bundle. 
 * 
 * @author Stefan Makswit
 */
public class StandardRelaxNGProvider implements RelaxNGProvider
{
	/** Default manifest file relative to the library */
	private static final String DEFAULT_MANIFEST_FILE = "relaxng/schemaManifest.xml";
	/** search paths for include files */
	private List<URI> includePaths;
	private Map<String, URI> schemaMap;
	
	/**
	 * Default constructor which uses the default manifest file.
	 * 
	 * @throws SchemaServiceException Thrown if the input files are not found or could not be loaded.
	 */
	public StandardRelaxNGProvider() throws SchemaServiceException
	{
		this(null);
	}


	/**
	 * Constructs and initializes an {@link com.adobe.xmp.schema.service.RelaxNGProvider} with custom manifest.
	 * 
	 * @param schemaManifestURI the URI to the manifest XML file 
	 * @throws SchemaServiceException Thrown if the URI contains syntax errors. 
	 */
	StandardRelaxNGProvider(URI schemaManifestURI) throws SchemaServiceException
	{
		try
		{
			// use default manifest if URI is null
			if (schemaManifestURI == null)
			{
				schemaManifestURI = getResource(DEFAULT_MANIFEST_FILE);
			}
			
			// load from configurable "schema-registry.xml"
			SchemaManifest manifest = SchemaManifestParser.parse(schemaManifestURI);

			// convert to full qualified paths to RNG files
			schemaMap = new HashMap<String, URI>();
			for (String schemaURI : manifest.schemaFiles.keySet())
			{
				String path = manifest.schemaFiles.get(schemaURI);
				schemaMap.put(schemaURI, getResource(path));
			}
			
			// convert to full qualified paths to include files
			includePaths = new ArrayList<URI>();
			for (String includePath : manifest.includePaths)
			{
				includePaths.add(getResource(includePath));
			}
		}
		catch (Exception e)
		{
			throw new SchemaServiceException(e.getMessage(), e);
		}
	}
	
	
	/**
	 * @see com.adobe.xmp.schema.service.RelaxNGProvider#getSchema(java.lang.String)
	 */
	public InputStream getSchema(String namespaceURI) throws IOException
	{
		URI filePath = schemaMap.get(namespaceURI);
		if (filePath != null)
		{
			URL url = filePath.toURL();
			return url.openStream();
		}
		else
		{
			return null;
		}
	}

	
	/**
	 * @see com.adobe.xmp.schema.service.RelaxNGProvider#getInclude(java.lang.String)
	 */
	public InputStream getInclude(String includeName) throws IOException
	{
		InputStream is = null;
		
		try {
			
			for (URI path : includePaths)
			{
				URI includePath;
				
					includePath = new URI(path + includeName);
				
				URL url = includePath.toURL();
				try
				{
					is = url.openStream();
					break;
				}
				catch (Exception e)
				{
					continue;
				}
			}
		} catch (URISyntaxException e) {
			throw new IOException(e.getMessage());
		}
		return is;
	}
	

	/**
	 * @see com.adobe.xmp.schema.service.RelaxNGProvider#isAvailable(java.lang.String)
	 */
	public boolean isAvailable(String namespaceURI)
	{
		return schemaMap.containsKey(namespaceURI);
	}

	
	/**
	 * @see com.adobe.xmp.schema.service.RelaxNGProvider#getNamespaces()
	 */
	public Set<String> getNamespaces()
	{
		return schemaMap.keySet();
	}
	

	private static ClassLoader getClassLoader() 
	{
        ClassLoader loader = StandardRelaxNGProvider.class.getClassLoader();
        if (loader == null) {
            loader = ClassLoader.getSystemClassLoader();
        }
        return loader;
    }
	 
	/**
	 * @return Returns the absolute path of this class' folder.
	 * @throws URISyntaxException If the base folder cannot be determined
	 */
	private static URI getResource(String path) throws URISyntaxException
	{
		ClassLoader loader = getClassLoader();
		return loader.getResource(path).toURI();
	}


	/**
	 * No default runtime decorators.
	 * @see com.adobe.xmp.schema.service.RelaxNGProvider#getRuntimeDecorators()
	 */
	public Map<XMPPath, Map<QName, Map<String, String>>> getRuntimeDecorators()
	{
		return null;
	}	
}
