/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 * 
 * Copyright 2024 Adobe
 * All Rights Reserved.
 * 
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 **************************************************************************/

package com.day.cq.commons.predicates.servlets;

import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.SlingHttpServletResponse;

import javax.servlet.ServletException;
import java.io.IOException;

/**
 * Abstract servlet to use for the client/server command interface.
 *
 */
public abstract class AbstractCommandServlet extends AbstractPredicateServlet {
    /**
     * default path parameter name
     */
    public static final String PATH_PARAM = "path";

    /**
     * default action parameter name
     */
    public static final String ACTION_PARAM = "cmd";

    @Override
    protected void doPost(SlingHttpServletRequest request,
            SlingHttpServletResponse response)
            throws ServletException, IOException {
        if (this.hasCommand(request)) {
            this.performCommand(request, response);
        } else {
            handleMethodNotImplemented(request, response);
        }
    }

    @Override
    protected void doGet(SlingHttpServletRequest request,
            SlingHttpServletResponse response)
            throws ServletException, IOException {
        if (this.hasCommand(request)) {
            this.performCommand(request, response);
        } else {
            handleMethodNotImplemented(request, response);
        }
    }

    /**
     * Checks if the command in the request is supported.
     * Returns <code>false</code> by default.
     *
     * @param request servlet request
     * @return <code>true</code> if the command is supported.
     * @throws ServletException if a servlet error occurs
     */
    protected boolean hasCommand(SlingHttpServletRequest request)
            throws ServletException {
        return false;
    }

    /**
     * Perform the command of the current request.
     * This method is invoked if {@link #hasCommand(SlingHttpServletRequest)} returns <code>true</code>
     * for a POST request.
     *
     * @param request servlet request
     * @param response servlet response
     * @throws ServletException if an error occurs.
     * @throws IOException if an I/O error occurs.
     */
    abstract protected void performCommand(SlingHttpServletRequest request, SlingHttpServletResponse response)
            throws ServletException, IOException;

    /**
     * Get value of specified parameter, trim and verify that it's provided
     *
     * @param request the servlet request
     * @param name the name of the parameter
     *
     * @return the value of the parameter
     *
     * @throws ServletException if the parameter is missing or empty.
     */
    protected String requireParameter(SlingHttpServletRequest request, String name)
            throws ServletException {
        String val = request.getParameter(name);
        if (val != null) {
            val = val.trim();
        }
        if (val == null || val.length() == 0) {
            throw new ServletException("Missing request parameter '" + name + "'");
        }
        return val;
    }

}