/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.dam.commons.handler;

import java.io.IOException;

/**
 * Continuously process bytes received from an <code>InputStream</code> and
 * write the ones contained in an xpacket to an <code>OutputStream</code>.
 *
 * @author dpfister
 */
public class XPacketFilter extends AbstractFilter {

    /**
     * XPacket begin marker.
     */
    private static final byte[] XPACKET_BEGIN = "<?xpacket begin=".getBytes();

    /**
     * XPacket end marker.
     */
    private static final byte[] XPACKET_END = "<?xpacket end=".getBytes();

    /**
     * XPacket terminator.
     */
    private static final byte[] XPACKET_TERMINATOR = "?>".getBytes();

    /**
     * Parse state definition.
     */
    enum ParseState {
        XPACKET_BEGIN,
        XPACKET_BEGIN_TERM,
        XPACKET_END,
        XPACKET_END_TERM,
        END
    };

    /**
     * Current parse state.
     */
    private ParseState ps = ParseState.XPACKET_BEGIN;

    /**
     * Current xpacket marker position.
     */
    private int markerPos;

    /**
     * {@inheritDoc}
     */
    protected void doFilter(byte[] buf, int off, int len) throws IOException {
        int startIndex = ps != ParseState.XPACKET_BEGIN ? 0 : -1;
        for (int i = 0; i < len; i++) {
            byte b = buf[off + i];

            switch (ps) {
            case XPACKET_BEGIN:
                if (b == XPACKET_BEGIN[markerPos]) {
                    markerPos++;
                    if (markerPos == XPACKET_BEGIN.length) {
                        notifyStarted();
                        put(XPACKET_BEGIN);
                        ps = ParseState.XPACKET_BEGIN_TERM;
                        markerPos = 0;
                        startIndex = i + 1;
                    }
                } else {
                    markerPos = 0;
                    if (b == XPACKET_BEGIN[markerPos]) {
                        markerPos++;
                    }
                }
                break;
            case XPACKET_BEGIN_TERM:
                if (b == XPACKET_TERMINATOR[markerPos]) {
                    markerPos++;
                    if (markerPos == XPACKET_TERMINATOR.length) {
                        put(buf, off + startIndex, i - startIndex + 1);
                        ps = ParseState.XPACKET_END;
                        markerPos = 0;
                        startIndex = i + 1;
                    }
                } else {
                    markerPos = 0;
                    if (b == XPACKET_TERMINATOR[markerPos]) {
                        markerPos++;
                    }
                }
                break;
            case XPACKET_END:
                if (b == XPACKET_END[markerPos]) {
                    markerPos++;
                    if (markerPos == XPACKET_END.length) {
                        put(buf, off + startIndex, i - startIndex + 1);
                        ps = ParseState.XPACKET_END_TERM;
                        markerPos = 0;
                        startIndex = i + 1;
                    }
                } else {
                    markerPos = 0;
                    if (b == XPACKET_END[markerPos]) {
                        markerPos++;
                    }
                }
                break;
            case XPACKET_END_TERM:
                if (b == XPACKET_TERMINATOR[markerPos]) {
                    markerPos++;
                    if (markerPos == XPACKET_TERMINATOR.length) {
                        put(buf, off + startIndex, i - startIndex + 1);
                        ps = ParseState.END;
                        startIndex = -1;
                        notifyEnded();
                    }
                } else {
                    markerPos = 0;
                    if (b == XPACKET_TERMINATOR[markerPos]) {
                        markerPos++;
                    }
                }
                break;
            }
        }
        if (startIndex >= 0) {
            put(buf, off + startIndex, len - startIndex);
        }
    }

    /**
     * {@inheritDoc}
     *
     * Reset internal state.
     */
    @Override
    protected void doReset() {
        super.doReset();

        ps = ParseState.XPACKET_BEGIN;
        markerPos = 0;
    }

    /**
     * {@inheritDoc}
     *
     * Contents is only valid if the state equals {@link ParseState#END}.
     */
    @Override
    public boolean isValid() {
        return ps == ParseState.END;
    }
}
