/*************************************************************************
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2011 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 **************************************************************************/
package com.day.cq.replication;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.sling.event.dea.DEAConstants;
import org.osgi.service.event.Event;
import org.osgi.service.event.EventProperties;

/**
 * An event fired when some {@link ReplicationAction} needs to be handled
 */
public class ReplicationEvent implements Serializable {

    public static final String EVENT_TOPIC = "com/adobe/granite/replication";
    public static final String PATHS = "paths";

    private static final String PROPERTY_MODIFICATIONS = "modifications";
    private static final String TYPE = "type";
    private static final String TIME = "time";
    private static final String USER_ID = "userId";
    private static final String REVISION = "revision";

    /**
     * the replication action associated with this event
     */
    private ReplicationAction replicationAction;

    /**
     * Is this a local event?
     */
    private final boolean isLocal;


    public ReplicationEvent(ReplicationAction replicationAction) {
        this(replicationAction, true);
    }

    public ReplicationEvent(ReplicationAction replicationAction, boolean isLocal) {
        this.replicationAction = replicationAction;
        this.isLocal = isLocal;
    }

    /**
     * Returns the underlying {@link ReplicationAction} which fired this event
     *
     * @return the replication action which caused the event
     */
    public ReplicationAction getReplicationAction() {
        return this.replicationAction;
    }

    /**
     * Convert an OSGi event to a replication event if possible.
     *
     * @param evt The OSGi event.
     * @return The replication event
     */
    public static ReplicationEvent fromEvent(Event evt) {
        if (!evt.getTopic().equals(EVENT_TOPIC)) {
            return null;
        }

        ReplicationAction action = null;
        String[] paths;
        ReplicationActionType type;
        Long time;
        String userId;
        String revision;
        @SuppressWarnings("unchecked")
        final List<Map<String, Object>> modProps = (List<Map<String, Object>>) evt.getProperty(PROPERTY_MODIFICATIONS);

        // sanity check for null and list size
        if (modProps != null && modProps.size() == 1) {
            final Map<String, Object> modProp = modProps.get(0);
            paths = (String[]) modProp.get(PATHS);
            type = (ReplicationActionType) modProp.get(TYPE);
            time = (Long) modProp.get(TIME);
            userId = (String) modProp.get(USER_ID);
            revision = (String) modProp.get(REVISION);
            action = new ReplicationAction(type, paths, time, userId, revision);
        }

        // the action should never be null, if that happens it's unexpected
        if (action == null)
            throw new RuntimeException();

        return new ReplicationEvent(action, evt.getProperty(DEAConstants.PROPERTY_APPLICATION) == null);
    }

    /**
     * Generate a dictionary for the OSGi event.
     *
     * @param distribute Whether this event should be distributed across the cluster.
     * @return OSGi event properties
     * @since 5.18.0
     */
    protected Map<String, Object> getEventProperties(final boolean distribute) {
        final Map<String, Object> properties = new HashMap<String, Object>();
        final List<Map<String, Object>> modProps = new ArrayList<Map<String, Object>>();

        final Map<String, Object> actionProps = new HashMap<String, Object>();
        actionProps.put(PATHS, replicationAction.getPaths());
        actionProps.put(TYPE, replicationAction.getType());
        actionProps.put(TIME, replicationAction.getTime());
        actionProps.put(USER_ID, replicationAction.getUserId());
        actionProps.put(REVISION, replicationAction.getRevision());
        modProps.add(actionProps);

        properties.put(PROPERTY_MODIFICATIONS, modProps);
        if (!isLocal) {
            // TODO - an application should never set this property!
            properties.put(DEAConstants.PROPERTY_APPLICATION, "unknown");
        }
        if ( distribute ) {
            properties.put(DEAConstants.PROPERTY_DISTRIBUTE, "");
        }

        return new EventProperties(properties);
    }

    /**
     * Create a distributable event.
     *
     * @return An event.
     */
    public Event toEvent() {
        return new Event(EVENT_TOPIC, this.getEventProperties(true));
    }

    /**
     * Create a non distributable event.
     *
     * @return An event.
     */
    public Event toNonDistributableEvent() {
        return new Event(EVENT_TOPIC, this.getEventProperties(false));
    }
}
