    /*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */

package com.day.cq.search.result;

import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.apache.sling.api.resource.Resource;

import com.day.cq.search.Query;
import com.day.cq.search.SimpleSearch;
import com.day.cq.search.facets.Facet;

import aQute.bnd.annotation.ProviderType;

/**
 * <code>SearchResult</code> represents a search result of a JCR query, returned
 * by {@link Query} and {@link SimpleSearch}.
 */
@ProviderType
public interface SearchResult {

    /**
     * Returns the total number of matches or the total as far as it is known. This
     * depends on the query, i.e. it might just retrieve the minimum necessary
     * up to {@link #getStartIndex()} + {@link #getHitsPerPage()}. Check
     * {@link #hasMore()} to see if there are more matches.
     *
     * @return the total number of matches.
     */
    public long getTotalMatches();

    /**
     * Returns whether there are more matches than given by {@link #getTotalMatches()}.
     * @return whether there are more matches than given by {@link #getTotalMatches()}
     */
    public boolean hasMore();

    /**
     * @return the start index. i.e. from where to start to display the hits.
     */
    public long getStartIndex();

    /**
     * @return the number of hits to display on a page (0 for all).
     */
    public long getHitsPerPage();

    /**
     * @return a List of {@link Hit}s to display on the result page.
     */
    public List<Hit> getHits();
    
    /**
     * @return the same list as {@link #getHits()} but directly with the underlying nodes
     * 
     * @since 5.3
     */
    public Iterator<Node> getNodes();

    /**
     * @return the same list as {@link #getHits()} but returning Resource objects
     *
     * @since 5.6
     */
    public Iterator<Resource> getResources();

    /**
     * @return a List of {@link ResultPage}es to display the navigation through the
     *         result pages. It may not return all the pages in the result due to performance concerns.
     *         The actual result is dependent on implementation i.e. it may return X previous pages and X next pages.
     */
    public List<ResultPage> getResultPages();

    /**
     * @return the page, which contains the information about the previous page.
     *         Returns <code>null</code> if there is no previous page (i.e. the
     *         current page is the first page).
     */
    public ResultPage getPreviousPage();

    /**
     * @return the page, which contains the information about the next page.
     *         Returns <code>null</code> if there is no next page (i.e. the
     *         current page is the last page).
     */
    public ResultPage getNextPage();

    /**
     * Returns the execution time in fractions of a second.
     * <br/>
     * Example: 0.08 (means, the query took 80 milliseconds to execute).
     * @return the execution time of the query.
     */
    public String getExecutionTime();

    /**
     * Returns the execution time in milliseconds.
     *
     * @return the execution time of the query.
     */
    public long getExecutionTimeMillis();

    /**
     * Returns the facets for this search result.
     *
     * @return the facets for this search result.
     * @throws RepositoryException if an error occurs while executing the query
     *          or calculating the facets.
     */
    public Map<String, Facet> getFacets() throws RepositoryException;

    /**
     * Returns the actual query string run against the repository, eg. an XPath
     * statement. Please note that this does not describe the full query - for a
     * simple list of the predicates that filter see
     * {@link #getFilteringPredicates()}.
     * 
     * @return a string of the query that was created
     */
    public String getQueryStatement();

    /**
     * Returns a string with a list of the predicates that were filtering the
     * result set (as opposed to using XPath, see {@link #getQueryStatement()}).
     * The returned string is for informational purposes only and might not
     * fully describe the predicates and their parameters.
     * 
     * @return a string mentioning all the predicates that filtered the result
     *         set
     */
    public String getFilteringPredicates();
}
