/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.wcm.designimporter.parser.taghandlers;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Arrays;

import javax.jcr.RepositoryException;

import com.day.cq.wcm.designimporter.ErrorCodes;
import org.xml.sax.Attributes;

import com.day.cq.wcm.designimporter.DesignImporterContext;
import com.day.cq.wcm.designimporter.DesignImportException;
import com.day.cq.wcm.designimporter.UnsupportedTagContentException;
import com.day.cq.wcm.designimporter.api.EntryTagHandler;
import com.day.cq.wcm.designimporter.api.HTMLContentProvider;
import com.day.cq.wcm.designimporter.api.TagHandler;
import com.day.cq.wcm.designimporter.api.TagHandlerProvider;
import com.day.cq.wcm.designimporter.parser.HTMLContentType;

/**
 * The script tag handler is used to extract the name of the scripts referenced
 * within the input HTML document.
 *
 * <p>
 * This tag handler is registered against the markup of type < script src="myscript.js" >. That implies
 * this tag handler will come into action whenever the design importer framework
 * encounters an HTML tag matching the rule this handler is defined against.
 * </p>
 *
 * <p>
 * Note: This tag handler does not handle inline script tags. {@see InlineScriptTagHandler}
 * </p>
 *
 * @see com.day.cq.wcm.designimporter.parser.taghandlers.factory.ScriptTagHandlerFactory
 */
public class ScriptTagHandler implements TagHandler, EntryTagHandler, HTMLContentProvider {


    private boolean isAbsolutePath = false;

    private DesignImporterContext designImporterContext;

    private String scriptPath;

    public void beginHandling(String uri, String localName, String qName, Attributes atts) {
        scriptPath = atts.getValue("src");
        if (isValidURL(scriptPath)) {
            isAbsolutePath = true;
        } else {
            // Sanitize the script url
            if (scriptPath != null) scriptPath = scriptPath.replaceAll("([^?]*)?.*", "$1");

            // Check if the script was existent in the archive. If not, log a warning.
            try {
                if (!designImporterContext.designNode.hasNode(scriptPath)) {
                    designImporterContext.importWarnings.add("Could not locate the referenced script '" + scriptPath
                        + "' in the design package");
                }
            } catch (RepositoryException e) {
            }
        }
    }

    public void characters(char[] ch, int start, int length) throws DesignImportException {
        if (new String(ch, start, length).trim().length() > 0)
            throw new UnsupportedTagContentException(ErrorCodes.FREE_FLOATING_TEXT_SCRIPT_TAG);
    }

    public void endElement(String uri, String localName, String qName) throws DesignImportException {
        // Invalid. External script tag must not contain any nested tags.
        throw new UnsupportedTagContentException();
    }

    public void endHandling(String uri, String localName, String qName) {

    }

    public Object getContent(HTMLContentType htmlContentType) {

        if (htmlContentType == HTMLContentType.SCRIPT_INCLUDE)
            return Arrays.asList(new String[] { scriptPath });
        else if (htmlContentType == HTMLContentType.MARKUP) return "<script type='text/javascript' src='" + scriptPath + "'></script>";
        return null;
    }

    public String getScriptPath() {
        return scriptPath;
    }

    public void setDesignImporterContext(DesignImporterContext designImporterContext) {
        this.designImporterContext = designImporterContext;
    }

    public void setTagHandlerProvider(TagHandlerProvider tagHandlerProvider) {
    }

    public void startElement(String uri, String localName, String qName, Attributes atts) throws DesignImportException {
        // Invalid. External script tag must not contain any nested tags.
        throw new UnsupportedTagContentException();
    }

    public boolean supportsContent(HTMLContentType htmlContentType) {

        if (htmlContentType == HTMLContentType.SCRIPT_INCLUDE)
            return !isAbsolutePath;
        else if (htmlContentType == HTMLContentType.MARKUP)
            return isAbsolutePath;
        else
            return false;
    }

    private boolean isValidURL(String scriptPath) {
        // if(scriptPath.startsWith("http") && scriptPath.endsWith(".js"))
        // return true;

        try {
            URL uri = new URL(scriptPath);
            return true;
        } catch (MalformedURLException e) {
            return false;
        }
    }
}