/*
 * File: PDFDocEncoding.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.util;

import java.nio.ByteBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CodingErrorAction;

import com.adobe.internal.pdftoolkit.core.exceptions.PDFParseException;

/**
 * 
 * @author Isak Tenenboym
 *
 *	This class contains methods that convert text representation in PDF
 *	(which is in PDFDocEncoding or UCS-2) from/to Unicode.
 */

public final class PDFDocEncoding
{
	//Decoder for UTF-16
	private static final ThreadLocal<CharsetDecoder> UTF16DECODER = new ThreadLocal<CharsetDecoder>(){
		@Override
		protected CharsetDecoder initialValue(){
			return Charset.forName(
					UTFSupport.UTF16).newDecoder().onMalformedInput(CodingErrorAction.IGNORE);
		}
	};

	private PDFDocEncoding() {}

	/**
	 * Converts byte array to same string format as Acrobat does.
	 */
	public static String toAcrobatString(byte[] bytes)
	{
		return toAcrobatString(bytes, 0, bytes.length);
	}

	/**
	 * Converts byte array to same string format as Acrobat does.
	 */
	public static String toAcrobatString(byte[] bytes, int start, int length)
	{
		try {
			return toUnicodeString(bytes, start, length);
		} catch (PDFParseException e1) {
			// String unable to be converted - do nothing and try other conversion
		}
		
		// last try
		return toEscapeString(bytes, start, length);
	}

	/**
	 * Converts byte array to unicode string.
	 */
	public static String toUnicodeString(byte[] bytes)
		throws PDFParseException
	{
		return toUnicodeString(bytes, 0, bytes.length);
	}

	//Removes the escaped bytes as mentioned in PDF Reference 1.7 Page 159
	private static ByteBuffer removeEscapedBytes(byte[] bytes) {
		ByteBuffer byteBuffer = ByteBuffer.allocate(bytes.length);
		boolean escapeMode = false;
		/*
		 * (From PDF Reference 1.7 Page 159
		 * The escape sequence consists of the following elements, in order:
		 * 1.The Unicode value U+001B (that is, the byte sequence 0 followed by
		 * 27). 
		 * 2.A 2-character ISO 639 language codefor example, en for
		 * English or ja for Japanese. Character in this context means byte (as
		 * in ASCII character), not Unicode character. 
		 * 3.(Optional) A 2-character ISO 3166 country codefor example, US for 
		 * the United States or JP for Japan. 
		 * 4.The Unicode value U+001B.
		 */
		/*
		 * From the above description, we just need to locate the escape START
		 * and END sequence and ignore anything lying in between
		 */
		for (int i = 0; i < bytes.length; i++) {
			if (!escapeMode) {
				if (i != bytes.length - 1 && bytes[i] == 0
						&& bytes[i + 1] == 27) {
					escapeMode = true;
					i += 1;
				} else {
					byteBuffer.put(bytes[i]);
				}
			} else {
				if (bytes[i] == 0 && bytes[i + 1] == 27) {
					escapeMode = false;
					i += 1;
				}
			}
		}
		byteBuffer.limit(byteBuffer.position());
		byteBuffer.rewind();
		return byteBuffer;
	}
	
	/**
	 * 
	 * Make a Unicode string from a byte array
	 * @throws PDFParseException
	 * @throws RuntimeException, hard failure, if basic encodings aren't supported
	 *  This represents a "can't happen" situation
	 */
	public static String toUnicodeString(byte[] bytes, int start, int length)
		throws PDFParseException
	{
		if (bytes == null)
			return null;
		if(length < 1)
			return "";
		if ((length >= 2) && (bytes[start] == -2) && (bytes[start + 1] == -1)) {   // We have UTF-16 encoded byte array
			try {
				return UTF16DECODER.get().decode(removeEscapedBytes(bytes)).toString();
			} catch (CharacterCodingException e) {
				throw new RuntimeException(e);
			}
		} else {
			StringBuilder strBuf = new StringBuilder(bytes.length);
			int bytesInd;
			for (bytesInd = start; bytesInd < length; bytesInd++)
				strBuf.append(toUnicodeChar(bytes[bytesInd], true));
			return strBuf.toString();
		}
	}

	
	/**
	 * Converts the byte array to escaped string.
	 */
	public static String toEscapeString(byte[] bytes, int start, int length)
	{
		StringBuilder encodedStr = new StringBuilder(bytes.length);
		int end = start + length;
		while (start < end) {
			int curByte = bytes[start++];
			if (curByte < 0)
				curByte += 256;
			if ((curByte < 32) && !ByteOps.isWhitespace((char)curByte)) {
				encodedStr.append(ESCAPE_START);
				encodedStr.append(Integer.toOctalString(curByte));
			} else {
				encodedStr.append((char)curByte);
			}
		}
		return encodedStr.toString();
	}

	static boolean isEscapeChar(String str, int pos)
	{
		if (str.length() < (pos + 3))
			return false;
		if ((str.charAt(pos) != ESCAPE_START) || !Character.isDigit(str.charAt(pos + 1)) ||
		    !Character.isDigit(str.charAt(pos + 2)) || !Character.isDigit(str.charAt(pos + 3)))
			{
				return false;
			}
		return true;
	}

	static int getEscapeChar(String str, int pos)
	{
		return Integer.parseInt(str.substring(pos + 1, pos + 3), 8);
	}

	/**
	 * Converts escaped string to a byte array.
	 */
	public static byte[] fromEscapeString(String str)
	{
		byte[] bytes = new byte[str.length()];
		int byteCount = 0;
		int strInd;
		for (strInd = 0; strInd < str.length(); strInd++) {
			if (isEscapeChar(str, strInd)) {
				bytes[byteCount++] = (byte)getEscapeChar(str, strInd);
				strInd +=3;
			} else {
				bytes[byteCount++] = (byte)str.charAt(strInd);
			}
		}
		if (byteCount < bytes.length)
			return bytes;
		else {
			byte[] newBytes = new byte[byteCount];
			for (strInd = 0; strInd < byteCount; strInd++)
				newBytes[strInd] = bytes[strInd];
			return newBytes;
		}
	}

	/**
	 * Converts a unicode string to byte array.
	 */
	public static byte[] fromUnicodeString(String uniStr)
   	{
 		return  fromUnicodeString(uniStr, !isPDFDocEncoding(uniStr));
 	}

 	public static byte[] fromUnicodeString(String uniStr, boolean utf16)
 	{
 		if (utf16)
			{
				int byteInd = 0;
				int strInd;
				char curChar;
				int byteLen = ((uniStr.length()) * 2) + 2;
				byte[] bytes = new byte[byteLen];
				bytes[byteInd++] = (byte)0xFE;
				bytes[byteInd++] = (byte)0xFF;
				for (strInd = 0; strInd < uniStr.length(); strInd++) {
					curChar = uniStr.charAt(strInd);
					bytes[byteInd++] = (byte)(curChar >> 8);
					bytes[byteInd++] = (byte)curChar;
				}
				return bytes;
			} else {
			byte[] bytes = new byte[uniStr.length()];
			int stringInd;
			for (stringInd = 0; stringInd < uniStr.length(); stringInd++) {
				byte curChar = getPDFValue(uniStr.charAt(stringInd));
				if (curChar == 0)
					curChar = (byte)'.';
				bytes[stringInd] = curChar;
			}
			return bytes;
		}
	}

	static char toUnicodeChar(byte curByte, boolean forgiving)
		throws PDFParseException
	{
		char unicodeVal = 0;
		int byteInd = curByte;
		if (curByte < 0)
			byteInd += 256;
		unicodeVal = toUnicodeMap[byteInd];
		if (unicodeVal == 0) {
			if (forgiving)
				return DEFAULT_CHARACTER;
			else
				throw new PDFParseException("Unrecognizable character in the PDF text object");
		}
		return unicodeVal;
	}

	/** PDFDocEncoding-encoded bytes to string conversion.
	 *
	 * @throws CharacterCodingException the byte stream is not a
	 * legal encoding.
	 *
	 */
	public static char[] decode (byte[] bytes, int start, int length) throws CharacterCodingException {
		char[] chars = new char [bytes.length];
		for (int i = 0; i < bytes.length; i++) {
			chars [i] = toUnicodeMap [(bytes [i]) & 0xff];
			if (chars [i] == 0) {
				throw new CharacterCodingException (); }}
		return chars;
	}

	/** String to PDFDocEncoding-encoded bytes conversion.
	 *
	 * @throws CharacterCodingException the input string contains a
	 * characters which is not in PDFDocEncoding.
	 *
	 */
	public static byte[] encode (String s) throws CharacterCodingException {
		byte[] bytes = new byte [s.length ()];
		for (int i = 0; i < s.length (); i++) {
			char c = s.charAt (i);
			if (toUnicodeMap [c] == c) {
				bytes [i] = (byte) c; }
			else {
				int j = findPDFValue (c);
				if (j >= 0) {
					bytes [i] = (byte) fromUnicodeMap [j + 1]; }
				else {
					throw new CharacterCodingException (); }}}
		return bytes;
	}

	static private byte getPDFValue(char uniVal)
	{
		int uniInd;
		if (uniVal < 256)
			return (byte)uniVal;
		else if ((uniInd = findPDFValue(uniVal)) >= 0)
			return (byte)fromUnicodeMap[uniInd + 1];
		else
			return 0;
	}

	/**
	 * Returns true if passed string represents unicode, else false.
	 */
	public static boolean isUnicode(String uniStr)
	{
		int strInd;
		for (strInd = 0; strInd < uniStr.length(); strInd++) {
			if (uniStr.charAt(strInd) > 255) {
				if (findPDFValue(uniStr.charAt(strInd)) < 0)
					return false;
			}
		}
		return true;
	}

	static private int findPDFValue(char uniVal)
	{
		int low = 0, high = fromUnicodeMap.length/2, diff, step;
		while( high > low) {
			step = low + (high - low)/2;
			diff = uniVal - fromUnicodeMap[step * 2];
			if (diff > 0)
				low = step + 1;
			else
				high = step;
			if (diff == 0)
				return (step * 2);
		}
		return -1;
	}

	public static boolean isPDFDocEncoding(String uniStr)
	{
		int strInd;
		for (strInd = 0; strInd < uniStr.length(); strInd++)
			{
				char thisChar = uniStr.charAt(strInd);
				if (thisChar > 255)
					{
						if (findPDFValue(thisChar) < 0)
							{
								return false;
							}
					}
				else
					{
						if (toUnicodeMap[thisChar] != thisChar)
							{
								return false;
							}
					}
			}
		return true;
	}


	
	/**
	 * Code points in the ranges 0x20-0x7e, 0xa1-0xac and
	 * 0xae-0xff have the same interpretation in PDFDocEncoding and
	 * in Unicode. Code points in the ranges 0x18-0x1f, 0x80-0x9e and 0xa0
	 * have different interpretations in PDFDocEncoding and Unicode.
	 * The remaining code points are unused in PDFDocEncoding.
	 *
	 * For performance reasons this map contains values for all 256
	 * code points of PDFDocEncoding. Unsused PDFDocEncoding code points
	 * are mapped to 0.
	 */
	private static final char[] toUnicodeMap =
	{
		0,	0,	0,	0,	0,	0,	0,	0,	// 0x0
		0,	0x0009, 0x000A, 0,	0,	0x000D, 0,	0,
		0,	0,	0,	0,	0,	0,	0,	0,	// 0x1
		0x02D8, 0x02C7, 0x02C6, 0x02D9, 0x02DD, 0x02DB, 0x02DA, 0x02DC,
		0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,	// 0x2
		0x0028, 0x0029, 0x002A, 0x002B, 0x002C, 0x002D, 0x002E, 0x002F,
		0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,	// 0x3
		0x0038, 0x0039, 0x003A, 0x003B, 0x003C, 0x003D, 0x003E, 0x003F,
		0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,	// 0x4
		0x0048, 0x0049, 0x004A, 0x004B, 0x004C, 0x004D, 0x004E, 0x004F,
		0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,	// 0x5
		0x0058, 0x0059, 0x005A, 0x005B, 0x005C, 0x005D, 0x005E, 0x005F,
		0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,	// 0x6
		0x0068, 0x0069, 0x006A, 0x006B, 0x006C, 0x006D, 0x006E, 0x006F,
		0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,	// 0x7
		0x0078, 0x0079, 0x007A, 0x007B, 0x007C, 0x007D, 0x007E, 0,
		0x2022, 0x2020, 0x2021, 0x2026, 0x2014, 0x2013, 0x0192, 0x2044,	// 0x8
		0x2039, 0x203A, 0x2212, 0x2030, 0x201E, 0x201C, 0x201D, 0x2018,
		0x2019, 0x201A, 0x2122, 0xFB01, 0xFB02, 0x0141, 0x0152, 0x0160,	// 0x9
		0x0178, 0x017D, 0x0131, 0x0142, 0x0153, 0x0161, 0x017E, 0,
		0x20AC, 0x00A1, 0x00A2, 0x00A3, 0x00A4, 0x00A5, 0x00A6, 0x00A7,	// 0xa
		0x00A8, 0x00A9, 0x00AA, 0x00AB, 0x00AC, 0,	0x00AE, 0x00AF,
		0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x00B4, 0x00B5, 0x00B6, 0x00B7,	// 0xb
		0x00B8, 0x00B9, 0x00BA, 0x00BB, 0x00BC, 0x00BD, 0x00BE, 0x00BF,
		0x00C0, 0x00C1, 0x00C2, 0x00C3, 0x00C4, 0x00C5, 0x00C6, 0x00C7,	// 0xc
		0x00C8, 0x00C9, 0x00CA, 0x00CB, 0x00CC, 0x00CD, 0x00CE, 0x00CF,
		0x00D0, 0x00D1, 0x00D2, 0x00D3, 0x00D4, 0x00D5, 0x00D6, 0x00D7,	// 0xd
		0x00D8, 0x00D9, 0x00DA, 0x00DB, 0x00DC, 0x00DD, 0x00DE, 0x00DF,
		0x00E0, 0x00E1, 0x00E2, 0x00E3, 0x00E4, 0x00E5, 0x00E6, 0x00E7,	// 0xe
		0x00E8, 0x00E9, 0x00EA, 0x00EB, 0x00EC, 0x00ED, 0x00EE, 0x00EF,
		0x00F0, 0x00F1, 0x00F2, 0x00F3, 0x00F4, 0x00F5, 0x00F6, 0x00F7,	// 0xf
		0x00F8, 0x00F9, 0x00FA, 0x00FB, 0x00FC, 0x00FD, 0x00FE, 0x00FF
	};

	private static final char[] fromUnicodeMap =
	{
		0x0131, 0x9A,	// Latin small letter dotless i			dotlessi
		0x0141, 0x95,	// Latin capital ltter L with stroke		Lslash
		0x0142, 0x9B,	// Latin small letter l with stroke		lslash
		0x0152, 0x96,	// Latin capital ligature OE			OE
		0x0153, 0x9C,	// Latin small ligature oe			oe
		0x0160, 0x97,	// Latin capital letter S with caron		Scaron
		0x0161, 0x9D,	// Latin small letter s with caron		scaron
		0x0178, 0x98,	// Latin capital letter Y with diaeresis	Ydieresis
		0x017D, 0x99,	// Latin capital letter Z with caron		Zcaron
		0x017E, 0x9E,	// Latin small letter Z with caron		zcaron
		0x0192, 0x86,	// Latin small letter f with hook		florin
		0x02C6, 0x1A,	// modified letter circumflex accent		circumflex
		0x02C7, 0x19,	// caron					caron
		0x02D8, 0x18,	// breve					breve
		0x02D9, 0x1B,	// dot above					dotaccent
		0x02DA, 0x1E,	// ring above					ring
		0x02DB, 0x1D,	// ogonek					ogonek
		0x02DC, 0x1F,	// small tilde					tilde
		0x02DD, 0x1C,	// double acute accent				hungarumlaut
		0x2013, 0x85,	// en dash					endash
		0x2014, 0x84,	// em dash					emdash
		0x2018, 0x8F,	// left single quotation mark			quoteleft
		0x2019, 0x90,	// right single quotation mark			quoteright
		0x201A, 0x91,	// single low-9 Quotation mark			quotesinglbase
		0x201C, 0x8D,	// left double quotation mark			quotedblleft
		0x201D, 0x8E,	// right double quotation mark			quotedblright
		0x201E, 0x8C,	// double low-9 quotation mark			quotedblbase
		0x2020, 0x81,	// dagger					dagger
		0x2021, 0x82,	// double dagger				daggerdbl
		0x2022, 0x80,	// bullet					bullet
		0x2026, 0x83,	// horizontal ellipsis				ellipsis
		0x2030, 0x8B,	// per mille sign				perthousand
		0x2039, 0x88,	// single left-pointing angle quotation mark	guilsinglleft
		0x203A, 0x89,	// single right-pointing angle quotation mark	guilsinglright
		0x2044, 0x87,	// fraction slash				fraction
		0x20AC, 0xA0,	// Euro						Euro
		0x2122, 0x92,	// trade mark sign				trademark
		0x2212, 0x8A,	// minus sign					minus
		0xFB01, 0x93,	// LATIN SMALL LIGATURE FI			fi
		0xFB02, 0x94,	// LATIN SMALL LIGATURE FL			fl
	};

	private static final char ESCAPE_START = '\\';
	private static final char DEFAULT_CHARACTER = '.';
	
	
	/**
	 * Same as UCSDecomposeTable defined in UcsUtil.cpp file of Acrobat
	 */
//	private static Character UCSDecomposeTable[][] = null;
//	static{
//		ObjectInputStream is = null;
//		try {
//			is = new ObjectInputStream(PDFDocEncoding.class.getResourceAsStream("encoding/serTable"));
//			UCSDecomposeTable = (Character[][]) is.readObject();
//		} catch (IOException e) {
//			throw new PDFRuntimeException("Error while initializing UCSDecomposeTable table.", e);
//		} catch (ClassNotFoundException e) {
//			throw new PDFRuntimeException("Error while initializing UCSDecomposeTable table.", e);
//		}
//	}
	
	/**
	 * Same as UCS2PDFDocTable defined in UnicodeStrings.cpp file of Acrobat
	 */
	private static final char[][] UCS2PDFDocTable =
		{
		{0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f
		,0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17,0x18,0x19,0x1a,0x1b,0x1c,0x1d,0x1e,0x1f
		,0x20,0x21,0x22,0x23,0x24,0x25,0x26,0x27,0x28,0x29,0x2a,0x2b,0x2c,0x2d,0x2e,0x2f
		,0x30,0x31,0x32,0x33,0x34,0x35,0x36,0x37,0x38,0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f
		,0x40,0x41,0x42,0x43,0x44,0x45,0x46,0x47,0x48,0x49,0x4a,0x4b,0x4c,0x4d,0x4e,0x4f
		,0x50,0x51,0x52,0x53,0x54,0x55,0x56,0x57,0x58,0x59,0x5a,0x5b,0x5c,0x5d,0x5e,0x5f
		,0x60,0x61,0x62,0x63,0x64,0x65,0x66,0x67,0x68,0x69,0x6a,0x6b,0x6c,0x6d,0x6e,0x6f
		,0x70,0x71,0x72,0x73,0x74,0x75,0x76,0x77,0x78,0x79,0x7a,0x7b,0x7c,0x7d,0x7e,0x95
		,0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80	/* 0x80 through 0x9F are undefined in Unicode */
		,0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80	 
		,0x20,0xa1,0xa2,0xa3,0xa4,0xa5,0xa6,0xa7,0xa8,0xa9,0xaa,0xab,0xac,0x00,0xae,0xaf	/* 0xA0,0xAD are legal unicode, but not PDFDoc */
		,0xb0,0xb1,0xb2,0xb3,0xb4,0xb5,0xb6,0xb7,0xb8,0xb9,0xba,0xbb,0xbc,0xbd,0xbe,0xbf
		,0xc0,0xc1,0xc2,0xc3,0xc4,0xc5,0xc6,0xc7,0xc8,0xc9,0xca,0xcb,0xcc,0xcd,0xce,0xcf
		,0xd0,0xd1,0xd2,0xd3,0xd4,0xd5,0xd6,0xd7,0xd8,0xd9,0xda,0xdb,0xdc,0xdd,0xde,0xdf
		,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe8,0xe9,0xea,0xeb,0xec,0xed,0xee,0xef
		,0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,0xf8,0xf9,0xfa,0xfb,0xfc,0xfd,0xfe,0xff
		}, /* high byte 0 */
		{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x9a,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x95,0x9b,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x96,0x9c,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x97,0x9d,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x98,0x00,0x00,0x00,0x00,0x99,0x9e,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x86,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		}, /* high byte 1 */
		{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x1a,0x19,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x1b,0x00,0x1d,0x1f,0x1c,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		}, /* high byte 2 */
		null, /* high byte 3 */
		null, /* high byte 4 */
		null, /* high byte 5 */
		null, /* high byte 6 */
		null, /* high byte 7 */
		null, /* high byte 8 */
		null, /* high byte 9 */
		null, /* high byte 10 */
		null, /* high byte 11 */
		null, /* high byte 12 */
		null, /* high byte 13 */
		null, /* high byte 14 */
		null, /* high byte 15 */
		null, /* high byte 16 */
		null, /* high byte 17 */
		null, /* high byte 18 */
		null, /* high byte 19 */
		null, /* high byte 20 */
		null, /* high byte 21 */
		null, /* high byte 22 */
		null, /* high byte 23 */
		null, /* high byte 24 */
		null, /* high byte 25 */
		null, /* high byte 26 */
		null, /* high byte 27 */
		null, /* high byte 28 */
		null, /* high byte 29 */
		null, /* high byte 30 */
		null, /* high byte 31 */
		{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x85,0x84,0x00,0x00,0x00,0x8f,0x90,0x91,0x00,0x8d,0x8e,0x8c,0x00
		,0x81,0x82,0x80,0x00,0x00,0x00,0x83,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x8b,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x88,0x89,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x87,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xa0,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		}, /* high byte 32 */
		{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x92,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		}, /* high byte 33 */
		{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x8a,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		}, /* high byte 34 */
		null, /* high byte 35 */
		null, /* high byte 36 */
		null, /* high byte 37 */
		null, /* high byte 38 */
		null, /* high byte 39 */
		null, /* high byte 40 */
		null, /* high byte 41 */
		null, /* high byte 42 */
		null, /* high byte 43 */
		null, /* high byte 44 */
		null, /* high byte 45 */
		null, /* high byte 46 */
		null, /* high byte 47 */
		null, /* high byte 48 */
		null, /* high byte 49 */
		null, /* high byte 50 */
		null, /* high byte 51 */
		null, /* high byte 52 */
		null, /* high byte 53 */
		null, /* high byte 54 */
		null, /* high byte 55 */
		null, /* high byte 56 */
		null, /* high byte 57 */
		null, /* high byte 58 */
		null, /* high byte 59 */
		null, /* high byte 60 */
		null, /* high byte 61 */
		null, /* high byte 62 */
		null, /* high byte 63 */
		null, /* high byte 64 */
		null, /* high byte 65 */
		null, /* high byte 66 */
		null, /* high byte 67 */
		null, /* high byte 68 */
		null, /* high byte 69 */
		null, /* high byte 70 */
		null, /* high byte 71 */
		null, /* high byte 72 */
		null, /* high byte 73 */
		null, /* high byte 74 */
		null, /* high byte 75 */
		null, /* high byte 76 */
		null, /* high byte 77 */
		null, /* high byte 78 */
		null, /* high byte 79 */
		null, /* high byte 80 */
		null, /* high byte 81 */
		null, /* high byte 82 */
		null, /* high byte 83 */
		null, /* high byte 84 */
		null, /* high byte 85 */
		null, /* high byte 86 */
		null, /* high byte 87 */
		null, /* high byte 88 */
		null, /* high byte 89 */
		null, /* high byte 90 */
		null, /* high byte 91 */
		null, /* high byte 92 */
		null, /* high byte 93 */
		null, /* high byte 94 */
		null, /* high byte 95 */
		null, /* high byte 96 */
		null, /* high byte 97 */
		null, /* high byte 98 */
		null, /* high byte 99 */
		null, /* high byte 100 */
		null, /* high byte 101 */
		null, /* high byte 102 */
		null, /* high byte 103 */
		null, /* high byte 104 */
		null, /* high byte 105 */
		null, /* high byte 106 */
		null, /* high byte 107 */
		null, /* high byte 108 */
		null, /* high byte 109 */
		null, /* high byte 110 */
		null, /* high byte 111 */
		null, /* high byte 112 */
		null, /* high byte 113 */
		null, /* high byte 114 */
		null, /* high byte 115 */
		null, /* high byte 116 */
		null, /* high byte 117 */
		null, /* high byte 118 */
		null, /* high byte 119 */
		null, /* high byte 120 */
		null, /* high byte 121 */
		null, /* high byte 122 */
		null, /* high byte 123 */
		null, /* high byte 124 */
		null, /* high byte 125 */
		null, /* high byte 126 */
		null, /* high byte 127 */
		null, /* high byte 128 */
		null, /* high byte 129 */
		null, /* high byte 130 */
		null, /* high byte 131 */
		null, /* high byte 132 */
		null, /* high byte 133 */
		null, /* high byte 134 */
		null, /* high byte 135 */
		null, /* high byte 136 */
		null, /* high byte 137 */
		null, /* high byte 138 */
		null, /* high byte 139 */
		null, /* high byte 140 */
		null, /* high byte 141 */
		null, /* high byte 142 */
		null, /* high byte 143 */
		null, /* high byte 144 */
		null, /* high byte 145 */
		null, /* high byte 146 */
		null, /* high byte 147 */
		null, /* high byte 148 */
		null, /* high byte 149 */
		null, /* high byte 150 */
		null, /* high byte 151 */
		null, /* high byte 152 */
		null, /* high byte 153 */
		null, /* high byte 154 */
		null, /* high byte 155 */
		null, /* high byte 156 */
		null, /* high byte 157 */
		null, /* high byte 158 */
		null, /* high byte 159 */
		null, /* high byte 160 */
		null, /* high byte 161 */
		null, /* high byte 162 */
		null, /* high byte 163 */
		null, /* high byte 164 */
		null, /* high byte 165 */
		null, /* high byte 166 */
		null, /* high byte 167 */
		null, /* high byte 168 */
		null, /* high byte 169 */
		null, /* high byte 170 */
		null, /* high byte 171 */
		null, /* high byte 172 */
		null, /* high byte 173 */
		null, /* high byte 174 */
		null, /* high byte 175 */
		null, /* high byte 176 */
		null, /* high byte 177 */
		null, /* high byte 178 */
		null, /* high byte 179 */
		null, /* high byte 180 */
		null, /* high byte 181 */
		null, /* high byte 182 */
		null, /* high byte 183 */
		null, /* high byte 184 */
		null, /* high byte 185 */
		null, /* high byte 186 */
		null, /* high byte 187 */
		null, /* high byte 188 */
		null, /* high byte 189 */
		null, /* high byte 190 */
		null, /* high byte 191 */
		null, /* high byte 192 */
		null, /* high byte 193 */
		null, /* high byte 194 */
		null, /* high byte 195 */
		null, /* high byte 196 */
		null, /* high byte 197 */
		null, /* high byte 198 */
		null, /* high byte 199 */
		null, /* high byte 200 */
		null, /* high byte 201 */
		null, /* high byte 202 */
		null, /* high byte 203 */
		null, /* high byte 204 */
		null, /* high byte 205 */
		null, /* high byte 206 */
		null, /* high byte 207 */
		null, /* high byte 208 */
		null, /* high byte 209 */
		null, /* high byte 210 */
		null, /* high byte 211 */
		null, /* high byte 212 */
		null, /* high byte 213 */
		null, /* high byte 214 */
		null, /* high byte 215 */
		null, /* high byte 216 */
		null, /* high byte 217 */
		null, /* high byte 218 */
		null, /* high byte 219 */
		null, /* high byte 220 */
		null, /* high byte 221 */
		null, /* high byte 222 */
		null, /* high byte 223 */
		null, /* high byte 224 */
		null, /* high byte 225 */
		null, /* high byte 226 */
		null, /* high byte 227 */
		null, /* high byte 228 */
		null, /* high byte 229 */
		null, /* high byte 230 */
		null, /* high byte 231 */
		null, /* high byte 232 */
		null, /* high byte 233 */
		null, /* high byte 234 */
		null, /* high byte 235 */
		null, /* high byte 236 */
		null, /* high byte 237 */
		null, /* high byte 238 */
		null, /* high byte 239 */
		null, /* high byte 240 */
		null, /* high byte 241 */
		null, /* high byte 242 */
		null, /* high byte 243 */
		null, /* high byte 244 */
		null, /* high byte 245 */
		null, /* high byte 246 */
		null, /* high byte 247 */
		null, /* high byte 248 */
		null, /* high byte 249 */
		null, /* high byte 250 */
		{0x00,0x93,0x94,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		}, /* high byte 251 */
		null, /* high byte 252 */
		null, /* high byte 253 */
		null, /* high byte 254 */
		{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0xa3,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
		,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x95,0x00,0x00
		} /* high byte 255 */
		};
	
	/**
	 * Tries to convert the UTF-16 bytes passed to PDF Doc encoding.
	 * This method copies the logic of ASTextGetPDTextCopy method in ASText.cpp file of Acrobat. 
	 * @param input
	 * @param encInfo This is the encoding information of the output.
	 */
	public static void convertUTF16ToPDFDocEncoding(byte[] input, EncodedData<char[]> encInfo){
		
		BooleanHolder bOutOfRange = new BooleanHolder(false);
		int finalChars = UCSToEncStrInternal(input, encInfo.getEncodedOutput(), bOutOfRange);
		if(bOutOfRange.getValue() || finalChars < 0){
//			ASUns16 *newUCS;
//			ASUns16 *putIt;
//
////			if (result)
////			{
////				ASfree(result);
//				result = null;
////			}
//
//			numChars = (str->ucsLen / sizeof(ASUns16)) + 1; /* For 0xFE 0xFF */
//
//			if (str->language)
//			{
//				numChars += 3; /* For two escape characters and language */
//				if (str->country)
//					numChars++; /* For country code */
//			}
//
//			if (len)
//				*len = numChars * sizeof(ASUns16);
//			
//			newUCS = (ASUns16 *) ASSureMalloc((numChars + 1) * sizeof(ASUns16));
// 
//			putIt = newUCS;
//			WriteBigEndian(putIt++, (ASUns16) 0xFEFF);
//			numChars -= 1;
//			if (str->language)
//			{
//				WriteBigEndian(putIt++, UniESCAPE);
//				WriteBigEndian(putIt++, str->language);
//				numChars -= 2;
//				if (str->country)
//				{
//					WriteBigEndian(putIt++, str->country);
//					numChars -= 1;
//				}
//				WriteBigEndian(putIt++, UniESCAPE);
//				numChars -= 1;
//			}
//
//			ASmemcpy(putIt, str->ucs, (numChars + 1) * sizeof(ASUns16));
//			result = (char *) newUCS;
//			char[] toReturnData = new char[input.byteData.length-2];
			for(int i=2;i<input.length;i++)// first two bytes should represent BOM character because input is UTF-16 encoded.
				encInfo.getEncodedOutput()[i-2] = (char) input[i];
			encInfo.setEncodedDataLength(input.length-2);
		}
		else{
			encInfo.setEncodedDataLength(finalChars);
			encInfo.setEncoding(Encoding.PDFDoc);// successfully converted to PDF Doc encoding.
			
		}
	}
	
	/**
	 * Tries to convert the UTF-16 bytes passed to PDF Doc encoding.
	 * This method copies the logic of UCSToEncStrInternal method in PDConvrt.cpp file of Acrobat. 
	 * @param textIn
	 * @param textOut
	 * @param bOutOfRange
	 * @return int
	 */
	private static int UCSToEncStrInternal(byte[] textIn, char[] textOut, BooleanHolder bOutOfRange){
		int outLen = 0;
		int outByteSize = textOut.length;
		BooleanHolder bOut = new BooleanHolder(false);
		char byteChar = 0;
//		Character[] decStr = null;
//		int decLen = 0;
		if(bOutOfRange != null)
			bOutOfRange.setValue(false);
		for(int i=2;i<textIn.length;i+=2){
			int secondByte = 0;
			if(i+1!=textIn.length){
				secondByte = (textIn[i+1]<0?textIn[i+1]+256:textIn[i+1]);
			}
			int code =  ((textIn[i]<0 ? textIn[i]+256:textIn[i]) << 8) + secondByte;
			if ((code < 0x202A) || (code > 0x202E)){
				byteChar = convProc(code, bOut);
				if(textOut != null && outLen < outByteSize)
					textOut[outLen] = byteChar;
				outLen++;
				if (bOutOfRange != null && bOut.getValue())
					bOutOfRange.setValue(true);
				if (bOut.getValue())	/* Decompose the UCS code if no character found in the destination encoding */
				{
					return -1;
//					decStr = ASUCSDecomposeHost(code);
//					decLen = decStr.length;
//					if (decStr != null && decLen > 0)
//					{
//						outLen--;	/* replace the undefined character */
//						for (int j = 0; j < decLen; j++)
//						{
//							if ( (textOut != null) && (outLen < outByteSize) )
//								textOut[outLen] = convProc(decStr[j], bOut);
//							outLen++;
//						}
//					}
				}
			}
		}
		if(outLen > outByteSize)
			outLen = -outByteSize;
		return outLen;
	}
	
//	/**
//	 * Method copied from ASUCSDecomposeHost method in UcsUtil.cpp file of Acrobat.
//	 * @param ucs
//	 * @return {@link Character}[]
//	 */
//	private static Character[] ASUCSDecomposeHost(int ucs)
//	{
//		/* Adobe patent application tracking # B106, entitled Method to Improve application launch time by reducing code paging of static initializers, inventors: Dylan Ashe, Brent Rosenquist, and Dave Sawyer */
//
//		int gUCSDecomposeTableLength = UCSDecomposeTable.length;
//		char gUCSDecomposeTableMin = UCSDecomposeTable[0][0];
//		char gUCSDecomposeTableMax = UCSDecomposeTable[gUCSDecomposeTableLength - 1][0];
////
//	    int low, mid = 0, high;
//		Character[] decompStr = null;
//		assert(ucs != 0xFFFE);		/* ASUCSDecompose() must be called with big-endian UCS number. */
////
//	    if ((ucs < gUCSDecomposeTableMin) || (ucs > gUCSDecomposeTableMax) ||
//	        ((0x33FE < ucs) && (ucs < 0xFB00)))
//	        return null;
////
////		BEGIN_THREAD_LOCAL_BLOCK
////		if (ThreadLocalBlockAccess(gUCSCharDepot)->decomposeCache == NULL)	/* need initialization? */
////			ThreadLocalBlockAccess(gUCSCharDepot)->decomposeCache = new UCSCharCache<const ASUns16 *>(UCS_CHARCACHE_LIMIT);
////
////		//if (ThreadLocalBlockAccess(gUCSCharDepot)->decomposeCache->GetCharInfo(ucs, &decompStr))
////			//return decompStr;
////
////	    /* do the binary search */
//	    low = 0;
//	    high = gUCSDecomposeTableLength - 1;
//	    while (low <= high)
//	    {
//	    	char val;
////
//	        mid = (low + high) / 2;
//	        val = UCSDecomposeTable[mid][0];
////
//	        if (ucs < val)
//	            high = mid - 1;
//	        else if (ucs > val)
//	            low = mid + 1;
//	        else    /* found match */
//	            break;
//	    }
////
//	    int outLength = UCSDecomposeTable[mid].length - 1;
//	    if(outLength == 0 || low > high)
//	    	return null;
//	    
//	    decompStr = new Character[outLength];
//	    System.arraycopy(UCSDecomposeTable[mid], 1, decompStr, 0, outLength);
////		ThreadLocalBlockAccess(gUCSCharDepot)->decomposeCache->AddCharInfo(ucs, decompStr);
//		return decompStr;
////		END_THREAD_LOCAL_BLOCK
//	}
	
	/**
	 * This method is copied from PDFDocEncFromUCSChar method in PDConvrt.cpp file
	 * of Acrobat.
	 */
	private static char convProc(int code, BooleanHolder outOfRange){
		char[] row = UCS2PDFDocTable[code >> 8];
		char rcode = row != null ? row[code & 0xff] : 0;

		// we got a code back.  return it
		if(rcode != 0)
		{
			outOfRange.setValue(false);
			return rcode;
		}
		// we didn't get a code back but our input was 0... return 0
		else if(code == 0)
		{
			outOfRange.setValue(false);
			return 0;
		}
		else {
			outOfRange.setValue(true);
			/* They didn't pass 0, and we got 0 from the table */
			return PDFDOCENC_UNDEFINED_CHAR;
		}
	}

	/**
	 * All of these constants are copied from PDConvrt.cpp file of Acrobat.
	 */
	private static final char PDFDOCENC_UNDEFINED_CHAR = 149;
	private static final char UCS2_UNDEFINED_CHAR = 0xfffd;
	private static final char UCS2_SPECIAL_FI_LIGATURE = 0xfffe;
	private static final char UCS2_SPECIAL_FL_LIGATURE = 0xffff;
	private static final char PDFDoc2UCSTable[] =
	{
		0x0000,
		0x0001,
		0x0002,
		0x0003,
		0x0004,
		0x0005,
		0x0006,
		0x0007,
		0x0008,
		0x0009,
		0x000a,
		0x000b,
		0x000c,
		0x000d,
		0x000e,
		0x000f,
		0x0010,
		0x0011,
		0x0012,
		0x0013,
		0x0014,
		0x0015,
		0x0016,
		0x0017,
		0x02d8,
		0x02c7,
		0x02c6,
		0x02d9,
		0x02dd,
		0x02db,
		0x00b0,
		0x02dc,
		0x0020,
		0x0021,
		0x0022,
		0x0023,
		0x0024,
		0x0025,
		0x0026,
		0x0027,
		0x0028,
		0x0029,
		0x002a,
		0x002b,
		0x002c,
		0x002d,
		0x002e,
		0x002f,
		0x0030,
		0x0031,
		0x0032,
		0x0033,
		0x0034,
		0x0035,
		0x0036,
		0x0037,
		0x0038,
		0x0039,
		0x003a,
		0x003b,
		0x003c,
		0x003d,
		0x003e,
		0x003f,
		0x0040,
		0x0041,
		0x0042,
		0x0043,
		0x0044,
		0x0045,
		0x0046,
		0x0047,
		0x0048,
		0x0049,
		0x004a,
		0x004b,
		0x004c,
		0x004d,
		0x004e,
		0x004f,
		0x0050,
		0x0051,
		0x0052,
		0x0053,
		0x0054,
		0x0055,
		0x0056,
		0x0057,
		0x0058,
		0x0059,
		0x005a,
		0x005b,
		0x005c,
		0x005d,
		0x005e,
		0x005f,
		0x0060,
		0x0061,
		0x0062,
		0x0063,
		0x0064,
		0x0065,
		0x0066,
		0x0067,
		0x0068,
		0x0069,
		0x006a,
		0x006b,
		0x006c,
		0x006d,
		0x006e,
		0x006f,
		0x0070,
		0x0071,
		0x0072,
		0x0073,
		0x0074,
		0x0075,
		0x0076,
		0x0077,
		0x0078,
		0x0079,
		0x007a,
		0x007b,
		0x007c,
		0x007d,
		0x007e,
		UCS2_UNDEFINED_CHAR,
		0x2022,
		0x2020,
		0x2021,
		0x2026,
		0x2014,
		0x2013,
		0x0192,
		0x2044,
		0x2039,
		0x203a,
		0x2212,
		0x2030,
		0x201e,
		0x201c,
		0x201d,
		0x2018,
		0x2019,
		0x201a,
		0x2122,
		UCS2_SPECIAL_FI_LIGATURE, // Windows doesn't support ligature unicode values
		UCS2_SPECIAL_FL_LIGATURE,
		0x0141,
		0x0152,
		0x0160,
		0x0178,
		0x017d,
		0x0131,
		0x0142,
		0x0153,
		0x0161,
		0x017e,
		UCS2_UNDEFINED_CHAR,
		0x20ac,
		0x00a1,
		0x00a2,
		0x00a3,
		0x00a4,
		0x00a5,
		0x00a6,
		0x00a7,
		0x00a8,
		0x00a9,
		0x00aa,
		0x00ab,
		0x00ac,
		UCS2_UNDEFINED_CHAR,
		0x00ae,
		0x00af,
		0x00b0,
		0x00b1,
		0x00b2,
		0x00b3,
		0x00b4,
		0x00b5,
		0x00b6,
		0x00b7,
		0x00b8,
		0x00b9,
		0x00ba,
		0x00bb,
		0x00bc,
		0x00bd,
		0x00be,
		0x00bf,
		0x00c0,
		0x00c1,
		0x00c2,
		0x00c3,
		0x00c4,
		0x00c5,
		0x00c6,
		0x00c7,
		0x00c8,
		0x00c9,
		0x00ca,
		0x00cb,
		0x00cc,
		0x00cd,
		0x00ce,
		0x00cf,
		0x00d0,
		0x00d1,
		0x00d2,
		0x00d3,
		0x00d4,
		0x00d5,
		0x00d6,
		0x00d7,
		0x00d8,
		0x00d9,
		0x00da,
		0x00db,
		0x00dc,
		0x00dd,
		0x00de,
		0x00df,
		0x00e0,
		0x00e1,
		0x00e2,
		0x00e3,
		0x00e4,
		0x00e5,
		0x00e6,
		0x00e7,
		0x00e8,
		0x00e9,
		0x00ea,
		0x00eb,
		0x00ec,
		0x00ed,
		0x00ee,
		0x00ef,
		0x00f0,
		0x00f1,
		0x00f2,
		0x00f3,
		0x00f4,
		0x00f5,
		0x00f6,
		0x00f7,
		0x00f8,
		0x00f9,
		0x00fa,
		0x00fb,
		0x00fc,
		0x00fd,
		0x00fe,
		0x00ff,
	};
	
	/**
	 * This method converts PDF Doc encoded data to UCS format.
	 * Logic is copied from EncToUCSStrInternal method in PDConvrt.cpp file of Acrobat.
	 * @param inputEncInfo
	 * @param outputEncInfo
	 */
	public static void encToUCSStrInternal(EncodedData<char[]> inputEncInfo, EncodedData<byte[]> outputEncInfo){
		char code = 0;
		int i,j = 0;
		if(inputEncInfo.encoding == outputEncInfo.encoding){
			for(int k=0;k<inputEncInfo.encodedDataLength;k++){
				outputEncInfo.getEncodedOutput()[k] = (byte) inputEncInfo.getEncodedOutput()[k];
			}
			outputEncInfo.encodedDataLength = inputEncInfo.encodedDataLength;
			return;
		}
			
		for (i=j=0; i < inputEncInfo.encodedDataLength; i++)
		{
				byte index = (byte) inputEncInfo.getEncodedOutput()[i];
				code = PDFDoc2UCSTable[(index < 0 ? index + 256 : index)];
			
			/* Is this "fi" ligature? */
			if (UCS2_SPECIAL_FI_LIGATURE == code)
			{
				code = 'f';
				if (outputEncInfo.encodedDataLength > j + 1)
				{
					// textOut may not be aligned, so write must
	 				// be done bytewise (in general, casting a
	 				// pointer to a pointer of a larger type is
	 				// dangerous).  Bug 1187816

					// *((ASUns16 *) (&textOut[j])) = WORD2BIGENDIAN(code);
					outputEncInfo.getEncodedOutput()[j] = (byte) (code >> 8);
					outputEncInfo.getEncodedOutput()[j+1] = (byte) (code & 0xff);
				}
				code = 'i';
				j += 2;
			}
			/* Is this "fl" ligature? */
			else if (UCS2_SPECIAL_FL_LIGATURE == code)
			{
				code = 'f';
				if (outputEncInfo.encodedDataLength > j + 1)
				{
					// *((ASUns16 *) (&textOut[j])) = WORD2BIGENDIAN(code);
					outputEncInfo.getEncodedOutput()[j] = (byte) (code >> 8);
					outputEncInfo.getEncodedOutput()[j+1] = (byte) (code & 0xff);
				}
				code = 'l';
				j += 2;
			}
			if (outputEncInfo.encodedDataLength > j + 1)
			{
				// *((ASUns16 *) (&textOut[j])) = WORD2BIGENDIAN(code);
				outputEncInfo.getEncodedOutput()[j] = (byte) (code >> 8);
				outputEncInfo.getEncodedOutput()[j+1] = (byte) (code & 0xff);
			}
			j += 2;
		}
		
		outputEncInfo.encodedDataLength = outputEncInfo.encodedDataLength >= j ? j : -j;
	}
}
