/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.analytics.testandtarget.util;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.apache.commons.lang.StringUtils;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ValueMap;

import com.day.cq.analytics.testandtarget.TargetMediator;
import com.day.cq.analytics.testandtarget.TestandtargetService;
import com.day.cq.personalization.impl.util.PersonalizationConstants;
import com.day.cq.wcm.api.Page;
import com.day.cq.wcm.api.WCMMode;

public class OfferHelper {
    
    private static final String DISPLAY_MBOX = "* display mbox *";

    /** Third-Party property name */
    public static final String PN_THIRDPARTYID = "ttThirdPartyId";

    /** Campaign resource type */
    public static final String RT_CAMPAIGN = "cq/personalization/components/campaignpage";
    
    /** Experience resource type */
    public static final String RT_EXPERIENCE = "cq/personalization/components/experiencepage";
    
    /** Offer resource type*/
    public static final String RT_TEASER = "cq/personalization/components/teaserpage";

    /** Offer proxy resource type **/
    public static final String RT_OFFER_PROXY="cq/personalization/components/offerproxy";
    
    /** Account-Reference property name */
    public static final String PN_TESTANDTARGETCONFIG = "testandtargetconfig";
    
    /** Mbox location property name */
    public static final String PN_LOCATION = "location";

    /**
     * Returns an Offer name constructed by concatenating the path and replacing
     * slashes (/) with dashes (-).
     * 
     * @param path path
     * @return offername constructed by path
     */
    public static String getOfferName(String path) {
        return path.substring(1).replace("/", "-");
    }

    /**
     * Returns the offer name based on this offer's location. If this offer doesn't have a
     * location then this method calls {@link OfferHelper#getOfferName(String)}
     * @param offerPage the offer page
     * @param wcmMode the WCM mode. This is required to determine the offer location
     * @return the offer name
     */
    public static String getOfferName(Page offerPage, WCMMode wcmMode) {
        String location = offerPage.getProperties().get("location", "");
        if ("".equals(location)) {
            return getOfferName(offerPage.getPath());
        }

        StringBuilder offerName = new StringBuilder(getOfferLocation(offerPage,wcmMode));
        offerName.append("-").append(offerPage.getParent().getName())
                .append("-").append(offerPage.getName());

        return offerName.toString();
    }

    /**
     * Returns the offer name based on this offer's location. If this offer doesn't have a
     * location then this method calls {@link OfferHelper#getOfferName(String)}
     * @param offerPage the offer page
     * @param wcmMode the WCM mode. This is required to determine the offer location
     * @param ambitName the name of the ambit that this offer is part of.
     * @return the offer name, following this pattern - {location-name}--{wcm mode}-{ambit-name}-{experience name}-{offer page name}.
     * If the WCM mode is {@link WCMMode#DISABLED} then it's ommited. If the ambit name is {@link PersonalizationConstants#AMBIT_DEFAULT_NAME} then it's omitted.
     */
    public static String getOfferName(Page offerPage, WCMMode wcmMode, String ambitName) {
        if (ambitName.equals(PersonalizationConstants.AMBIT_DEFAULT_NAME)) {
            return getOfferName(offerPage, wcmMode);
        }

        StringBuilder offerName = new StringBuilder(getOfferLocation(offerPage,wcmMode));
        return  StringUtils.join(new String[] { offerName.toString(), ambitName, offerPage.getParent().getName(), offerPage.getName()}, "-");
    }

    /**
     * Returns a campaign name based on a repository path, following the same rules as {@link #getOfferName(String)}
     * 
     * @param path path
     * @return the campaign name
     */
    public static String getCampaignName(String path) {
        
        return getOfferName(path);
    }
    
    /**
     * Saves or deletes the third party ID property in the provided {@link Node}
     * according to the executed operation.
     * 
     * @param node {@link Node}
     * @param operation operation
     * @param thirdpartyid thirdpartyid
     * @throws RepositoryException {@link RepositoryException}
     */
    public static synchronized void setThirdPartyID(Node node, String operation, String thirdpartyid) throws RepositoryException {    
        if(TestandtargetService.OPERATION_SAVE.equals(operation)) {
            // save property
            node.setProperty(PN_THIRDPARTYID, thirdpartyid);
        }else if(TestandtargetService.OPERATION_DELETE.equals(operation)) {
            // remove property from node
            if(node.hasProperty(PN_THIRDPARTYID)) {
                node.getProperty(PN_THIRDPARTYID).remove();
            }
        }
    }
    
    /**
     * Returns the third party ID of a Node if property exists, otherwise one is
     * generated in Timestamp format.
     * 
     * @param node
     *            Offer node
     * @return Third-Party ID
     * @throws RepositoryException {@link RepositoryException}
     */
    public static String getThirdPartyID(Node node) throws RepositoryException {
        String thirdPartyID = null;
        if(node.hasProperty(PN_THIRDPARTYID)) {
            thirdPartyID = node.getProperty(PN_THIRDPARTYID).getString();
        } else {
            thirdPartyID = Long.toString(System.currentTimeMillis());
        }
        return thirdPartyID;
    }

    /**
     * Returns the third party id for a campaign page's resource node
     * 
     * @param contentResourceNode  {@link Node} of pages contentresourcenode
     * @return the third party id
     * @throws RepositoryException {@link RepositoryException}
     */
    public static String getThirdPartyCampaignId(Node contentResourceNode) throws RepositoryException{
        
        if ( contentResourceNode.hasProperty(PN_THIRDPARTYID) )
            return contentResourceNode.getProperty(PN_THIRDPARTYID).getString();
        
        return getCampaignName(contentResourceNode.getParent().getPath());
    }
    
    /**
     * Saves the configuration reference to the campaign page
     * 
     * @param path Configuration path
     * @param page Current offer page
     * @throws RepositoryException in case an error occurs when node property is set 
     */
    public static synchronized void setConfigurationPath(String path, Page page) throws RepositoryException {
        Page campaignPage = OfferHelper.getCampaign(page);
        if(campaignPage != null) {
            Node campaignContent = campaignPage.getContentResource().adaptTo(Node.class);
            campaignContent.setProperty(PN_TESTANDTARGETCONFIG, path);
        }
    }
    
    /**
     * Returns a page with resource type
     * 'cq/personalization/components/campaignpage' or {@code null}. The method
     * walks ascending up the hierarchy until a page with the corresponding resource
     * type is found.
     * 
     * @param page
     *            Page with resource type
     *            'cq/personalization/components/campaignpage' or initially provided
     *            page.
     * @return page with campaignpage resourcetype or {@code null}
     */
    public static Page getCampaign(Page page) {
        if (page == null) {
            return null;
        }

        Page campaignPage = page;
        while (campaignPage != null) {
            Resource content = campaignPage.getContentResource();
            if (content != null && RT_CAMPAIGN.equals(content.getResourceType())) {
                return campaignPage;
            }
            campaignPage = campaignPage.getParent();
        }

        return null;
    }


    /**
     * Returns the offer location as it should be sent to Adobe Target
     * 
     * @param offerPage The offer page from CQ
     * @param wcmMode The mode for which the location is calculated . Used to create different locations for author and
     *            publish environments
     * @return the offer location or {@value #DISPLAY_MBOX} if no location property is found in the offerPage
     */
    public static String getOfferLocation(Page offerPage, WCMMode wcmMode) {
        
        String location = offerPage.getProperties().get("location", "");
        if ( "".equals(location) )
            return DISPLAY_MBOX;
        
        return MboxHelper.qualifyMboxNameOrId(MboxHelper.getMboxId(location), wcmMode);
    }

    /**
     * Returns the offer location as it should be sent to Adobe Target
     *
     * @param offerPage The offer page from CQ
     * @param wcmMode The mode for which the location is calculated . Used to create different locations for author and
     *            publish environments
     * @param ambitName the name of the MSM ambit that this offer is part of
     * @return the offer location or {@value #DISPLAY_MBOX} if no location property is found in the offerPage
     */
    public static String getOfferLocation(Page offerPage, WCMMode wcmMode, String ambitName) {

        if (StringUtils.isEmpty(ambitName) || (PersonalizationConstants.AMBIT_DEFAULT_NAME.equals(ambitName))) {
            return getOfferLocation(offerPage, wcmMode);
        }

        String location = offerPage.getProperties().get("location", "");
        if ( "".equals(location) )
            return DISPLAY_MBOX;

        return MboxHelper.qualifyMboxNameOrId(MboxHelper.getMboxId(location), wcmMode);
    }


    /**
     * Returns the offer id from Target
     * @param offerPage the offer page from CQ
     * @return a value of type long representing the Id, or 0 if the id is not present.
     */
    public static long getOfferId(Page offerPage) {
       return getOfferId(offerPage, WCMMode.DISABLED);
    }

    /**
     * Returns the id of the offer set by Target. The id is one of the two *externalId properties of the offer page
     * @param offerPage the offer page from AEM
     * @param wcmMode the {@link WCMMode}. If the {@link WCMMode} is {@link WCMMode#EDIT} then the id value is read from the cq:authorExternalId property, otherwise the value of the cq:publishExternalId property is returned
     * @return a value of type long representing the Id, or 0 if the id is not present.
     */
    public static long getOfferId(Page offerPage, WCMMode wcmMode) {
        String idProperty = wcmMode == WCMMode.DISABLED ?
                TargetMediator.PN_PUBLISH_EXTERNAL_ID:
                TargetMediator.PN_AUTHOR_EXTERNAL_ID;

        ValueMap pageProps = offerPage.getProperties();
        String legacyExternalId = idProperty.replace("cq:", "");
        long offerId = pageProps.get(idProperty, 0L) != 0L
                ? pageProps.get(idProperty, 0L)
                : pageProps.get(legacyExternalId, 0L);

        return offerId;
    }
}
