/*
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2020 Adobe
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 */

package com.adobe.cq.dam.download.api;

import org.osgi.annotation.versioning.ProviderType;

import com.adobe.cq.dam.download.spi.DownloadTargetProcessor;

/**
 * <p>
 * Specifies an entity that should be included in a download initiated through
 * the {@link DownloadService}.
 * </p>
 * <p>
 * Depending on its <code>type</code>, a target will be routed to a
 * {@link DownloadTargetProcessor} implementation, which will use the target's
 * <code>parameters</code> to retrieve one or more binary URIs to include in the
 * download. The parameters are flexible and specific to the type of target.
 * </p>
 * <p>
 * For example, assume there is an target whose type is "asset". A parameter
 * applicable to this type might be a "path", whose value would be the sling
 * path to the asset. The target would go through a
 * {@link DownloadTargetProcessor} designed to handle assets, and the
 * implementation would include the asset's original rendition URI as the binary
 * to include in the download. Alternatively, the processor might return URIs
 * for <i>all</i> of the asset's renditions, which would be an example of a
 * single target generating multiple URIs.
 * </p>
 * 
 * @see DownloadService
 * @see DownloadManifest
 * @see DownloadTargetProcessor
 */
@ProviderType
public interface DownloadTarget {

    /**
     * Retrieves the target's type, which is primarily used to send the target through
     * its {@link DownloadTargetProcessor}.
     * @return The target's type.
     */
    String getType();

    /**
     * Retrieves one of the target's parameters by name. The value will be converted to
     * the specified type.
     * @param <T> Target type for the parameter value.
     * @param parameterName The name of the parameter whose value to retrieve.
     * @param clazz The type to which that the parameter's value will be converted.
     * @return The given parameter's value.
     */
    <T> T getParameter(String parameterName, Class<T> clazz);

    /**
     * Retrieves one of the target's parameters by name. If the parameter doesn't have
     * a value then the method will return the given default value instead. The parameter
     * value will be converted to the specified type.
     * @param <T> Target type for the parameter value.
     * @param parameterName The name of the parameter whose value to retrieve.
     * @param defaultValue Value to return if the parameter has no value.
     * @return The given parameter's value, or the given defalt value.
     */
    <T> T getParameter(String parameterName, T defaultValue);

    /**
     * Retrieves the names of all parameters included in the target.
     * @return Parameter names.
     */
    String[] getParameterNames();
}
