/*
 * File: OutlineConsumer2BaseImpl.java
 * 
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is
 *      obtained from Adobe Systems Incorporated.
 *
 */

package com.adobe.fontengine.font;

import com.adobe.fontengine.math.F26Dot6;

/** A base class for OutlineConsumer2 implementations.
 * 
 * {start,end}{Outline,Contour} do nothing.
 * 
 * The 26.6 versions of line, quadraticCurve and cubicCurve convert their
 * arguments to double, and call the corresponding double versions.
 * 
 * The double versions of line, quadraticCurve and cubicCurve convert their
 * arguments to 26.6, and call the corresponding 26.6 versions.
 * 
 * This class is not useful on its own, as this organization results in 
 * an infinite mutual recursion between the 26.6 and the double version
 * (hence the marking of this class as abstract). 
 * A descendant class <b>needs</b> to implement at least one of the two sets. 
 */
abstract public class OutlineConsumer2BaseImpl implements OutlineConsumer2 {

  public void startOutline () {
    // nothing by default
  }

  public void startContour () {
    // nothing by default
  }

  public void line (double x1, double y1, double x2, double y2) {
    line (F26Dot6.fromDouble (x1), F26Dot6.fromDouble (y1),
          F26Dot6.fromDouble (x2), F26Dot6.fromDouble (y2));
  }

  public void line (int /*26.6*/ x1, int /*26.6*/ y1, 
                    int /*26.6*/ x2, int /*26.6*/ y2) {
    line (F26Dot6.toDouble (x1), F26Dot6.toDouble (y1), 
          F26Dot6.toDouble (x2), F26Dot6.toDouble (y2));
  }

  public void quadraticCurve (double x1, double y1,
                              double x2, double y2,
                              double x3, double y3) {
    quadraticCurve (F26Dot6.fromDouble (x1), F26Dot6.fromDouble (y1),
                    F26Dot6.fromDouble (x2), F26Dot6.fromDouble (y2),
                    F26Dot6.fromDouble (x3), F26Dot6.fromDouble (y3));
  }

  public void quadraticCurve (int /*26.6*/ x1, int /*26.6*/ y1,
                              int /*26.6*/ x2, int /*26.6*/ y2,
                              int /*26.6*/ x3, int /*26.6*/ y3) {
    quadraticCurve (F26Dot6.toDouble (x1), F26Dot6.toDouble (y1), 
                    F26Dot6.toDouble (x2), F26Dot6.toDouble (y2), 
                    F26Dot6.toDouble (x3), F26Dot6.toDouble (y3));
  }

  public void cubicCurve (double x1, double y1,
                          double x2, double y2,
                          double x3, double y3,
                          double x4, double y4) {
    cubicCurve (F26Dot6.fromDouble (x1), F26Dot6.fromDouble (y1),
                F26Dot6.fromDouble (x2), F26Dot6.fromDouble (y2),
                F26Dot6.fromDouble (x3), F26Dot6.fromDouble (y3),
                F26Dot6.fromDouble (x4), F26Dot6.fromDouble (y4));
  }

  public void cubicCurve (int /*26.6*/ x1, int /*26.6*/ y1, 
                          int /*26.6*/ x2, int /*26.6*/ y2, 
                          int /*26.6*/ x3, int /*26.6*/ y3, 
                          int /*26.6*/ x4, int /*26.6*/ y4) {
    cubicCurve (F26Dot6.toDouble (x1), F26Dot6.toDouble (y1), 
                F26Dot6.toDouble (x2), F26Dot6.toDouble (y2), 
                F26Dot6.toDouble (x3), F26Dot6.toDouble (y3), 
                F26Dot6.toDouble (x4), F26Dot6.toDouble (y4));
  }

  public void endContour () {
    // nothing by default
  }

  public void endOutline () {
    // nothing by default
  }
}
