/*
 * File: TTCompositeOutline.java
 * 
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is
 *      obtained from Adobe Systems Incorporated.
 *
 */

/*
 * Based on code with the following copyright notices:
 * 
 *  Copyright (c) 1987-1990, 1992 by Apple Computer, Inc., all rights reserved.
 *  Copyright (c) 1989-2002. Microsoft Corporation, all rights reserved.
 */

package com.adobe.fontengine.font.opentype;

import java.util.ArrayList;

import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.InvalidGlyphException;
import com.adobe.fontengine.font.Matrix;
import com.adobe.fontengine.font.OutlineConsumer;
import com.adobe.fontengine.font.OutlineConsumer2;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.math.F26Dot6;
import com.adobe.fontengine.math.F26Dot6Vector;

/** A TrueType composite outline.
 */

public final class TTCompositeOutline extends TTOutline {
  
  public static final class TTComponent {
    public TTOutline outline;
    public int argument1;
    public int argument2;
    
    public boolean alignByPosition;
    public boolean roundXYToGrid;
    public boolean useThisMetrics;
    public boolean scaledOffsets;
    public boolean unscaledOffset; 
    
    public F26Dot6Vector offset;
    
    public int /*2.16*/ a, b, c, d;
    
    public void scale (int unitsPerEm, Matrix em2px) {
      if (alignByPosition) {
        int /*26.6*/ x = F26Dot6.fromDouble (em2px.applyToXYGetX (argument1 / (double) unitsPerEm, 0));
        int /*26.6*/ y = F26Dot6.fromDouble (em2px.applyToXYGetY (0, argument2 / (double) unitsPerEm));
        if (roundXYToGrid) {
          x = F26Dot6.roundHalfUp (x);
          y = F26Dot6.roundHalfUp (y); }
        offset = new F26Dot6Vector (x, y); }        

      outline.scale (unitsPerEm, em2px);
    }
    
    public void instruct (TTInterpreter interpreter)
    throws InvalidGlyphException, UnsupportedFontException {
      outline.instruct (interpreter);
    }
  }
  
  private TTComponent[] components;
  public TTSimpleOutline merged;
  private OTByteArray instructions;
  private int instructionsOffset;
  private int instructionsLength;

  public TTCompositeOutline (ArrayList comps, int l, int r, int t, int b,
      OTByteArray instructions, int instructionsOffset, int instructionsLength)
  throws InvalidFontException {
    this.components = (TTComponent []) comps.toArray (new TTComponent [] {});
    
    this.instructions = instructions;
    this.instructionsOffset = instructionsOffset;
    this.instructionsLength = instructionsLength;
    
    merged = new TTSimpleOutline ();
    merged.unscaledCoordinatesAreInvalid = true;
    
    int numCountours = 0;
    int numPoints = 0;
    for (int c = 0; c < components.length; c++) {
      numCountours += components [c].outline.getNumContours ();
      numPoints += components [c].outline.getNumOutlinePoints (); }
    
    merged.contourEndPoints = new int [numCountours];
    merged.points = new TTPoint [numPoints + 4];
    
    
    merged.points [numPoints+0] = new TTPoint (l, 0, false);
    merged.points [numPoints+1] = new TTPoint (r, 0, false);
    merged.points [numPoints+2] = new TTPoint (0, t, false);
    merged.points [numPoints+3] = new TTPoint (0, b, false);
  }

  //----
  TTPoint getPoint (int index) throws InvalidGlyphException {
    return merged.getPoint (index);
  }
  
  int getNumOutlinePoints () throws InvalidGlyphException {
    return merged.getNumOutlinePoints ();
  }
  
  int getNumContours () {
    return merged.getNumContours (); 
  }
  
  int getContourNextPoint (int contour, int index)  throws InvalidGlyphException {
    return merged.getContourNextPoint (contour, index);
  }

  int getContourFirstPoint (int contour) throws InvalidGlyphException {
    return merged.getContourFirstPoint (contour);
  }
  
  int getContourLastPoint (int contour) throws InvalidGlyphException {
    return merged.getContourLastPoint(contour);
  }
  
  public TTSimpleOutline getMerged () {
    mergeComponents ();
    return merged;
  }
  
  //----------------
  private boolean isScaled = false;
  private boolean isInstructed = false;
  private int currentUnitsPerEm = -1;
  private Matrix currentEm2px = null;
  
  public void scale (int unitsPerEm, Matrix em2px) { 
    if (isScaled && currentUnitsPerEm == unitsPerEm && em2px.equals (currentEm2px)) {
      return; }
    
    for (int c = 0; c < components.length; c++) {
      components [c].scale (unitsPerEm, em2px); }

    int nbOutlinePoints = merged.points.length - 4;
    
    for (int i = nbOutlinePoints; i < nbOutlinePoints + 4; i++) {
      merged.points [i].scale (unitsPerEm, em2px); }
    
    // move the right side bearing to make the rsb-lsb an integer number
    // of pixels
    
    int /*f26.6*/ width 
    = F26Dot6.roundHalfUp (F26Dot6.fromDouble (em2px.applyToXYGetX ((merged.points [nbOutlinePoints + 1].unscaled.x - merged.points [nbOutlinePoints].unscaled.x) / (double) unitsPerEm, 0)));
    merged.points [nbOutlinePoints + 1].hinted.x = merged.points [nbOutlinePoints].hinted.x + width;
    
    merged.points [nbOutlinePoints + 2].hinted.y = F26Dot6.round (merged.points [nbOutlinePoints + 2].hinted.y);
    
    int /*f26.6*/ height 
    = F26Dot6.roundHalfUp (F26Dot6.fromDouble (em2px.applyToXYGetY (0, (merged.points[nbOutlinePoints + 3].unscaled.y - merged.points [nbOutlinePoints + 2].unscaled.y) / (double) unitsPerEm)));
    merged.points [nbOutlinePoints + 3].hinted.y = merged.points [nbOutlinePoints + 2].hinted.y + height;

    isScaled = true;
    currentUnitsPerEm = unitsPerEm;
    currentEm2px = em2px;
    isInstructed = false;
  }
  
  private int scanType;
  public int getScanType () {
    return scanType;
  }
  
  public void instruct (TTInterpreter interpreter) throws InvalidGlyphException, UnsupportedFontException {
    if (isInstructed) {
      return; }
    
    for (int c = 0; c < components.length; c++) {
      components [c].instruct (interpreter);  }
    
    isMerged = false;
    mergeComponents ();
    
    int roundedPosition = F26Dot6.roundHalfUp(merged.points[merged.points.length - 4].unhinted.x);    
    shiftPoints(roundedPosition - merged.points[merged.points.length - 4].unhinted.x, 0, 0, merged.points.length); 

    if (instructions != null) {
      interpreter.runGlyf (merged, instructions, instructionsOffset, instructionsOffset + instructionsLength);
      scanType = interpreter.getScanType (); }
    else {
      scanType = 0; }
    
    isInstructed = true;
  }
  
  boolean isMerged = false;
  
  protected void mergeComponents () {
    if (isMerged) {
      return; }
    
    int k = 0;
    int p = 0;
    for (int c = 0; c < components.length; c++) {

      TTSimpleOutline src = components [c].outline.getMerged ();
      for (int i = 0; i < src.contourEndPoints.length; i++) {
        merged.contourEndPoints [k] = src.contourEndPoints [i] + p;
        k++; }
      for (int i = 0; i < src.points.length - 4; i++) {
        merged.points [p] = (TTPoint) (src.points [i].clone ()); 
        
        merged.points [p].hinted.x = 
          F26Dot6.multiplyByF2Dot14 (src.points [i].hinted.x, components [c].a)
        + F26Dot6.multiplyByF2Dot14 (src.points [i].hinted.y, components [c].b);
        merged.points [p].hinted.y = 
          F26Dot6.multiplyByF2Dot14 (src.points [i].hinted.x, components [c].c)
        + F26Dot6.multiplyByF2Dot14 (src.points [i].hinted.y, components [c].d);

        merged.points [p].hinted.x += components [c].offset.x;
        merged.points [p].hinted.y += components [c].offset.y;
                
//        merged.points [p].orig.x = 
//          F26Dot6.multiplyByF2Dot14 (src.points [i].orig.x, components [c].a)
//        + F26Dot6.multiplyByF2Dot14 (src.points [i].orig.y, components [c].b);
//        merged.points [p].orig.y = 
//          F26Dot6.multiplyByF2Dot14 (src.points [i].orig.x, components [c].c)
//        + F26Dot6.multiplyByF2Dot14 (src.points [i].orig.y, components [c].d);

        merged.points [p].unhinted.x = merged.points [p].hinted.x;
        merged.points [p].unhinted.y = merged.points [p].hinted.y;
        
        merged.points [p].original.x = 
          F26Dot6.multiplyByF2Dot14 (src.points [i].original.x, components [c].a)
        + F26Dot6.multiplyByF2Dot14 (src.points [i].original.y, components [c].b);
        merged.points [p].original.y = 
          F26Dot6.multiplyByF2Dot14 (src.points [i].original.x, components [c].c)
        + F26Dot6.multiplyByF2Dot14 (src.points [i].original.y, components [c].d);

        merged.points [p].original.x += components [c].offset.x;
        merged.points [p].original.y += components [c].offset.y;
 
        p++; }}
    
    isMerged = true;
  }
  
  public void toConsumer (OutlineConsumer c) throws InvalidFontException {
    mergeComponents (); 
    merged.toConsumer (c);
  }

  public void toConsumer2 (OutlineConsumer2 c) throws InvalidFontException {
    mergeComponents ();
    merged.toConsumer2 (c);
  }

	public void translate() {
		//	 Move the outline by the difference between the lsb in the hmtx and the lsb in the glyph rounded to the nearest pixel boundary.
	    int roundedX = F26Dot6.roundHalfUp(merged.points[merged.points.length - 4].hinted.x);   
	    int roundedY = F26Dot6.roundHalfUp(merged.points[merged.points.length - 4].hinted.y); 
	    shiftPoints(-roundedX, -roundedY, 0, merged.points.length); 
	}
	
	private void shiftPoints(int dx, int dy, int startingPoint, int numPoints)
	  {
		  for (int i = startingPoint; i < numPoints; i++)
		  {
			  merged.points[i].hinted.x += dx;
			  merged.points[i].hinted.y += dy;
		  }
	  }
}
