/*
 * File: TTSimpleOutline.java
 * 
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is
 *      obtained from Adobe Systems Incorporated.
 *
 */

/*
 * Based on code with the following copyright notices:
 * 
 *  Copyright (c) 1987-1990, 1992 by Apple Computer, Inc., all rights reserved.
 *  Copyright (c) 1989-2002. Microsoft Corporation, all rights reserved.
 */

package com.adobe.fontengine.font.opentype;

import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.InvalidGlyphException;
import com.adobe.fontengine.font.Matrix;
import com.adobe.fontengine.font.OutlineConsumer;
import com.adobe.fontengine.font.OutlineConsumer2;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.math.F26Dot6;

/** A TrueType simple outline.
 */

public final class TTSimpleOutline extends TTOutline {
  public TTPoint [] points;
  public int[] contourEndPoints;
  
  public OTByteArray instructions;
  public int instructionsOffset;
  public int instructionsLength;
   
  //----------------------------------------------------------------------------
  
  private boolean isScaled = false;
  private boolean isInstructed = false;
  private int currentUnitsPerEm = -1;
  private Matrix currentEm2px = null;
  
  public void scale (int unitsPerEm, Matrix em2px) {
    if (isScaled && currentUnitsPerEm == unitsPerEm && em2px.equals (currentEm2px)) {
      return; }
    
    for (int p = 0; p < points.length; p++) {
      points [p].scale (unitsPerEm, em2px); }
    
    // move the right side bearing to make the rsb-lsb an integer number
    // of pixels

    int nbOutlinePoints = points.length - 4;
    
    int /*f26.6*/ width 
      = F26Dot6.roundHalfUp (F26Dot6.fromDouble (em2px.applyToXYGetX ((points [nbOutlinePoints + 1].unscaled.x - points [nbOutlinePoints].unscaled.x) / (double) unitsPerEm, 0)));
    points [nbOutlinePoints + 1].hinted.x = points [nbOutlinePoints].hinted.x + width;
    
    points [nbOutlinePoints + 2].hinted.y = F26Dot6.round (points [nbOutlinePoints + 2].hinted.y);
    
    int /*f26.6*/ height 
      = F26Dot6.roundHalfUp (F26Dot6.fromDouble (em2px.applyToXYGetY (0, (points[nbOutlinePoints + 3].unscaled.y - points [nbOutlinePoints + 2].unscaled.y) / (double) unitsPerEm)));
    points [nbOutlinePoints + 3].hinted.y = points [nbOutlinePoints + 2].hinted.y + height;
    
    isScaled = true;
    currentUnitsPerEm = unitsPerEm;
    currentEm2px = em2px;
    isInstructed = false;
   }
  
  private void shiftPoints(int dx, int dy, int startingPoint, int numPoints)
  {
	  for (int i = startingPoint; i < numPoints; i++)
	  {
		  points[i].hinted.x += dx;
		  points[i].hinted.y += dy;
	  }
  }
  
  private int scanType;
  public int getScanType () {
    return scanType;
  }
  
  public void instruct (TTInterpreter interpreter)
  throws InvalidGlyphException, UnsupportedFontException {
    if (isInstructed) {
      return; }
    
    // The TrueType spec suggests that the first phantom point (which is the difference between the hmtx and the
    // glyf lsbs) be 0 prior to hinting, but it doesn't require it. 
    // If it isn't 0, CoolType moves the whole outline by the subpixel portion of the outline
    // prior to hinting. (see scl_AdjustOldCharSideBearing)
    int roundedPosition = F26Dot6.roundHalfUp(points[points.length - 4].unhinted.x);    
    shiftPoints(roundedPosition - points[points.length - 4].unhinted.x, 0, 0, points.length);
    
    interpreter.runGlyf (this, instructions, instructionsOffset, instructionsOffset + instructionsLength);
    scanType = interpreter.getScanType ();
    
    isInstructed = true;
  }

  //----------------------------------------------------------------------------
   
  
  TTPoint getPoint (int index) throws InvalidGlyphException {
    if (index < 0 || points.length <= index) {
      throw new InvalidGlyphException ("attempt to access invalid point " + index + " in zone 0"); }
    return points [index];
  }
  
  int getNumOutlinePoints () {
    return points.length - 4; }
  
  int getNumContours () {
    return contourEndPoints.length;
  }
  
  int getContourNextPoint (int contour, int index)  throws InvalidGlyphException {
    index++;
    if (index > contourEndPoints [contour]) {
      index = getContourFirstPoint (contour); }
    return index;
  }

  int getContourFirstPoint (int contour) throws InvalidGlyphException {
    if (contour < 0 || contourEndPoints.length <= contour) {
      throw new InvalidGlyphException ("attempt to access invalid contour " + contour); }
    if (contour == 0) {
      return 0; }
    else {
      return contourEndPoints [contour - 1] + 1; }
  }
  
  int getContourLastPoint (int contour) throws InvalidGlyphException {
    if (contour < 0 || contourEndPoints.length <= contour) {
      throw new InvalidGlyphException ("attempt to access invalid contour " + contour); }
    
    return contourEndPoints [contour];
  }
  
  public TTSimpleOutline getMerged () {
    return this;
  }

  //----------------------------------------------------------------------------
  
  public void toConsumer (OutlineConsumer c) throws InvalidFontException {
    for (int contour = 0; contour < contourEndPoints.length; contour++) {
      int current = getContourFirstPoint (contour);
      int last = getContourLastPoint (contour);
      
      TTPoint currentPoint = getPoint (current);
      double currentX = F26Dot6.toDouble (currentPoint.hinted.x);
      double currentY = F26Dot6.toDouble (currentPoint.hinted.y);

      double startX, startY;
      
      if (currentPoint.onCurve) {
        startX = F26Dot6.toDouble (currentPoint.hinted.x);
        startY = F26Dot6.toDouble (currentPoint.hinted.y); }
      else {
        TTPoint previousPoint = getPoint (last);
        double previousX = F26Dot6.toDouble (previousPoint.hinted.x);
        double previousY = F26Dot6.toDouble (previousPoint.hinted.y);
        if (previousPoint.onCurve) {
          startX = previousX;
          startY = previousY; }
        else {
          startX = (previousX + currentX) / 2;
          startY = (previousY + currentY) / 2; }}
      c.moveto (startX, startY);
     
      while (current < last) {
        TTPoint previousPoint = currentPoint;
        double previousX = currentX;
        double previousY = currentY;

        current++;
        currentPoint = getPoint (current);
        currentX = F26Dot6.toDouble (currentPoint.hinted.x);
        currentY = F26Dot6.toDouble (currentPoint.hinted.y);
       
        if (currentPoint.onCurve) {
          if (previousPoint.onCurve) {
            c.lineto (currentX, currentY); }
          else {
            c.curveto (previousX, previousY, currentX,  currentY); }}
        else {
          if (previousPoint.onCurve) {
             /* nothing */ }
          else {
            c.curveto (previousX, previousY,
                (previousX + currentX) / 2.0d,
                (previousY + currentY) / 2.0d); }}}
      
      if (currentPoint.onCurve) {
        if (currentX != startX || currentY != startY) {
          c.lineto (startX, startY); }}
      else {
        c.curveto (currentX, currentY, startX, startY); }}   
    
    c.endchar();
  }

  public void toConsumer2 (OutlineConsumer2 c) throws InvalidFontException {
    c.startOutline ();
    
    for (int contour = 0; contour < contourEndPoints.length; contour++) {
      c.startContour ();

      int current = getContourFirstPoint (contour);
      int last = getContourLastPoint (contour);
      
      TTPoint currentPoint = getPoint (current);

      int /*26.6*/ startX, startY;
      
      if (currentPoint.onCurve) {
        startX = currentPoint.hinted.x;
        startY = currentPoint.hinted.y; }
      else {
        TTPoint previousPoint = getPoint (last);
        if (previousPoint.onCurve) {
          startX = previousPoint.hinted.x;
          startY = previousPoint.hinted.y; }
        else {
          startX = (previousPoint.hinted.x + currentPoint.hinted.x + 1) / 2;
          startY = (previousPoint.hinted.y + currentPoint.hinted.y + 1) / 2; }}

      int /*26.6*/ currentX = startX;
      int /*26.6*/ currentY = startY;
     
      while (current < last) {
        TTPoint previousPoint = currentPoint;
        current++;
        currentPoint = getPoint (current);
       
        if (currentPoint.onCurve) {
          if (previousPoint.onCurve) {
            c.line (currentX, currentY, 
                    currentPoint.hinted.x, currentPoint.hinted.y);
            currentX = currentPoint.hinted.x;
            currentY = currentPoint.hinted.y; }
          else {
            c.quadraticCurve (currentX, currentY, 
                              previousPoint.hinted.x, previousPoint.hinted.y, 
                              currentPoint.hinted.x,  currentPoint.hinted.y);
            currentX = currentPoint.hinted.x;
            currentY = currentPoint.hinted.y; }}
        else {
          if (previousPoint.onCurve) {
             /* nothing */ }
          else {
            c.quadraticCurve (currentX, currentY, 
                              previousPoint.hinted.x, previousPoint.hinted.y,
                              (previousPoint.hinted.x + currentPoint.hinted.x + 1) / 2,
                              (previousPoint.hinted.y + currentPoint.hinted.y + 1) / 2); 
            currentX = (previousPoint.hinted.x + currentPoint.hinted.x + 1) / 2;
            currentY = (previousPoint.hinted.y + currentPoint.hinted.y + 1) / 2; }}}
      
      if (currentPoint.onCurve) {
        if (currentPoint.hinted.x != startX || currentPoint.hinted.y != startY) {
          c.line (currentX, currentY,
                  startX, startY); }}
      else {
        c.quadraticCurve (currentX, currentY, 
                          currentPoint.hinted.x, currentPoint.hinted.y, 
                          startX, startY); }
      
      c.endContour (); }  
    
    c.endOutline ();
  }

	public void translate() {
		// Hinting may have moved the delta LSB from 0 (or it may have never been 0). The delta needs to be
		// taken into account some time...this is that time.
	    int roundedX = F26Dot6.roundHalfUp(points[points.length - 4].hinted.x);   
	    int roundedY = F26Dot6.roundHalfUp(points[points.length - 4].hinted.y); 
	    shiftPoints(-roundedX, -roundedY, 0, points.length); 
	}
}