/*
 * File: DigitCase.java
 * 
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is
 *      obtained from Adobe Systems Incorporated.
 *
 */

package com.adobe.fontengine.inlineformatting;

/** Enumerated values for {@link ElementAttribute#digitCase}.
 * 
 * Some fonts provide multiple digits glyphs: lining digits which work well with 
 * uppercase letters, and old style digits which work well with lowercase
 * letters.
 * 
 * This class provides constants corresponding to this distinction.
 * 
 */

final public class DigitCase {
  
  private final String name;
  
  private DigitCase (String name) {
    this.name = name;
  }
  
  public String toString () {
    return name;
  }
  
  /** Lining digits.
   */
  public static final DigitCase LINING = new DigitCase ("LINING");
  
  /** Old style digits.
   */
  public static final DigitCase OLD_STYLE = new DigitCase ("OLD_STYLE");
  
  /** Fonts default.
   * 
   * This value can be used to request the default case setting, as 
   * determined by the font designer.
   */
  public static final DigitCase DEFAULT = new DigitCase ("DEFAULT");

  /** Derive the value from the {@link ElementAttribute#typographicCase} attribute.
   * 
   * The value is derived as follows:
   * <ul>
   * <li>{@link #LINING}, for {@link TypographicCase#TITLE} and {@link TypographicCase#CAPS},
   * <li>{@link #DEFAULT}, for {@link TypographicCase#NONE}
   * <li>{@link #OLD_STYLE}, for the others values
   * </ul>
   */
  public static final DigitCase FROM_TYPOGRAPHIC_CASE = new DigitCase ("FROM_TYPOGRAPHIC_CASE");
  
  
  private static final DigitCase[] allValues 
  = {LINING, OLD_STYLE, FROM_TYPOGRAPHIC_CASE, DEFAULT};
  
  public static DigitCase parse (String s) {
    for (int i = 0; i < allValues.length; i++) {
      if (allValues [i].name.compareToIgnoreCase (s) == 0) {
        return allValues [i]; }}
    return null;
  }
}