/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.crypto;

import java.security.KeyPair;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.Certificate;

import javax.security.auth.x500.X500Principal;

import org.osgi.annotation.versioning.ProviderType;

/**
 * The <code>CryptoSupport</code> provides a simple API to encrypt and decrypt
 * binary and string data.
 * <p>
 * This interface is not intended to be implemented by consumers. To use the API
 * get the service from the service registry under the name {@value #NAME}.
 */
@ProviderType
public interface CryptoSupport {

    /**
     * Name of the Encryption/Decryption service which may be used securily
     * store sensitive data.
     */
    String NAME = "com.adobe.granite.crypto.CryptoSupport";

    /**
     * Encrypts the given <code>plainText</code> data into a cipher text.
     * <p>
     * Note that this method and the {@link #decrypt(byte[])} method provide
     * full round trip support:
     *
     * <pre>
     * decrypt(encrypt(plainText)).equals(plainText) == true
     * </pre>
     * <p>
     * Please note, that calling this method twice on the same
     * <code>plainText</code> does <em>not</em> return the same cipher text:
     *
     * <pre>
     * encrypt(plainText).equals(encrypt(plainText)) == false
     * </pre>
     *
     * @param plainText The plain text data to encrypt
     * @return The encrypted data
     * @throws CryptoException If any problem occurrs encrypting the plain text
     *             data. The {@link CryptoException#getCause()} method may
     *             provide additional information on the encryption failure.
     */
    byte[] encrypt(byte[] plainText) throws CryptoException;

    /**
     * Decrypts the given <code>cipherText</code> data into plain text.
     * <p>
     * Note that this method and the {@link #encrypt(byte[])} method provide
     * full round trip support:
     *
     * <pre>
     * decrypt(encrypt(plainText)).equals(plainText) == true
     * </pre>
     *
     * @param cipherText The encrypted data to decrypt
     * @return The plain text data
     * @throws CryptoException If any problem occurrs decrypting the cipher
     *             text. The {@link CryptoException#getCause()} method may
     *             provide additional information on the decryption failure.
     */
    byte[] decrypt(byte[] cipherText) throws CryptoException;
    
    /**
     * Encrypts the given <code>plainText</code> data into a cipher text.
     * <p>
     * Note that this method and the {@link #decrypt(byte [], byte[])} method provide
     * full round trip support:
     *
     * <pre>
     * decrypt(encrypt(key,plainText)).equals(key,plainText) == true
     * </pre>
     * <p>
     * Please note that the implementation will not clear the byte[] key.
     * <p>
     * Please note, that calling this method twice on the same
     * <code>plainText</code> does <em>not</em> return the same cipher text:
     *
     * <pre>
     * encrypt(key, plainText).equals(encrypt(key, plainText)) == false
     * </pre>
     * @param key The bytes used to seed the algorithm. This must be a non-
     *            <code>null</code>, non empty array of bytes. If the array is
     *            larger than 512 elements, only the first 512 entries are used.           
     * @param plainText The plain text data to encrypt
     * @return The encrypted data
     * @throws CryptoException If any problem occurrs encrypting the plain text
     *             data. The {@link CryptoException#getCause()} method may
     *             provide additional information on the encryption failure.
     *             
     * @since 1.2
     */
    byte[] encrypt(byte[] key, byte[] plainText) throws CryptoException;
    
    /**
     * Decrypts the given <code>cipherText</code> data into plain text.
     * <p>
     * Please note that the implementation will not clear the byte[] key.
     * <p>
     * Note that this method and the {@link #encrypt(byte [], byte[])} method provide
     * full round trip support:
     *
     * <pre>
     * decrypt(encrypt(key, plainText)).equals(key, plainText) == true
     * </pre>
     *
     * @param key The bytes used to seed the algorithm. This must be a non-
     *            <code>null</code>, non empty array of bytes. If the array is
     *            larger than 512 elements, only the first 512 entries are used.
     * @param cipherText The encrypted data to decrypt
     * @return The plain text data
     * @throws CryptoException If any problem occurrs decrypting the cipher
     *             text. The {@link CryptoException#getCause()} method may
     *             provide additional information on the decryption failure.
     *             
     * @since 1.2
     */
    byte[] decrypt(byte[] key, byte[] cipherText) throws CryptoException;
    

    /**
     * Returns <code>true</code> if the given string is to be considered
     * protected by the {@link #protect(String)} method and can be converted to
     * plain text by calling the {@link #unprotect(String)} method.
     * 
     * @param text the string to test for protection
     * @return {@code true} if the given string is to be considered protected by
     * the {@link #protect(String)} method and can be converted to plain text by
     * calling the {@link #unprotect(String)} method
     */
    boolean isProtected(String text);

    /**
     * Encrypts the given <code>plainText</code> data into a cipher text.
     * <p>
     * This method is like {@link #encrypt(byte[])} but for character data.
     * <p>
     * Note that this method and the {@link #unprotect(String)} method provide
     * full round trip support:
     *
     * <pre>
     * unprotect(protect(plainText)).equals(plainText) == true
     * </pre>
     * <p>
     * Please note, that calling this method twice on the same
     * <code>plainText</code> does <em>not</em> return the same cipher text:
     *
     * <pre>
     * protect(plainText).equals(protect(plainText)) == false
     * </pre>
     *
     * @param plainText The plain text data to encrypt
     * @return The encrypted data
     * @throws CryptoException If any problem occurrs encrypting the plain text
     *             data. The {@link CryptoException#getCause()} method may
     *             provide additional information on the encryption failure.
     */
    String protect(String plainText) throws CryptoException;

    /**
     * Unprotects the given string such that the resulting plain text string if
     * given to the {@link #protect(String)} returns the protected string given
     * to this method.
     * <p>
     * Note that this method and the {@link #protect(String)} method provide
     * full round trip support:
     *
     * <pre>
     * unprotect(protect(plainText)).equals(plainText) == true
     * </pre>
     *
     * @param cipherText The encrypted data to decrypt
     * @return The plain text data
     * @throws CryptoException If any problem occurrs decrypting the cipher
     *             text. The {@link CryptoException#getCause()} method may
     *             provide additional information on the decryption failure.
     *             Particularly this exception may be thrown if the
     *             <code>cipherText</code> has obviously not been protected by
     *             the {@link #protect(String)} method and
     *             {@link #isProtected(String)} would return <code>false</code>.
     */
    String unprotect(String cipherText) throws CryptoException;
    
    /**
     * Encrypts the given <code>plainText</code> data into a cipher text.
     * <p>
     * This method is like {@link #encrypt(byte[], byte[])} but for character data.
     * <p>
     * Please note that the implementation will not clear the byte[] key.
     * <p>
     * Note that this method and the {@link #unprotect(byte[], String)} method provide
     * full round trip support:
     *
     * <pre>
     * unprotect(protect(key, plainText)).equals(key, plainText) == true
     * </pre>
     * <p>
     * Please note, that calling this method twice on the same
     * <code>plainText</code> does <em>not</em> return the same cipher text:
     *
     * <pre>
     * protect(key, plainText).equals(protect(key, plainText)) == false
     * </pre>
     *
     * @param key The bytes used to seed the algorithm. This must be a non-
     *            <code>null</code>, non empty array of bytes. If the array is
     *            larger than 512 elements, only the first 512 entries are used.
     * @param plainText The plain text data to encrypt
     * @return The encrypted data
     * @throws CryptoException If any problem occurrs encrypting the plain text
     *             data. The {@link CryptoException#getCause()} method may
     *             provide additional information on the encryption failure.
     * @since 1.2
     */
    String protect(byte[] key, String plainText) throws CryptoException;

    /**
     * Unprotects the given string such that the resulting plain text string if
     * given to the {@link #protect(byte[], String)} returns the protected string given
     * to this method.
     * <p>
     * Please note that the implementation will not clear the byte[] key.
     * <p>
     * Note that this method and the {@link #protect(byte[], String)} method provide
     * full round trip support:
     *
     * <pre>
     * unprotect(protect(key, plainText)).equals(key, plainText) == true
     * </pre>
     * @param key The bytes used to seed the algorithm. This must be a non-
     *            <code>null</code>, non empty array of bytes. If the array is
     *            larger than 512 elements, only the first 512 entries are used.
     * @param cipherText The encrypted data to decrypt
     * @return The plain text data
     * @throws CryptoException If any problem occurrs decrypting the cipher
     *             text. The {@link CryptoException#getCause()} method may
     *             provide additional information on the decryption failure.
     *             Particularly this exception may be thrown if the
     *             <code>cipherText</code> has obviously not been protected by
     *             the {@link #protect(String)} method and
     *             {@link #isProtected(String)} would return <code>false</code>.
     *             
     * @since 1.2
     */
    String unprotect(byte[] key, String cipherText) throws CryptoException;

    /**
     * Wraps the given <code>keyData</code> using a symmetric key wrap algorithm.
     * <p>
     * Note that this method and the {@link #unwrapKey(byte[], byte[])} method provide
     * full round trip support:
     *
     * <pre>
     * unwrapKey(wrapKey(kek,keyData)).equals(kek,keyData) == true
     * </pre>
     * <p>
     * Please note that the implementation will not clear the byte[] key.
     * <p>
     * Please note, that unlike for encryption methods, calling this method
     * twice with the same <code>keyData</code> <em>may</em> return the same cipher text.
     *
     * @param kek the key-encryption key used to seed the key wrap algorithm.
     *            This must be a non-<code>null</code>, non empty array of bytes.
     *            Refer to the implementation for supported algorithm and key lengths.
     * @param keyData The key data to be wrapped.
     *                This must ve a non-<code>null</code>, non empty array of bytes.
     *                Refer to the implementation for limitations regarding the size.
     * @return The wrapped key data
     * @throws CryptoException If any problem occurs wrapping the key data.
     *         The {@link CryptoException#getCause()} method may
     *         provide additional information on the wrapping failure.
     *
     * @since 1.5
     */
    byte[] wrapKey(byte[] kek, byte[] keyData) throws CryptoException;

    /**
     * Wraps the given <code>keyData</code> using a symmetric key wrap algorithm.
     * <p>
     * Note that the kek the key-encryption key used to seed the key wrap algorithm is selected by the implementation.
     * <p>
     * Note that this method and the {@link #unwrapKey(byte[])} method provide
     * full round trip support:
     *
     * <pre>
     * unwrapKey(wrapKey(keyData)).equals(keyData) == true
     * </pre>
     * <p>
     * Please note that the implementation will not clear the byte[] key.
     * <p>
     * Please note, that unlike for encryption methods, calling this method
     * twice with the same <code>keyData</code> <em>may</em> return the same cipher text.
     *
     * @param keyData The key data to be wrapped.
     *                This must ve a non-<code>null</code>, non empty array of bytes.
     *                Refer to the implementation for limitations regarding the size.
     * @return The wrapped key data
     * @throws CryptoException If any problem occurs wrapping the key data.
     *         The {@link CryptoException#getCause()} method may
     *         provide additional information on the wrapping failure.
     *
     * @since 1.5
     */
    byte[] wrapKey(byte[] keyData) throws CryptoException;

    /**
     * Unwraps the given <code>wrappedKey</code> using a symmetric key wrap algorithm.
     * <p>
     * Note that this method and the {@link #wrapKey(byte[], byte[])} method provide
     * full round trip support:
     *
     * <pre>
     * unwrapKey(wrapKey(kek,keyData)).equals(kek,keyData) == true
     * </pre>
     * <p>
     * Please note that the implementation will not clear the byte[] key.
     *
     * @param kek the key-encryption key used to seed the key wrap algorithm.
     *            This must be a non-<code>null</code>, non empty array of bytes.
     *            Refer to the implementation for supported algorithm and key lengths.
     * @param wrappedKeyData The key data to be wrapped.
     *            This must ve a non-<code>null</code>, non empty array of bytes.
     *            Refer to the implementation for limitations regarding the size.
     * @return The wrapped key data
     * @throws CryptoException If any problem occurs wrapping the key data.
     *         The {@link CryptoException#getCause()} method may
     *         provide additional information on the wrapping failure.
     *
     * @since 1.5
     */
    byte[] unwrapKey(byte[] kek, byte[] wrappedKeyData) throws CryptoException;

    /**
     * Unwraps the given <code>wrappedKey</code> using a symmetric key wrap algorithm.
     * <p>
     * Note that the kek the key-encryption key used to seed the key wrap algorithm is selected by the implementation.
     * <p>
     * Note that this method and the {@link #wrapKey(byte[])} method provide
     * full round trip support:
     *
     * <pre>
     * unwrapKey(wrapKey(keyData)).equals(keyData) == true
     * </pre>
     * <p>
     * Please note that the implementation will not clear the byte[] key.
     *
     * @param wrappedKeyData The key data to be wrapped.
     *            This must ve a non-<code>null</code>, non empty array of bytes.
     *            Refer to the implementation for limitations regarding the size.
     * @return The wrapped key data
     * @throws CryptoException If any problem occurs wrapping the key data.
     *         The {@link CryptoException#getCause()} method may
     *         provide additional information on the wrapping failure.
     *
     * @since 1.5
     */
    byte[] unwrapKey(byte[] wrappedKeyData) throws CryptoException;
    
    /**
     * Fill the byte buffer with securely-generated pseudo-random bytes.
     *
     * @param bytes Buffer to fill with random bytes.
     * @throws CryptoException If any problem occurrs calculating the random
     *             data. The {@link CryptoException#getCause()} method may
     *             provide additional information on the failure.
     * @since 1.1, Crypto Support 0.4
     */
    public void nextRandomBytes(byte[] bytes) throws CryptoException;

    /**
     * Generate HMAC bytes given a key and some text. In other, perhaps less
     * cryptographically correct words, generates and returns a hash of 'text'
     * encrypted by 'keyBytes'.
     * <p>
     * The implementation is expected to implement the keyed hashing function
     * using SHA-256 as the hash algorithm. See <a
     * href="http://www.ietf.org/rfc/rfc2104.txt">RFC 2104</a> for the HMAC
     * specification.
     * <p>
     * Please note that the implementation will not clear the byte[] key.
     * <p>
     * If a string of character is to be hashed, it is suggested but not
     * required to convert the String to a byte array using <i>UTF-8</i>.
     *
     * @param key The bytes used to seed the algorithm. This must be a non-
     *            <code>null</code>, non empty array of bytes. If the array is
     *            larger than 512 elements, only the first 512 entries are used.
     * @param text The clear text to apply the hash algorithm to.
     * @return The hash code.
     * @throws CryptoException If any problem occurrs calculating the hash code
     *             of the text. The {@link CryptoException#getCause()} method
     *             may provide additional information on the failure.
     * @throws IllegalArgumentException if the <code>key</code> or
     *             <code>text</code> is <code>null</code> or an empty array.
     * @since 1.1, Crypto Support 0.4
     */
    public byte[] hmac_sha256(byte[] key, byte[] text)
            throws CryptoException;
    
    /**
     * Generate HMAC bytes given some text. In other, perhaps less
     * cryptographically correct words, generates and returns a hash of 'text'
     * encrypted by 'keyBytes'.
     * <p>
     * The implementation is expected to implement the keyed hashing function
     * using SHA-256 as the hash algorithm. See <a
     * href="http://www.ietf.org/rfc/rfc2104.txt">RFC 2104</a> for the HMAC
     * specification.
     * <p>
     * If a string of character is to be hashed, it is suggested but not
     * required to convert the String to a byte array using <i>UTF-8</i>.
     *
     * @param text The clear text to apply the hash algorithm to.
     * @return The hash code.
     * @throws CryptoException If any problem occurrs calculating the hash code
     *             of the text. The {@link CryptoException#getCause()} method
     *             may provide additional information on the failure.
     * @throws IllegalArgumentException if the <code>key</code> or
     *             <code>text</code> is <code>null</code> or an empty array.
     * @since 1.2
     */
    public byte[] hmac_sha256(byte[] text)
            throws CryptoException;
     
    /**
     * Generates a key pair. 
     * This will generate a new key pair every time it is called.
     * 
     * @param algorithm the standard string name of the algorithm
     * @return the generated key pair
     * @throws CryptoException If any problem occurs creating the 
     *             	key pair. The {@link CryptoException#getCause()} method may
     *             	provide additional information on the encryption failure.
     *  @throws IllegalArgumentException if the <code>algorithm</code>
     *  			is <code>null</code> or incorrect.
     * @since 1.3                         
     */
    KeyPair createKeyPair(String algorithm) throws CryptoException;

    /**
     * Sign a {@link Certificate} either using a provided issuer certificate or using the {@link Certificate} subject
     * as issuer (self signed).
     *
     * @param issuerCertificate the {@link Certificate} of the issuer or {@code null} to self-sign the certificate.
     * @param keyPair the key pair containing the certificate subject {@link java.security.PublicKey} and the
     *                issuer {@link java.security.PrivateKey} key.
     * @param subject the subject of the certificate to be issued
     * @param before the {@code notBefore} UTC timestamp for the certificate validity period
     * @param after the {@code notAfter} UTC timestamp for the certificate validity period
     * @return the signed {@link Certificate}
     * @throws CryptoException if any problem occurs when signing
     *
     * @since 1.4
     */
    Certificate sign(Certificate issuerCertificate, KeyPair keyPair, X500Principal subject, long before, long after)
            throws CryptoException;
    
	/**
	 * Sign some data using the given private key
	 * <p>
     * Please note that the implementation will not clear the private key.
     * <p>
	 * 
	 * @param text the clear text to sign
	 * @param privateKey the private key used to sign the clear text
	 * @param algorithm the standard string name of the algorithm
	 * @return the signedText
	 * @throws CryptoException  If any problem occurs signing the 
     *             	clear text. The {@link CryptoException#getCause()} method may
     *             	provide additional information on the encryption failure.
     * @throws IllegalArgumentException if the <code>algorithm</code> or 
     *  			<code>privateKey</code> is <code>null</code> or incorrect.            
     * @since 1.3  
	 */
	byte[] sign(byte[] text, PrivateKey privateKey, String algorithm) throws CryptoException;
	
	/**
	 * Perform a signature verification with the given public key.
	 * <p>
     * Please note that the implementation will not clear the public key.
     * <p>
	 * 
	 * @param text The clear text which has been signed
	 * @param signedText the signed text to be verified
	 * @param publicKey the public key used to verify the signature
	 * @param algorithm the standard string name of the algorithm
	 * @return <code>true</code> if the alleged signature (signedText) is the actual 
	 * 			signature of the specified data (text)
	 * @throws CryptoException If any problem occurs verifying the 
     *  	      	signed text. The {@link CryptoException#getCause()} method may
     * 	            provide additional information on the encryption failure.    
     * @throws IllegalArgumentException if the <code>algorithm</code> or 
     *  			<code>publicKey</code> is <code>null</code> or incorrect. 
     * @since 1.3             
	 */
	boolean verify(byte[] text, byte[] signedText, PublicKey publicKey, String algorithm) throws CryptoException;
}
