/*************************************************************************
* ADOBE CONFIDENTIAL
* __________________
*
* Copyright 2011 Adobe
* All Rights Reserved.
*
* NOTICE: All information contained herein is, and remains
* the property of Adobe and its suppliers, if any. The intellectual
* and technical concepts contained herein are proprietary to Adobe
* and its suppliers and are protected by all applicable intellectual
* property laws, including trade secret and copyright laws.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe.
**************************************************************************/
package com.adobe.granite.jmx.annotation;

import javax.management.MBeanAttributeInfo;
import javax.management.MBeanConstructorInfo;
import javax.management.MBeanInfo;
import javax.management.MBeanOperationInfo;
import javax.management.MBeanParameterInfo;
import javax.management.NotCompliantMBeanException;
import javax.management.openmbean.OpenMBeanAttributeInfo;
import javax.management.openmbean.OpenMBeanAttributeInfoSupport;
import javax.management.openmbean.OpenMBeanConstructorInfo;
import javax.management.openmbean.OpenMBeanConstructorInfoSupport;
import javax.management.openmbean.OpenMBeanInfoSupport;
import javax.management.openmbean.OpenMBeanOperationInfo;
import javax.management.openmbean.OpenMBeanOperationInfoSupport;
import javax.management.openmbean.OpenMBeanParameterInfo;
import javax.management.openmbean.OpenMBeanParameterInfoSupport;
import javax.management.openmbean.OpenType;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The extension of {@link AnnotatedStandardMBean} that will generate
 * {@link javax.management.openmbean.OpenMBeanInfo}. If any exception occur
 * during generation, then {@link MBeanInfo} generated by the parent class is
 * returned instead.
 * <p>
 * This implementation will look for the required {@link OpenTypeInfo}
 * annotation if the data type of attribute, operation or constructor is
 * {@link javax.management.openmbean.CompositeData} or
 * {@link javax.management.openmbean.TabularData}. This annotation will provide
 * a class that will be a template to generate the respective
 * {@link javax.management.openmbean.CompositeType} or
 * {@link javax.management.openmbean.TabularType}.
 *
 * @see javax.management.openmbean.OpenMBeanInfo
 * @see AnnotatedStandardMBean
 * @see OpenTypeInfo
 * @see TabularTypeInfo
 */
public class OpenAnnotatedStandardMBean extends AnnotatedStandardMBean {
    private static final Logger log = LoggerFactory.getLogger(OpenAnnotatedStandardMBean.class);

    public <T> OpenAnnotatedStandardMBean(T implementation,
            Class<T> mbeanInterface) throws NotCompliantMBeanException {
        super(implementation, mbeanInterface);
    }

    protected OpenAnnotatedStandardMBean(Class<?> mbeanInterface)
            throws NotCompliantMBeanException {
        super(mbeanInterface);
    }

    @Override
    public MBeanInfo getMBeanInfo() {
        MBeanInfo info = super.getMBeanInfo();
        try {
            if (info instanceof OpenMBeanInfoSupport) {
                return info;
            }

            MBeanAttributeInfo[] attrs = info.getAttributes();
            OpenMBeanAttributeInfo[] openAttrs = new OpenMBeanAttributeInfo[attrs.length];
            for (int i = 0; i < attrs.length; i++) {
                openAttrs[i] = convert(attrs[i]);
            }

            MBeanConstructorInfo[] consts = info.getConstructors();
            OpenMBeanConstructorInfo[] openConsts = new OpenMBeanConstructorInfo[consts.length];
            for (int i = 0; i < consts.length; i++) {
                openConsts[i] = convert(consts[i]);
            }

            MBeanOperationInfo[] ops = info.getOperations();
            OpenMBeanOperationInfo[] openOps = new OpenMBeanOperationInfo[ops.length];
            for (int i = 0; i < ops.length; i++) {
                openOps[i] = convert(ops[i]);
            }

            MBeanInfo open = new OpenMBeanInfoSupport(info.getClassName(),
                info.getDescription(), openAttrs, openConsts, openOps,
                info.getNotifications());

            cacheMBeanInfo(open);

            return open;
        } catch (Exception e) {
            log.warn("Error occur coverting to OpenMBeanInfo", e);
            return info;
        }
    }

    private OpenMBeanAttributeInfo convert(MBeanAttributeInfo info)
            throws Exception {
        Class<?> ret = info.isReadable()
                ? getReadMethod(info).getReturnType()
                : getWriteMethod(info).getParameterTypes()[0];

        OpenTypeInfo typeInfo = getAnnotation(info, OpenTypeInfo.class);
        OpenType type = OpenTypeUtils.createOpenType(ret, typeInfo);

        return new OpenMBeanAttributeInfoSupport(info.getName(),
            info.getDescription(), type, info.isReadable(), info.isWritable(),
            info.isIs());
    }

    private OpenMBeanConstructorInfo convert(MBeanConstructorInfo info)
            throws Exception {
        MBeanParameterInfo[] sig = info.getSignature();
        OpenMBeanParameterInfo[] openSig = new OpenMBeanParameterInfo[sig.length];
        for (int i = 0; i < sig.length; i++) {
            openSig[i] = convert(info, sig[i], i);
        }

        return new OpenMBeanConstructorInfoSupport(info.getName(),
            info.getDescription(), openSig);
    }

    private OpenMBeanOperationInfo convert(MBeanOperationInfo info)
            throws Exception {
        Class<?> returnType = getClass(info.getReturnType());

        OpenTypeInfo typeInfo = getMethod(info).getAnnotation(
            OpenTypeInfo.class);
        OpenType openType = OpenTypeUtils.createOpenType(returnType,
            typeInfo);

        MBeanParameterInfo[] sig = info.getSignature();
        OpenMBeanParameterInfo[] openSig = new OpenMBeanParameterInfo[sig.length];
        for (int i = 0; i < sig.length; i++) {
            openSig[i] = convert(info, sig[i], i);
        }

        return new OpenMBeanOperationInfoSupport(info.getName(),
            info.getDescription(), openSig, openType, info.getImpact());
    }

    private OpenMBeanParameterInfo convert(MBeanConstructorInfo ctor,
            MBeanParameterInfo info, int seq) throws Exception {
        Class<?> clazz = getClass(info.getType());

        OpenTypeInfo typeInfo = getParamAnnotation(ctor, seq,
            OpenTypeInfo.class);
        OpenType openType = OpenTypeUtils.createOpenType(clazz, typeInfo);

        return new OpenMBeanParameterInfoSupport(info.getName(),
            info.getDescription(), openType);
    }

    private OpenMBeanParameterInfo convert(MBeanOperationInfo op,
            MBeanParameterInfo info, int seq) throws Exception {
        Class<?> clazz = getClass(info.getType());

        OpenTypeInfo typeInfo = getParamAnnotation(op, seq, OpenTypeInfo.class);
        OpenType openType = OpenTypeUtils.createOpenType(clazz, typeInfo);

        return new OpenMBeanParameterInfoSupport(info.getName(),
            info.getDescription(), openType);
    }

    private Class<?> getClass(String type) throws ClassNotFoundException {
        if ("void".equals(type)) {
            return Void.class;
        } else if ("boolean".equals(type)) {
            return Boolean.TYPE;
        } else if ("byte".equals(type)) {
            return Byte.TYPE;
        } else if ("char".equals(type)) {
            return Character.TYPE;
        } else if ("double".equals(type)) {
            return Double.TYPE;
        } else if ("float".equals(type)) {
            return Float.TYPE;
        } else if ("int".equals(type)) {
            return Integer.TYPE;
        } else if ("long".equals(type)) {
            return Long.TYPE;
        } else if ("short".equals(type)) {
            return Short.TYPE;
        } else {
            return getMBeanInterface().getClassLoader().loadClass(type);
        }
    }

}
