/*************************************************************************
* ADOBE CONFIDENTIAL
* ___________________
*
* Copyright 2016 Adobe
* All Rights Reserved.
*
* NOTICE: All information contained herein is, and remains
* the property of Adobe and its suppliers, if any. The intellectual
* and technical concepts contained herein are proprietary to Adobe
* and its suppliers and are protected by all applicable intellectual
* property laws, including trade secret and copyright laws.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe.
**************************************************************************/
package com.adobe.granite.ui.components;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import javax.servlet.ServletRequest;

/**
 * A mechanic to allow a certain customization for {@link ExpressionResolver}.
 *
 * <p>
 * The customization is only scoped to the current request, not a global one. The resource type implementation needs to
 * use this class explicitly to make the customization available in the request scope going forward.
 * </p>
 *
 * <p>
 * For example, the resource type implementation of CQ Dialog ({@code /apps/my/dialog}) wants to register
 * {@code cqDesign} variable so that it can be used in EL expression:
 * </p>
 *
 * <pre>
 * <code>
 * Design design = getDesign();
 * ExpressionCustomizer customizer = ExpressionCustomizer.from(request);
 * customizer.setVariable("cqDesign", design);
 * </code>
 * </pre>
 *
 * <p>
 * Then in the content structure, the content author can use {@code cqDesign} variable in the property that accepts
 * EL:
 * </p>
 *
 * <pre>
 * <code>
 * + /apps/content/my/dialog
 *   - sling:resourceType = "my/dialog"
 *   + field1
 *     - sling:resourceType = "my/textfield"
 *     - value = "${cqDesign.id}"
 * </code>
 * </pre>
 */
public class ExpressionCustomizer {
    private static final String KEY_CACHE = ExpressionCustomizer.class.getName();

    /**
     * Returns the instance from the given request.
     *
     * @param request
     *            the current request acting as the scope of the customization
     * @return the instance of the customizer
     */
    @Nonnull
    public static ExpressionCustomizer from(@Nonnull ServletRequest request) {
        ExpressionCustomizer instance = (ExpressionCustomizer) request.getAttribute(KEY_CACHE);

        if (instance == null) {
            instance = new ExpressionCustomizer();
            request.setAttribute(KEY_CACHE, instance);
        }

        return instance;
    }

    private Map<String, Object> variables = new HashMap<String, Object>();

    /**
     * Returns the value of the variable with the given name.
     *
     * @param name
     *            the name of the variable
     * @return the value of the variable, or {@code null} if there is no variable with that name
     */
    @CheckForNull
    public Object getVariable(@Nonnull String name) {
        return variables.get(name);
    }

    /**
     * Creates a variable with the given name and value.
     *
     * <p>
     * Setting an existing variable with the same name will overwrite its value.
     * </p>
     *
     * @param name
     *            the name of the variable
     * @param value
     *            the value of the variable
     */
    public void setVariable(@Nonnull String name, @Nonnull Object value) {
        variables.put(name, value);
    }

    /**
     * Returns {@code true} if there is a variable with the given name; {@code false} otherwise.
     *
     * @param name
     *            the name of the variable
     * @return {@code true} if there is a variable with the given name; {@code false} otherwise
     */
    public boolean hasVariable(@Nonnull String name) {
        return variables.containsKey(name);
    }
}
