/*
 * File: CosRepairList.java
 * ************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2010 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.internal.pdftoolkit.core.cos;

import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map.Entry;

import com.adobe.internal.io.stream.InputByteStream;
import com.adobe.internal.pdftoolkit.core.types.ASName;

/**
 * This class maintains a parallel cache of modifications made to cosobjects
 * while repairing. 
 * TODO This only supports Indirect CosDictionary. Only candidate cases found so far for 
 * cos level repairs are of incorrect stream length entries, so this works. If more cases
 * are found then this class should be updated to handle those as well.   
 * @author hraghav
 */
class CosRepairList 
{
	private HashMap<Integer, RepairContainer> repairList = new HashMap<Integer, RepairContainer>();
	
	private CosDocument document;
	
	public CosRepairList(CosDocument document) 
	{
		this.document = document;
	}
	
	/**
	 * Sets repaired value. 
	 * @param obj CosDictionary being repaired
	 * @param key Key in CosDictionary whose value is being changed
	 * @param value Changed value for the key.
	 */
	void setRepairedValue(CosDictionary obj, ASName key, CosObject value)
	{
		Integer objectNum = obj.getObjNum();
		RepairContainer container = repairList.get(objectNum);
		if(container == null)
		{
			container = new RepairContainer(null);
			repairList.put(objectNum, container);
		}
		container.put(key, value);
	}
	/**
	 * Sets the repaired cosobject of the specified object number.
	 * @param objNum
	 * @param obj
	 */
	void setRepairedValue(Integer objNum, CosObject obj){
		if(obj == null)
			return;
		RepairContainer container = repairList.get(objNum);
		if(container == null)
		{
			container = new RepairContainer(obj);
			repairList.put(objNum, container);
			obj.setRepaired(true);
		}
	}
			
	/**
	 * Checks if repair has been done for this object number.
	 * @param objectNum 
	 * @return true if repair has been done, else false
	 */
	boolean isObjectRepaired(Integer objectNum)
	{
		return repairList.containsKey(objectNum);
	}
	
	/**
	 * Gets repaired value for this object number and key.
	 * @param objectNum 
	 * @param key
	 * @return Updated cosobject
	 */
	CosObject getRepairedValue(Integer objectNum, ASName key)
	{
		if(! document.getUseRepairList())
			return null;
		
		RepairContainer container = repairList.get(objectNum);
		if(container == null)
			return null;
		return container.get(key);
	}
	/**
	 * Returns the repaired object for the specified object number.
	 * @param objectNum
	 * @return {@link CosObject}
	 */
	CosObject getRepairedValue(Integer objectNum){
		if(! document.getUseRepairList())
			return null;
		RepairContainer container = repairList.get(objectNum);
		if(container == null)
			return null;
		return container.getRepairedObject();
	}
	/**
	 * Clears the repairlist.
	 */
	void clear()
	{
		repairList.clear();
	}
	
	/**
	 * Updates the mappings t ouse new object numbers as keys instead of old object numbers. This
	 * is required to update repair list when objects are reordered while linear-save.
	 * @param objectNumMap
	 */
	void updateObjectNumbers(HashMap<Integer, Integer> objectNumMap)
	{
		Iterator<Entry<Integer, Integer>> iter = objectNumMap.entrySet().iterator();
		HashMap<Integer, RepairContainer> tempRepairList = new HashMap<Integer, RepairContainer>();
		
		while(iter.hasNext())
		{
			Entry<Integer, Integer> entry = iter.next();
			Integer oldNum = entry.getKey();
			Integer newNum = entry.getValue();
			tempRepairList.put(newNum, repairList.get(oldNum));			
		}
		repairList.clear();
		repairList = tempRepairList;
	}
	
}

/**
 * This class represents a mapping between keys and repaired values.
 */
class RepairContainer
{
	private CosObject obj = null;
	private HashMap<ASName, CosObject> mapping;
	
	public RepairContainer(CosObject obj) 
	{
		this.obj = obj;
		mapping = new HashMap<ASName, CosObject>();
	}
	
	void put(ASName key, CosObject value)
	{
		mapping.put(key, value);
	}
	
	CosObject get(ASName key)
	{
		return mapping.get(key);
	}
	
	CosObject getRepairedObject(){
		return this.obj;
	}
}

/**
 * This class contain helper methods for repairing cos objects
 */
class CosRepairUtils
{
	/**
	 * This method gets the stream length of the cosstream. 
	 * Stream length is the number of bytes from start of stream to 
	 * first occurrence of keyword 'endstream' 
	 * 
	 * @param doc
	 * @param start
	 * @return stream length
	 */
	static long getStreamLength(CosDocument doc, long start)
	{
		InputByteStream fileStream = null; 
		try {
			fileStream = doc.getStream();
			fileStream.seek(start);
			CosParseBuf buf = new CosParseBuf(fileStream, 4096);
			for(int b = buf.read(); b != InputByteStream.EOF;) {
				if(b == 'e') {
					if((b = buf.read()) == 'n'
						&& (b = buf.read()) == 'd'
						&& (b = buf.read()) == 's'
						&& (b = buf.read()) == 't'
						&& (b = buf.read()) == 'r'
						&& (b = buf.read()) == 'e'
						&& (b = buf.read()) == 'a'
						&& (b = buf.read()) == 'm') {
						return buf.getPosition() - start - 9; 
					}
				} else
					b = buf.read();
			}
		}
		catch(Exception e)
		{
			// ignore exceptions while repairing
		}
		finally {
			if(fileStream != null)
				try {
					fileStream.close();
				} catch (IOException e) {
					// ignore exceptions while closing stream
				}
		}
		return -1;
	}
	
	/**
	 * This method checks is the stream end is valid, i.e the stream ends 
	 * with "endstream"
	 * 
	 * @param doc
	 * @param endPos Position where stream is supposed to end.
	 * @return <code>true</code> it stream end with endstream, else <code>false</code>
	 */
	static boolean isStreamEndValid(CosDocument doc, long endPos)
	{
		InputByteStream fileStream = null; 
		try {
			fileStream = doc.getStream();
			fileStream.seek(endPos);
			CosParseBuf buf = new CosParseBuf(fileStream, 4096);
//			There may be an additional EOL	marker, preceding endstream, that is 
//			not included in the count and is not logically part of the stream data.
			
//			The CARRIAGE RETURN (0Dh) and LINE FEED (0Ah) characters, also called newline characters, shall be
//			treated as end-of-line (EOL) markers. The combination of a CARRIAGE RETURN followed immediately by a
//			LINE FEED shall be treated as one EOL marker.
//			fileStream.seek(endPos - 9); // endpos - length(endstream)
			
			if(buf.read() == '\r')
			{
				if(buf.read() != '\n')
				{
					buf.unget();
				}
				
			}
			else
			{
				buf.unget();
				if(buf.read() != '\n')
				{
					buf.unget();
				}
			}
			
			if(buf.read() == 'e'
						&& buf.read() == 'n'
						&& buf.read() == 'd'
						&& buf.read() == 's'
						&& buf.read() == 't'
						&& buf.read() == 'r'
						&& buf.read() == 'e'
						&& buf.read() == 'a'
						&& buf.read() == 'm') {
						return true; 
					}
				else
					return false;
			
		}
		catch(Exception e)
		{
			// ignore exceptions while repairing
		}
		finally {
			if(fileStream != null)
				try {
					fileStream.close();
				} catch (IOException e) {
					// ignore exceptions while closing stream
				}
		}
		return false;
	}
	
}



