package com.adobe.internal.pdftoolkit.core.credentials.impl;

import java.security.PrivateKey;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;

import com.adobe.internal.pdftoolkit.core.credentials.Credentials;
import com.rsa.certj.cert.CertificateException;
import com.rsa.certj.cert.X509Certificate;
import com.rsa.jsafe.JSAFE_PrivateKey;
import com.rsa.jsafe.JSAFE_UnimplementedException;

/*
 * File: RSACredentials.java
 *  
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/


public class RSACredentials implements Credentials
{	
	protected com.rsa.certj.cert.X509Certificate rsaX509Cert;
	protected com.rsa.certj.cert.Certificate[] rsaX509CertChain;
	protected JSAFE_PrivateKey JSAFEPrivateKey;
	protected PrivateKey privateKey;
	
	public RSACredentials(JSAFE_PrivateKey jsafePrivateKey, com.rsa.certj.cert.X509Certificate cert, com.rsa.certj.cert.Certificate[] certChain) 
	{
		this.JSAFEPrivateKey = jsafePrivateKey;
		this.rsaX509Cert = cert;
		this.rsaX509CertChain = certChain;
	}
	public RSACredentials(PrivateKey privateKey, com.rsa.certj.cert.X509Certificate cert, com.rsa.certj.cert.Certificate[] certChain) throws JSAFE_UnimplementedException 
	{
		this.JSAFEPrivateKey = getRSAPrivateKey(privateKey);
		this.rsaX509Cert = cert;
		this.rsaX509CertChain = certChain;
		this.privateKey = privateKey;
	}
	
	public static RSACredentials newInstance(Credentials credentials)
	throws JSAFE_UnimplementedException, CertificateEncodingException, CertificateException 
	{
		if(credentials instanceof RSACredentials)
			return (RSACredentials) credentials;
		else if(credentials instanceof JCECredentials && !credentials.isHSM()) {
			JCECredentials jceCred = (JCECredentials)credentials;
			PrivateKey privateKey = jceCred.getPrivateKey();
			return new RSACredentials(privateKey,
					getRSACert(jceCred.getCertificate()), 
					buildRSACertChain(jceCred.getCertificateChain()));
		}
		throw new RuntimeException("Unsupported Credential type: " + credentials.getClass());
	}
	
	public static RSACredentials[] newInstance(Credentials[] credentials)
	throws JSAFE_UnimplementedException, CertificateEncodingException, CertificateException 
	{
		RSACredentials[] rsaCredentials = new RSACredentials[credentials.length];
		int credInd;
		for (credInd = 0; credInd < credentials.length; credInd++)
		{
			rsaCredentials[credInd] = RSACredentials.newInstance(credentials[credInd]);
		}
		return rsaCredentials;
	}	

	/**
	 * Converts a JCE Private Key to a JSAFE_PrivateKey
	 * 
	 * @param privateKey
	 * @return com.rsa.jsafe.JSAFE_PrivateKey
	 * @throws JSAFE_UnimplementedException
	 */
	static JSAFE_PrivateKey getRSAPrivateKey(java.security.PrivateKey privateKey) throws JSAFE_UnimplementedException
	{
		if(privateKey != null)
			return JSAFE_PrivateKey.getInstance(privateKey.getEncoded(), 0, "Java");
		return null;
	}

	public static X509Certificate getRSACert(java.security.cert.X509Certificate cert) throws CertificateEncodingException, CertificateException 
	{
		if(cert != null)
			return new com.rsa.certj.cert.X509Certificate(cert.getEncoded(), 0, 0);
		return null;
		
	}
	
	public com.rsa.certj.cert.X509Certificate getRSAX509Cert() {
		return rsaX509Cert;
	}
	
	public JSAFE_PrivateKey getJSAFEPrivateKey() {
		return JSAFEPrivateKey;
	}
	
	/**
	 * returns RSA certificate chain.
	 * @return com.rsa.certj.cert.Certificate[]
	 */
	public com.rsa.certj.cert.Certificate[] getRSAX509CertChain() {
		return rsaX509CertChain;
	}
	
	/**
	 * Builds an RSA certificate chain from an array of JCE certificates.
	 * 
	 * @param certChain
	 * @return Array of com.rsa.certj.cert.Certificate
	 * @throws CertificateEncodingException
	 * @throws CertificateException
	 */
	public static com.rsa.certj.cert.Certificate[] buildRSACertChain(Certificate[] certChain) throws CertificateEncodingException, CertificateException
	{

		if (certChain != null)
		{
			com.rsa.certj.cert.Certificate[] rsaCertChain = new com.rsa.certj.cert.Certificate[certChain.length];
			for (int i = 0; i < certChain.length; i++)
			{
				java.security.cert.X509Certificate javaCert = (java.security.cert.X509Certificate)certChain[i];
				com.rsa.certj.cert.X509Certificate rsaCert = new com.rsa.certj.cert.X509Certificate(javaCert.getEncoded(), 0, 0);
				rsaCertChain[i] = rsaCert;
			}
			return rsaCertChain;
		}
		return null;
	}

	public boolean isHSM() {
		return false;
	}
	
}
