/*
 * File: ASCII85OutputStream.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.filter;

import java.io.OutputStream;
import java.io.FilterOutputStream;
import java.io.IOException;

/**
 * ASCII85OutputStream
 *  Writes data coded in ASCII85
 *
 *	Important: This filter assumes the output stream is an ISO/Latin-1
 *	stream, of 1-byte (not Unicode) characters!
 *
 * Copyright (C) 1996-2005 Adobe Systems Incorporated
 */
public class ASCII85OutputStream extends FilterOutputStream
{
	// Variables
	private int NEWLINE_EVERY = 64;		/* maximum line length */
	private long value;				/* because we really need an Unsigned int */
	private int nShift;
	private int lineLength;
	private long totalOut;

	// Constructors
	/**
	 * Creates an output stream filter.
	 * @param out	the output stream
	 */
	public ASCII85OutputStream(OutputStream  out, FilterParams p)
	{
		super(out);
		initOutput();
		if (p != null) {
			if (p.containsKey(FilterParams.NewlineEvery_K))
				NEWLINE_EVERY = ((Integer)p.get(FilterParams.NewlineEvery_K)).intValue();
			}
	}

	public ASCII85OutputStream(OutputStream  out)
	{
		this(out, null);
	}

	// Methods
	private void initOutput()
	{
		lineLength = NEWLINE_EVERY;
		nShift = 24;
		value = 0;
		totalOut = 0;
	}

	private void putEOL()
		throws IOException
	{
		out.write((byte)'\r');	/* The most general form of end of line */
		out.write((byte)'\n');
		lineLength = 0;
		totalOut += 2;
	}

	private void put1(int ch)
		throws IOException
	{
			if (lineLength > NEWLINE_EVERY - 1)
				putEOL();
			out.write((byte)ch);
			++lineLength;
			++totalOut;
	}

	/**
	 * Writes a byte. Will block until the byte is actually
	 * written.
	 * @param b the byte
	 * @exception IOException If an I/O error has occurred.
	 */
	@Override
	public void write(int b)
		throws IOException
	{
		value += (long)(b & 0x0ff) << nShift;
		nShift -= 8;
		if (nShift < 0) {
			if (value == 0) {
				put1('z');
			}
			else {
				int q = (int)(value / 7225);		/* upper 3 bytes */
				int r = (int)(value - 7225 * q);	/* lower 2 bytes */
				int d1 = q / 7225;			/* highest byte */
				int d2, d3, d4, d5;

				q -= 7225 * d1;
				d2 = q / 85;
				d3 = q - 85 * d2;
				d4 = r / 85;
				d5 = r - 85 * d4;

				put1(d1 + '!');
				put1(d2 + '!');
				put1(d3 + '!');
				put1(d4 + '!');
				put1(d5 + '!');

				value = 0;
			}
			nShift = 24;
		}
	}

	/**
	 * Writes a subarray of bytes.
	 * @param b	the data to be written
	 * @param off	the start offset in the data
	 * @param len	the number of bytes that are written
	 * @exception IOException If an I/O error has occurred.
	 */
	@Override
	public void write(byte  b[], int  off, int  len)
		throws IOException
	{
		int	maxWrite = b.length-off;
		if (maxWrite > len)
			maxWrite = len;
		while (maxWrite-- > 0) {
			write(b[off++] & 0x0ff);
		}
	}

	/**
	 * Writes an array of bytes. Will block until the bytes
	 * are actually written.
	 * @param b	the data to be written
	 * @exception IOException If an I/O error has occurred.
	 */
	@Override
	public void write(byte  b[])
		throws IOException
	{
		write(b, 0, b.length);
	}

	/**
	 * Closes the stream and writes the stream trailer "~>".
	 * This method must be called to release any resources
	 * associated with the stream.
	 * @exception IOException If an I/O error has occurred.
	 */
	@Override
	public void close()
		throws IOException
	{
		if (nShift >= 0 && nShift < 24) {
			int q = (int)(value / 7225);	/* upper 3 bytes */
			int r = (int)(value - 7225 * q);	/* lower 2 bytes */
			int d1 = q / 7225;			/* highest byte */
			int d2, d3, d4;

			q -= 7225 * d1;
			d2 = q / 85;
			d3 = q - 85 * d2;
			d4 = r / 85;

			/*  There is a remainder of from one to three binary
			 *	bytes.  The idea is to simulate padding with 0's
			 *	to 4 bytes, converting to 5 bytes of ASCII, and then
			 *	sending only those high-order ASCII bytes necessary to
			 *	express unambiguously the high-order unpadded binary.  Thus
			 *	one byte of binary turns into two bytes of ASCII, two to three,
			 *	and three to four.  This representation has the charm that
			 *	it allows arbitrary-length binary files to be reproduced exactly, and
			 *	yet the ASCII is a prefix of that produced by the Unix utility
			 *	"btoa" (which rounds binary files upward to a multiple of four
			 *	bytes).
			 */

			switch (nShift) {
			case 16:
				put1(d1 + '!');
				put1(d2 + '!');
				break;

			case 8:
				put1(d1 + '!');
				put1(d2 + '!');
				put1(d3 + '!');
				break;

			case 0:
				put1(d1 + '!');
				put1(d2 + '!');
				put1(d3 + '!');
				put1(d4 + '!');
				break;
			}
		}

		if (lineLength > NEWLINE_EVERY - 2)
			putEOL();

		out.write((byte)'~');		/* End of stream marker */
		out.write((byte)'>');
		totalOut += 2;
		super.close();
	}

	/**
	 * Counts the number of bytes written by this filter.
	 * @return	actual number of bytes written
	 */
	public long getTotalOut()
	{
		return totalOut;
	}
}
