/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa;

import java.io.IOException;
import java.io.OutputStream;

import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.ResId;
import com.adobe.xfa.ut.StringUtils;


/**
 * A class to represent the XML character data nodes in the DOM.
 */
public class Chars extends Node {

	private String mText;

	/**
	 * @exclude from published api.
	 */
	public Chars(Element parent, Node prevSibling, char[] text, int start,
			int length) {
		super(parent, prevSibling);
		if (text == null) {
			mText = "";
		} else {
			mText = new String(text, start, length);
		}
	}

	/**
	 * Instantiates a character data node with the given text.
	 * @param parent the node's parent, if any.
	 * @param prevSibling the node's previous sibling, if any.
	 * @param text the node's text.
	 */
	public Chars(Element parent, Node prevSibling, String text) {
		super(parent, prevSibling);
		mText = text;
	}

	/**
	 * @exclude from published api.
	 */
	public Node clone(Element parent) {
		return new Chars(parent, parent.getLastXMLChild(), mText);
	}

	/**
	 * @see Node#getData()
	 */
	public String getData() {
		return mText;
	}
	
	/**
	 * Gets this node's name.
	 * @return the character data name, which is the constant value "".
	 */
	public String getName() {
		return "";
	}

	/**
	 * Gets this node's text.
	 * @return the character data text.
	 */
	public String getText() {
		return mText;
	}
	
	/**
	 * @exclude from published api.
	 */
	public boolean isLeaf() {
		return true;
	}

	/**
	 * Determines if this character data node consists
	 * entirely of XML white spaces.
	 * In XML, a white space is defined as a space, carriage return,
	 * linefeed or tab (i.e. U+0020, U+000A, U+000D, U+0009).
	 * @return true if this node's text consists entirely
	 *         of XML whitespace characters, and false otherwise.
	 */
	public boolean isXMLSpace() {
		for (int i = 0; i < mText.length(); i++)
			if (!isXMLSpace(mText.charAt(i)))
					return false;

		return true;
	}
	
	/**
	 * @exclude from published api.
	 */
	public static boolean isXMLSpace(int offset, int length, char[] chars) {
		for (int i = offset; i < length; i++)
			if (!isXMLSpace(chars[i]))
				return false;
		
		return true;
	}
	
	private static boolean isXMLSpace(char c) {
		// The XML whitespace characters are '\t', '\r', '\n' and ' '.
		// Since these are the only valid XML characters <= ' ', we don't
		// need to test individual characters, but can use a faster range check.
		return c <= ' ';
	}

	/**
	 * @exclude from published api.
	 */
	public void postSave() {
		// do nothing
	}

	/**
	 * @exclude from published api.
	 */
	public void preSave(boolean bSaveXMLScript) {
		// do nothing
	}

	/**
	 * @see Node#serialize(OutputStream, DOMSaveOptions, int, Node)
	 *
	 * @exclude from published api.
	 */
	public void serialize(OutputStream sOutFile, DOMSaveOptions options, int level, Node prevSibling) throws IOException {
		
		if (!options.canBeSaved(false, isDefault(false), isTransient()))
			return;
		
		if (options.getDisplayFormat() == DOMSaveOptions.PRETTY_OUTPUT) {
			if (isXMLSpace() && (getNextXMLSibling() != null || prevSibling != null)) {
				return;
			}
		}
		
		String content = StringUtils.toXML(mText, false);
		sOutFile.write(content.getBytes(Document.Encoding));
	}

	/**
	 * @exclude from published api.
	 */
	public void setScriptProperty(String sPropertyName, Arg propertyValue) {
		throw new ExFull(ResId.UNSUPPORTED_OPERATION, "Chars#setScriptProperty");
	}

	/**
	 * Sets this node's text.
	 * @param text the text.
	 */
	public void setText(String text) {
		mText = text;
		setDirty();
	}

}
